/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 20/05/2022
 */

#ifndef VAMPIRIA_OPENSSL_PKG_X509_H

#define VAMPIRIA_OPENSSL_PKG_X509_H 1

namespace vampiria { namespace openssl { namespace pkg {

//!Openssl wrapper class for using X509 certificates
class X509_Wrap
{
    private:
        //! Internal usage
        void pointer_except();
        
        vmp_bool free_; 
        
        //!Internal Usage
        vmp::str field_search(vmp::str str,vmp::str field);
    public:
        X509 *x509_;/*!<Pointer to a ssl certificate*/
        X509_Wrap *parent_;/*!<Pointer to a parent certificate*/

        //! A costructor
        X509_Wrap();

        //! A destructor
        ~X509_Wrap();

        //! Reset class
        void reset();

        //! Set certificate(in the class destroyer the certificate memory is not freed)
        /*!
            @param x509 pointer to a ssl cerificate
            @sa x509_
        */
        void set(X509 *x509);
        
        //! Reads a certified pem from file(in the class destroyer the certificate memory is freed)
        /*!
            @param filename input file
            @return void or except in case failure
        */
        void set_from_file(vmp::str filename);

        //! Gets parent cerificate
        /*!
            @sa parent_
            @return parent certificate
        */
        X509_Wrap *parent();

        //! Returns 509 version
        /*!
            @return pointer x509
        */
        vmp_int version();

        //! Gets 509 serial number(raw mode)
        /*!
            @param bout output buffer
        */
        void serial(vmp::Buf *bout);
        
        //!Returns information about the 509 signing algorithm
        /*!
            @return string info  
        */
        vmp::str signature_info();

        //!Returns x509 issuer(or field issuer)
        /*!
            @param issuer field(with input "" return all issuer)
            @return string issuer or field issuer
        */
        vmp::str issuer(vmp::str field="");

        //!Returns the initial validity date x509
        /*!
            @return string date
        */
        vmp::str validity_before();
        
        //!Returns the final validity date x509
        /*!
            @retrun string date
        */
        vmp::str validity_after();

        //!Returns x509 subject(or field subject)
        /*!
            @param subject field(with input "" return all issuer)
            @return string subject or field subject
        */
        vmp::str subject(vmp::str field="");

        //! Gets 509 public key(raw mode)
        /*!
            @param kout output buffer
        */
        void pubkey(openssl::pkg::PKey *kout);

        //! Gets 509 id issuer(raw mode)
        /*!
            @param out output buffer
        */
        void id_issuer(vmp::Buf *out);

        //! Gets 509 id subject(raw mode)
        /*!
            @param out output buffer
        */
        void id_subject(vmp::Buf *out);

        //!Returns x509 extensions
        /*!
            @return string extensions
        */
        vmp::str extensions();

        //! Gets 509 signature data(raw mode)
        /*!
            @param out output buffer
        */
        void signature_data(vmp::Buf *out);
        
        //! Gets 509 fingerprint sha1(raw mode)
        /*!
            @param out output buffer
        */
        void fingerprint_sha1(vmp::Buf *out);
        
        //! Gets 509 fingerprint sha256(raw mode)
        /*!
            @param out output buffer
        */
        void fingerprint_sha256(vmp::Buf *out);
        
        //!Checks whether the certificate is verified by the input public key
        /*!
            @param pubkey input public key
            @return true if is verified otherwise false
        */
        vmp_bool verify(openssl::pkg::PKey *pubkey);
};

}}}

#endif

