/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 * Date last update: 20/05/2022
 */

#ifndef VAMPIRIA_OPENSSL_PKG_SSLUI_H

#define VAMPIRIA_OPENSSL_PKG_SSLUI 1

namespace vampiria { namespace openssl { namespace pkg {

//! Ssl tcp gui
/*! Runs ssl tcp connection
*/
class SslUI:public event::UI
{
    private:
        vmp::utils::Storage<openssl::pkg::EventSsl> sref_;/*!<Storage event*/

        event::EVTCB     connect_;/*!< Client connect callback(Called by the client after the SSL handshake is completed)*/
        net::EVTCBRECV   crecv_;/*!< Client receive callback*/
        event::EVTCB     cclose_;/*!<Client close callback*/

        net::EVTCBACCEPT  acptevent_;/*!<Ssl server accept callback.Called when a connection is accepted*/
        event::EVTCB      svlcevent_;/*!<Close server listen connection*/       

        net::EVTCBRECV    svrevent_;/*!<Server recieve callback*/
        event::EVTCB      svcevent_;/*!<<Close connection server*/
    
        vmp::time::Time ctimeout_;/*!<Connection timeout*/
    public:
        //!A Constructor
        /*!
            @param manager input manager
            @param ctimeout Connection timeout
        */
        SslUI(event::Manager *manager,vmp::time::Time ctimeout=3.0);
        
        //!A Destructor
        ~SslUI();

        //!Virtual class implemented
        vmp::str identity(event::Cell *cell);

        //!Virtual class implemented
        void close_event(event::Cell *cell);

        //!Virtual class implemented
        void free_ref(event::Cell *cell);


        //!Virtual class implemented(for server connection)
        event::Event *child_event_new(event::Cell *cell);

        //!Set client event
        /*!
            @param connect connect callback
            @param crecv receive data stream
            @param cevent close client callback
        */
        void set_event_client(event::EVTCB connect,net::EVTCBRECV crecv,event::EVTCB cevent);
        
        //! Set server event
        /*!
           @param acptevent server accept callback
           @param svlcevent close listen event
           @param revent receive data stream
           @param svcevent close connection server
        */
        void set_event_server(net::EVTCBACCEPT acptevent,event::EVTCB svlcevent,net::EVTCBRECV svrevent,event::EVTCB svcevent);

        //! Add tcp client event. Callback events assigned with set_event_client
        /*!
            @param raddress remote server address
            @param ctx ssl context
            @return event cell (except error)
        */
        event::Cell *new_client(net::Address *raddress,openssl::pkg::Ctx *ctx);

        //! Add tcp server listen event. Callback events assigned with set_event_client
        /*!
            @param local address listener
            @param backlog max connection server
            @param ctx ssl context
            @return event cell (except error)
        */
        event::Cell *new_server(net::Address *local,vmp_size backlog,openssl::pkg::Ctx *ctx);

        //! Send packet
        /*!
            @param cell event cell
            @param buf packet buffer
	    @return except error
        */
        void send(event::Cell *cell,vmp::Buf *buf);
	
	//!Gets socket address local
        /*!
            @param cell cell associate
            @return address local 
        */
        net::Address *localaddr(event::Cell *cell);

        //!Gets socket address peer
        /*!
            @param cell cell associated
            @return address local 
        */
        net::Address *peeraddr(event::Cell *cell);

        //! Get SSL reference
        /*!
            @param cell event cell
            @return Ssl otherwise 0(except error)
        */
        openssl::pkg::Ssl *ssl(event::Cell *cell);
};

}}}

#endif

