/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 * Date last update: 20/05/2022
 */

#ifndef VAMPIRIA_OPENSSL_PKG_SSLEVENT_H

#define VAMPIRIA_OPENSSL_PKG_SSLEVENT 1

namespace vampiria { namespace openssl { namespace pkg {

//!Ssl class event(see framework event)
class EventSsl:public event::Event
{
    private:
       //! Reset Ssl event
       void evt_ssl_reset();
 
    public:
       //! A Constructor
       EventSsl();

       //! A Destructor
       ~EventSsl();

       event::EVTCB connect_;/*!<Ssl client connect callback.Called when a connection is complete.*/
       net::EVTCBACCEPT acptevent_;/*!<Ssl server accept callback.Called when a connection is accepted*/
       event::EVTCB cchild_;/*!< Close child event callback. Used for close accepted connection server*/
       net::EVTCBRECV recv_;/*!< Recv data stream.Called when socket receive data*/
       net::Address local_;/*!< Local address socket*/
       net::Address remote_;/*!< Remote address socket*/
       event::Cell *parent_;/*!< Used for SslServer for call callback acptevent_*/
       vmp::time::Time ctimeout_;/*!<Connection timeout*/
       openssl::pkg::Ssl *ssl_;/*!<Ssl connection associated*/
       
       //!Create a new event ssl client
       /*!
           @param ui user interface (see Class UI)
           @param server remote server address
           @param ctx ssl context
           @param connect function performed when connect is complete
           @param recv function performed when data is received
           @param close function performed at the end of the event
           @param ctimeout connection timeout
           @return the new cell associated with the event
       */
       event::Cell *evt_ssl_client(event::UI *ui,net::Address *server,openssl::pkg::Ctx *ctx,event::EVTCB connect,net::EVTCBRECV recv,event::EVTCB close,vmp::time::Time ctimeout);
       
       //!Create a new event tcp server
       /*!
           @param ui user interface (see Class UI)
           @param local address to listen server connection
           @param backlog the maximum length to which the  queue  of pending  connections
           @param ctx ssl context
           @param acptevent function performed when connection is accepted
           @param svlcevent function performed when close listen socket
           @param recv function performed when rececive data from socket
           @param svcevent function performed when close server connection
           @param ctimeout connection timeout
           @return the new cell associated with the event
       */
       event::Cell *evt_ssl_server(event::UI *ui,net::Address *local,vmp_uint backlog,openssl::pkg::Ctx *ctx,net::EVTCBACCEPT acptevent,event::EVTCB svlcevent,net::EVTCBRECV recv,event::EVTCB svcevent,vmp::time::Time ctimeout);

       //! Send stream packet
       /*!
           @param buf data to send
           @return void(except error)
       */
       void evt_ssl_send(vmp::Buf *buf);

       //!Gets socket address local
       /*!
           @sa local_
           @return address local 
       */
       net::Address *evt_ssl_localaddr();

       //!Gets socket address peer
       /*!
           @sa remote_
           @return address peer
       */
       net::Address *evt_ssl_peeraddr();
       
       //! Gets tcp connection type
       /*!
           @return connection type ("client","listen","server")
       */
       vmp::str evt_ssl_type();

       //! Gets Ssl connection associated with the event
       /*!
           @sa ssl_
           @return ssl connection
       */
       openssl::pkg::Ssl *evt_ssl_ref();
       
       //! Close event
       void evt_ssl_close();
       
       //! Free event
       void evt_ssl_free();
};

}}}

#endif
