/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 * Date last update: 20/05/2022
 */

#include "openssl1/openssl1.h"

namespace vampiria { namespace openssl { namespace pkg {

void ssl_recv_ev(event::Cell *cell)
{
    vmp_bool exec=false;
    event::Manager *manager=cell->get_manager();
    openssl::pkg::EventSsl *event=cell->event<openssl::pkg::EventSsl>();
    vmp::Buf buf;
    vmp_int ret;
    manager->lock();
    try
    {
        ret=event->ssl_->recvData(&buf);
        if(ret == 0)
            manager->cell_close(cell,event::SUCCESS);
        else if(ret > 0)
            exec=manager->cell_update(cell);
    }
    catch(vmp::exception &x)
    {
        manager->cell_close(cell,event::ERROR,x.what());
    }
    manager->unlock();
    if(exec)
        event->recv_(cell,&buf);
    buf.reset();  
}

void ssl_connect_handshake_ev(event::Cell *cell)
{
    vmp_bool exec=false;
    event::Manager *manager=cell->get_manager();
    openssl::pkg::EventSsl *event=cell->event<openssl::pkg::EventSsl>();
    openssl::pkg::SslClient *client=(openssl::pkg::SslClient *) event->ssl_;
    manager->lock();
    try
    {    
        if(client->ssl_connect())
        {    
            exec=manager->cell_update(cell);
            cell->read_=ssl_recv_ev;
            manager->cell_timewait(cell,0.0);
        }
    }
    catch(vmp::exception &x)
    {
        manager->cell_close(cell,event::ERROR,x.what());
    }
    manager->unlock();
    if(exec)
        event->connect_(cell);
}

void ssl_connect_ev(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    openssl::pkg::EventSsl *event=cell->event<openssl::pkg::EventSsl>();
    openssl::pkg::SslClient *client=(openssl::pkg::SslClient *) event->ssl_;
    manager->lock();
    try
    {    
        if(client->connect_check())
        {    
            manager->cell_update(cell);
            cell->read_=ssl_connect_handshake_ev;
            cell->writing_=false;
            manager->cell_timewait(cell,event->ctimeout_);
            client->ssl_connect();
        }
    }
    catch(vmp::exception &x)
    {
        manager->cell_close(cell,event::ERROR,x.what());
    }
    manager->unlock();
}

void ssl_accept_handshake_ev(event::Cell *cell)
{
    vmp_bool exec=false;
    event::Manager *manager=cell->get_manager();
    openssl::pkg::EventSsl  *event=cell->event<openssl::pkg::EventSsl>();
    openssl::pkg::SslServer *server=(openssl::pkg::SslServer *) event->ssl_;
    manager->lock();
    try
    {
        if(server->ssl_accept())
        {
            exec=manager->cell_update(cell);
            cell->read_=ssl_recv_ev;
            manager->cell_timewait(cell,0.0);
        }
    }
    catch(vmp::exception &x)
    {
        manager->cell_close(cell,event::ERROR,x.what());
    }
    manager->unlock();
    if(exec)
        event->acptevent_(event->parent_,cell);
}

void ssl_accept_ev(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    event::UI *ui=cell->ui<event::UI>();
    openssl::pkg::EventSsl  *event=cell->event<openssl::pkg::EventSsl>();
    openssl::pkg::SslListen *listen=(openssl::pkg::SslListen *) event->ssl_;
    manager->lock();
    try
    {    
        openssl::pkg::SslServer *server=listen->accept();
        if(server != 0)
        {
            vmp_bool exec=manager->cell_update(cell);
            if(!exec)
                sslserver_free_connection(server);
            else
            {
                openssl::pkg::EventSsl *chevent=(openssl::pkg::EventSsl *) ui->child_event_new(cell);
                if(chevent == 0)
                {
                     manager->cell_close(cell,event::ERROR,"Fatal Error Ssl accept connection uninitilaizated ui child_event_new()");
                     sslserver_free_connection(server);
                     
                }
                else
                {    
                     chevent->ssl_=server;
                     chevent->acptevent_=event->acptevent_;
                     chevent->recv_=event->recv_;
                     chevent->parent_=cell;
		     manager->cell_alloc(cell);
                     server->get_local_address(&chevent->local_);
                     server->get_peer_address(&chevent->remote_);
                     event::Cell *child=chevent->evt_new(ui,server->socket(),ssl_accept_handshake_ev,event->cchild_);
                     chevent->ctimeout_=event->ctimeout_;
                     child->evtype_=cell->evtype_;
                     manager->cell_timewait(child,chevent->ctimeout_);
                }
            }

        }
    }
    catch(vmp::exception &x)
    {
        manager->cell_close(cell,event::ERROR,x.what());
    }
    manager->unlock();
    
}

EventSsl::EventSsl():event::Event()
{
    ssl_=0;
    parent_=0;
}
       
EventSsl::~EventSsl()
{
    evt_ssl_reset();
}

void EventSsl::evt_ssl_reset()
{
    connect_=0;
    recv_=0;
    acptevent_=0;
    cchild_=0;
    if(parent_ != 0)
    {
        event::Manager *manager=cell_->get_manager();
        manager->cell_release(parent_);
        parent_=0;
    }
    local_.reset();
    remote_.reset();
    if(ssl_ != 0)
    {
        ssl_->reset();
        ssl_=0;
    }
}

event::Cell *EventSsl::evt_ssl_client(event::UI *ui,net::Address *server,openssl::pkg::Ctx *ctx,event::EVTCB connect,net::EVTCBRECV recv,event::EVTCB close,vmp::time::Time ctimeout)
{
    if(connect == 0)
        connect_=event::empty_ev;
    else
        connect_=connect;
    if(recv == 0)
        recv_=net::empty_recv_ev;
    else
        recv_=recv;
    openssl::pkg::SslClient *client;
    try
    {
        ctimeout_=ctimeout;
        ssl_=(openssl::pkg::Ssl *) new openssl::pkg::SslClient(ctx);
        client=(openssl::pkg::SslClient *) ssl_;
        client->noblock();
        client->connect(server);
        remote_.copy(server);
        client->get_local_address(&local_);
    }
    catch(vmp::exception &x)
    {
        evt_ssl_reset();
        vmp::except_s(x.what());
    }
    event::Cell *cell=evt_new(ui,client->socket(),ssl_connect_ev,close);
    cell->writing_=true;
    return cell;
}

event::Cell *EventSsl::evt_ssl_server(event::UI *ui,net::Address *local,vmp_uint backlog,openssl::pkg::Ctx *ctx,net::EVTCBACCEPT acptevent,event::EVTCB svlcevent,net::EVTCBRECV recv,event::EVTCB svcevent,vmp::time::Time ctimeout)
{
    if(acptevent == 0)
        acptevent_=net::empty_accept_ev;
    else
        acptevent_=acptevent;
    if(recv == 0)
        recv_=net::empty_recv_ev;
    else
        recv_=recv;
    if(svcevent == 0)
        cchild_=event::empty_ev;
    else  
        cchild_=svcevent;
    ssl_=(openssl::pkg::Ssl *)new openssl::pkg::SslListen(ctx);
    openssl::pkg::SslListen *listen=(openssl::pkg::SslListen *) ssl_;
    try
    {
        ctimeout_=ctimeout;
        listen->server(local,backlog);
        listen->noblock();
        local_.copy(local);
    }
    catch(vmp::exception &x)
    {
        evt_ssl_reset();
        vmp::except_s(x.what());
    }
    event::Cell *cell=evt_new(ui,listen->socket(),ssl_accept_ev,svlcevent);
    return cell;
}

void EventSsl::evt_ssl_send(vmp::Buf *buf)
{
     event::Manager *manager=cell_->get_manager();
     if(manager->cell_update(cell_) && (ssl_ != 0))
         ssl_->sendData(buf);
}

net::Address *EventSsl::evt_ssl_localaddr()
{
    return &local_;
}

net::Address *EventSsl::evt_ssl_peeraddr()
{
    return &remote_;
}

vmp::str EventSsl::evt_ssl_type()
{
    vmp::except_check_pointer((void *)ssl_,"openssl::pkg::EventSsl::evt_ssl_type() unitializated connection");
    return ssl_->type();
}

openssl::pkg::Ssl *EventSsl::evt_ssl_ref()
{
    return ssl_;
}


void EventSsl::evt_ssl_close()
{
    if(ssl_ != 0)
        ssl_->close();
}
       
void EventSsl::evt_ssl_free()
{
    evt_ssl_reset();
    evt_free();
}

}}}

