/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 23/01/2020
 */

#include "pylib.h"

namespace vampiria { namespace pylib {

void incref(pylib::Objref *ref)
{
    Py_XINCREF(ref);
}

void decref(pylib::Objref *ref)
{
    Py_XDECREF(ref);
}

pylib::Objref *none()
{
    return Py_None;
}

vmp_bool is_none(pylib::Objref *ref)
{
    if(ref ==  Py_None)
        return true;
    return false;
}

vmp_bool is_null(pylib::Objref *ref)
{
    if(ref == 0)
        return true;
    return false;    
}

void pyerr_clear()
{
    PyErr_Clear();
}

vmp::str pyunicode_asstring(pylib::Objref *obj)
{
    vmp::str ret=""; 
    if(PyUnicode_Check(obj))
    {
        pylib::Objref *tmp=PyUnicode_AsUTF8String(obj);
        if(tmp == 0)
            tmp=PyUnicode_AsLatin1String(obj);    
        if(tmp != 0)
        {
            ret=PyBytes_AsString(tmp);
            pylib::decref(tmp);
            try
            {
                pylib::pyerr_occurred();
                return ret;
            }
            catch(vmp::exception &x)
            {
                vmp::error(x.what());
            }
        }
    }
    vmp::except_s("python object in input is not a valid string(encode latin1 or utf8 required)");
    return ret;
}

vmp_char pychar_aschar(pylib::Objref *obj)
{
    try
    {
        vmp_int tmp=pylib::pylong_aslong(obj);
        if((tmp >= static_cast<vmp_int>(vmp::CHARMIN)) && (tmp <= static_cast<vmp_int>(vmp::CHARMAX)))
            return static_cast<vmp_char>(tmp);
    }
    catch(vmp::exception &x)
    {
    }
    try
    {
        vmp::str unicode=pyunicode_asstring(obj);
        if(unicode.size() == 1)
            return static_cast<vmp_char>(unicode[0]);
    }
    catch(vmp::exception &x)
    {
    }
    vmp::except_s("python object in input is not a char");
    return 0x00;
}

vmp_uchar pychar_asuchar(pylib::Objref *obj)
{
    try
    {
        vmp_int tmp=pylib::pylong_aslong(obj);
        if((tmp >= static_cast<vmp_int>(vmp::UCHARMIN)) && (tmp <= static_cast<vmp_int>(vmp::UCHARMAX)))
           return static_cast<vmp_uchar>(tmp);
    }
    catch(vmp::exception &x)
    {
    }
    try
    {
        vmp::str unicode=pyunicode_asstring(obj);
        if(unicode.size() == 1)
            return static_cast<vmp_uchar>(unicode[0]);
    }
    catch(vmp::exception &x)
    {
    }
    vmp::except_s("python object in input is not an unsigned char");
    return 0x00;    
}

vmp_int pylong_aslong(pylib::Objref *obj)
{
    if(PyLong_Check(obj))
    {
        vmp_int ret=PyLong_AsLong(obj);
        try
        {
            pylib::pyerr_occurred();
            return ret;
        }
        catch(vmp::exception &x)
        {
        }
    }
    vmp::except_s("python object in input is not an integer");
    return -1;
}

vmp_uint pylong_asulong(pylib::Objref *obj)
{
    if(PyLong_Check(obj))
    {
        vmp_uint ret=PyLong_AsUnsignedLong(obj);
        try
        {
            pylib::pyerr_occurred();
            return ret;
        }
        catch(vmp::exception &x)
        {
        }
    }
    vmp::except_s("python object in input is not an unsigned integer");
    return 0;
}

vmp_int16 pylong_asint16(pylib::Objref *obj)
{
    try
    {
        vmp_int tmp=pylib::pylong_aslong(obj);
        if((tmp >= static_cast<vmp_int>(vmp::INT16MIN)) && (tmp <= static_cast<vmp_int>(vmp::INT16MAX)))
           return static_cast<vmp_int16>(tmp);
    }
    catch(vmp::exception &x)
    {
    }
    vmp::except_s("python object in input is not an integer16");
    return -1;
}

vmp_uint16 pylong_asuint16(pylib::Objref *obj)
{
    try
    {
        vmp_uint tmp=pylib::pylong_asulong(obj);
        if((tmp >=  static_cast<vmp_uint>(vmp::UINT16MIN)) && (tmp <=  static_cast<vmp_uint>(vmp::UINT16MAX)))
           return static_cast<vmp_uint16>(tmp);
    }
    catch(vmp::exception &x)
    {
    }
    vmp::except_s("python object in input is not an unsigned integer16");
    return 0;
}

vmp_bool pylong_asbool(pylib::Objref *obj)
{
    try
    {
        vmp_uint tmp=pylib::pylong_asulong(obj);
        if((tmp == 0) || (tmp == 1))
            return static_cast<vmp_bool>(tmp);
    }
    catch(vmp::exception &x)
    {
    }
    vmp::except_s("python object in input is not a boolean");
    return false;
}

vmp_real pyfloat_asreal(pylib::Objref *obj)
{
    if(PyFloat_Check(obj))
    {
        vmp_real ret=PyFloat_AS_DOUBLE(obj);
        try
        {
            pylib::pyerr_occurred();
            return ret;
        }
        catch(vmp::exception &x)
        {
        }
    }
    else 
    {
        try
        {
            return static_cast<vmp_real>(pylib::pylong_aslong(obj));
        }
        catch(vmp::exception &x)
        {
        }
    }
    vmp::except_s("python object in input is not an real");
    return -1.0;
}

pylib::Objref *pyunicode_decode(vmp::str str)
{
    
    pylib::Objref *ret=PyUnicode_DecodeUTF8(str.c_str(),str.size(),0); 
    if(ret != 0)
        return ret;
    return PyUnicode_DecodeLatin1(str.c_str(),str.size(),0);
}

void pyerr_occurred()
{
    pylib::Objref *occurred=PyErr_Occurred();
    vmp::str error;
    if(occurred)
    {
        pylib::Objref *ptype,*pvalue,*ptraceback;
        PyErr_Fetch(&ptype,&pvalue, &ptraceback);
        PyErr_NormalizeException(&ptype,&pvalue,&ptraceback);
        pylib::Objref *strIO=PyObject_Str(pvalue);
        if((strIO == 0) || !(PyUnicode_Check(strIO)))
            error="pylib undef error";
        else
        {   vmp_char *data=(vmp_char *)PyUnicode_DATA(strIO);
            if(data == 0)   
                error="pylib undef error";
            else
                error=data;
        }
        PyErr_Restore(ptype,pvalue,ptraceback);
        pylib::decref(strIO);
        pylib::decref(occurred);
        pylib::decref(ptype);
        pylib::decref(pvalue);
        pylib::decref(ptraceback);
        pylib::pyerr_clear();
        vmp::except_s(error);
     }
}

void pyobj_delattr(pylib::Objref *obj,vmp::str name)
{
    if(PyObject_DelAttrString(obj,name.c_str()) == -1)
    {
        pylib::pyerr_clear();
        vmp::except("pyobj_delattr error delete '%s' value",name.c_str());
    }
}

void pymodule_addobj(pylib::Objref *module,vmp::str name,pylib::Objref *value)
{
    if(PyModule_AddObject(module,name.c_str(),value) == -1)
    {
        pylib::pyerr_clear();
        vmp::except("pymodule_addobj error insert '%s' value",name.c_str());
    }
}

pylib::Objref *pyobj_getattr_obj(pylib::Objref *obj,vmp::str name)
{
    pylib::Objref *ret=PyObject_GetAttrString(obj,name.c_str());
    if(ret == 0)
        pylib::pyerr_clear();
    return ret; 
}

pylib::Objref *pymodule_init(vmp::str name)
{
    static PyMethodDef initial_methods[] = { { 0, 0, 0, 0 } };
    static struct PyModuleDef base_init = {
       PyModuleDef_HEAD_INIT,
       "",
       0,
       -1,
       initial_methods,
       0,
       0,
       0,
       0
    };
    base_init.m_name=name.c_str();
    pylib::Objref *module=PyModule_Create(&base_init);
    if(module == 0)
        vmp::except("fatal error create module");
    pylib::pyobj_setattr<vmp::str>(module,"__name__",name);
    return module;
}

pylib::Objref *pymodule_addsubmodule(pylib::Objref *module,vmp::str name,vmp::str prefix)
{
    vmp::str fullname;
    vmp::unicode::str_write(&fullname,"%s.%s",prefix.c_str(),name.c_str());
    pylib::Objref *sub=PyImport_AddModule(name.c_str());
    pylib::pyobj_setattr<vmp::str>(sub,"__name__",fullname);
    pymodule_addobj(module,name,sub);
    return sub;
}

pylib::Objref *pylist_new(vmp_size size)
{
    pylib::Objref *list=PyList_New(size);
    if(list == 0)
    {
        pylib::pyerr_clear();
        vmp::except_s("pylist_new error create list");
    }
    return list;
}

vmp_bool pylist_check(pylib::Objref *list)
{
    if(pylib::is_null(list))
        return false;
    if(! PyList_Check(list))
       return false;
    return true;
}

vmp_size pylist_size(pylib::Objref *list)
{
   if(!pylist_check(list))
      vmp::except_s("pylist_size error input objref not list type"); 
   return PyList_Size(list);
}

pylib::Objref *pylist_getitem(pylib::Objref *list,vmp_index pos)
{
    if(!pylist_check(list))
        vmp::except_s("pylist_getitem error input objref not list type");
    pylib::Objref *ret=PyList_GetItem(list,pos);
    if(ret == 0)
    {
        pylib::pyerr_clear();
        vmp::except("pylist_getitem error get objref index %d",pos);
    }
    return ret;
}


void pylist_clear(pylib::Objref *list)
{
    vmp_size size;
    try
    {
        size=pylist_size(list);
    }
    catch(vmp::exception &x)
    {
        vmp::except_s("pylist_clear error input objref not list type");
    }
    for(vmp_index i=0;i<size;i++)
    {
        try
        {
            pylib::Objref *item=pylib::pylist_getitem(list,i);
            pylib::decref(item);
        }
        catch(vmp::exception &x)
        {
        }
    }
    pylib::decref(list);
}

pylib::Objref *pylist_astuple(pylib::Objref *list)
{
    if(!pylist_check(list))
        vmp::except_s("pylist_astuple error input objref not list type");
    return PyList_AsTuple(list);
}

vmp_bool pytuple_check(pylib::Objref *tuple)
{
    if(pylib::is_null(tuple))
       return false;
    if(! PyTuple_Check(tuple))
       return false;
    return true;
}

vmp_size pytuple_size(pylib::Objref *tuple)
{
    if(!pytuple_check(tuple))
        vmp::except_s("pytuple_size error input objref not tuple type");    
    return PyTuple_Size(tuple);
}

pylib::Objref *pytuple_getitem(pylib::Objref *tuple,vmp_size pos)
{
    if(!pytuple_check(tuple))
        vmp::except_s("pytuple_getitem error input objref not tuple type");
    pylib::Objref *ret=PyTuple_GetItem(tuple,pos);
    if(ret == 0)
    {
        pylib::pyerr_clear();
        vmp::except("pytuple_getitem error get objref in pos %d",pos);
    }
    return ret;
}

pylib::Objref *pydict_new()
{
    pylib::Objref *dict=PyDict_New();
    if(dict == 0)
    {
        pylib::pyerr_clear();
        vmp::except_s("pydict_new error create dict");
    }
    return dict;   
}

vmp_bool pydict_check(pylib::Objref *dict)
{
    if(pylib::is_null(dict))
        return false;
    if(! PyDict_Check(dict))
       return false;
    return true;
}

vmp_bool pydict_next(pylib::Objref *dict,Py_ssize_t *ppos,pylib::Objref **pkey,pylib::Objref **pvalue)
{
    if(!pydict_check(dict))
       vmp::except_s("pydict_next error input objref not dict type");
    if(PyDict_Next(dict,ppos,pkey,pvalue))
    {
        pylib::pyerr_clear();
        return true;
    }
    pylib::pyerr_clear();
    return false;    
}

void pydict_setitem(pylib::Objref *dict,pylib::Objref *key,pylib::Objref *val)
{
    if(!pydict_check(dict))
       vmp::except_s("pydict_setitem error input objref not dict type");
    try
    {
        if(PyDict_SetItem(dict,key,val) != 0)
            vmp::except_s();
    }
    catch(vmp::exception &x)
    {
        pylib::pyerr_clear();
        vmp::except_s("pydict_setitem insert error");
    }
}

void pydict_clear(pylib::Objref *dict)
{
    pylib::Objref *key,*value;
    Py_ssize_t pos=0;
    while(pylib::pydict_next(dict,&pos,&key,&value)) 
    {
        pylib::decref(key);
        pylib::decref(value);
    }
    PyDict_Clear(dict);
    pylib::decref(dict);
    pylib::pyerr_clear();
}

vmp_bool pycallable_check(pylib::Objref *callable)
{
   if(pylib::is_null(callable) || pylib::is_none(callable) || !PyCallable_Check(callable))
      return false;
   return true;
}

pylib::Objref *pycallable_call(pylib::Objref *callable,pylib::Objref *tuple)
{
    if(!pycallable_check(callable))
       vmp::except_s("pycallable_call input objref not callable");
    pylib::Objref *ret=PyObject_Call(callable,tuple,0);
    if(ret == 0)
    {
       try
       {
           pyerr_occurred();
       }
       catch(vmp::exception &x)
       {
           vmp::except_s(x.what());
       }
       vmp::except_s("Callback python error undef error");
    }
    return ret;
}

}}


