/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 06/04/2022
*/

#include "json.h"

namespace vampiria { namespace json {

void primitive_error(vmp::str label)
{
    vmp::except("primitive '%s' bad values",label.c_str());
}

void set_primitive_error(vmp::str label)
{
    vmp::except("set primitive '%s' bad values",label.c_str());
}

JData::JData()
{
    reset();
}

JData::~JData()
{
    reset();
}
 
void JData::get_jbody(vmp::str fname,json::JsonObj *ret)
{
    check_init(fname);
    try
    {
        root_->get_object("jbody",ret);
        if(!ret->isobject())
            vmp::except_s("");
    }
    catch(vmp::exception &x)
    {
        vmp::except("json::JData::%s() jbody bad object",fname.c_str());
    }
}

void JData::check_init(vmp::str fname)
{
    if(root_ == 0)
        vmp::except("json::JData::%s() root json obj not setting",fname.c_str());
}

void JData::reset() 
{
    root_=0;
}

void JData::new_data(json::JsonObj *root,vmp::str jtype)
{
    vmp::except_check_pointer((void *) root,"json::JData::new_data() null input root obj");
    if(!root->isobject())
        vmp::except_s("json::JData::new_data() bad input json object");
    root_=root;
    json::JsonObj obj;
    root_->add_object_str("jtype",jtype,&obj);
    root_->add_object_obj("jbody",&obj);
}

void JData::set(json::JsonObj *root)
{
    json::JsonObj jbody;
    vmp::except_check_pointer((void *) root,"json::JData::set() null input pointer root");
    if(!root->isobject())
        vmp::except_s("json::JData::set() bad input json object");
    try
    {
        root->get_object_str("jtype");
        root->get_object("jbody",&jbody);
        if(!jbody.isobject())
            vmp::except_s("");
    }
    catch(vmp::exception &x)
    {
        vmp::except_s("json::JData::set() bad input json object");
    }
    root_=root;
}

void JData::remove_data()
{
    check_init("remove_data");
    root_->del_object("jtype");
    root_->del_object("jbody");
}

json::JsonObj *JData::root()
{
    check_init("root");
    return root_;
}
vmp::str JData::jtype()
{
    check_init("jtype");
    return root_->get_object_str("jtype");
}
        
void JData::set_text(vmp::str label,vmp::str value)
{
    json::JsonObj jbody,obj;
    get_jbody("set_text",&jbody);
    jbody.add_object_str(label,value,&obj);
}

vmp::str JData::get_text(vmp::str label)
{
    json::JsonObj jbody;
    get_jbody("get_text",&jbody);
    try
    {
        return jbody.get_object_str(label);
    }
    catch(vmp::exception &x)
    {
        json::primitive_error(label);   
    }
    return "";
}

void JData::set_integer(vmp::str label,vmp_int value)
{
    json::JsonObj jbody,obj;
    get_jbody("set_integer",&jbody);
    jbody.add_object_number(label,(vmp_real)value,&obj);    
}
        
vmp_int JData::get_integer(vmp::str label)
{
    json::JsonObj jbody;
    get_jbody("get_integer",&jbody);
    try
    {
        return (vmp_int) jbody.get_object_number(label);
    }
    catch(vmp::exception &x)
    {
        json::primitive_error(label);   
    }
    return 0;
}
        
void JData::set_integer_range(vmp::str label,vmp_int min,vmp_int max,vmp_int value)
{
    if((value >= min) && (value <= max))
    {
        json::JsonObj jbody,obj;
        get_jbody("set_integer,range",&jbody);
        jbody.add_object_number(label,(vmp_real)value,&obj); 
    }
    else
        json::set_primitive_error(label);
}
        
vmp_int JData::get_integer_range(vmp::str label,vmp_int min,vmp_int max)
{
    json::JsonObj jbody;
    get_jbody("get_integer_range",&jbody);
    try
    {
         vmp_int value=(vmp_int) jbody.get_object_number(label);
         if((value < min) || (value > max))
             vmp::except_s("");
         return value;
    }
    catch(vmp::exception &x)
    {
        json::primitive_error(label);   
    }
    return 0;    
}
        
void JData::set_real(vmp::str label,vmp_real value)
{
    json::JsonObj jbody,obj;
    get_jbody("set_real",&jbody);
    jbody.add_object_number(label,value,&obj); 
}

vmp_real JData::get_real(vmp::str label)
{
    json::JsonObj jbody;
    get_jbody("get_real",&jbody);
    try
    {
        return jbody.get_object_number(label);
    }
    catch(vmp::exception &x)
    {
        json::primitive_error(label);   
    }
    return 0.0;
}

void JData::set_real_range(vmp::str label,vmp_real min,vmp_real max,vmp_real value)
{
    if((value >= min) && (value <= max))
    {
        json::JsonObj jbody,obj;
        get_jbody("set_real_range",&jbody);
        jbody.add_object_number(label,value,&obj); 
    }
    else
        json::set_primitive_error(label);   
}

vmp_real JData::get_real_range(vmp::str label,vmp_real min,vmp_real max)
{
    json::JsonObj jbody;
    get_jbody("get_real_range",&jbody);
    try
    {
         vmp_real value=jbody.get_object_number(label);
         if((value < min) || (value > max))
             vmp::except_s("");
         return value;
    }
    catch(vmp::exception &x)
    {
        json::primitive_error(label);   
    }
    return 0.0;    
}

void JData::new_list(vmp::str label,json::JList *list,vmp::str jtype)
{
    vmp::except_check_pointer((void *) list,"json::JData::new_list() null input pointer list");
    json::JsonObj jbody;
    get_jbody("new_list",&jbody);
    json::JsonObj *obj=new json::JsonObj();
    jbody.add_object_array(label,obj);
    list->set(obj,jtype);
}

void JData::set_list(vmp::str label,json::JList *list,vmp::str jtype)
{
    vmp::except_check_pointer((void *) list,"json::JData::set_list() null pointer input list");
    if(list->list_ == 0)
        vmp::except_check_pointer((void *) list,"json::JData::set_list() input list not setting");    
    if((jtype != "") && (jtype != list->jdata_type()))
        vmp::except_s("json::JData::set_list() input list bad type");
    json::JsonObj jbody;
    get_jbody("set_list",&jbody);
    jbody.add_object_ex(label,list->list_);
}

void JData::get_list(vmp::str label,json::JList *list,vmp::str jtype)
{
    vmp::except_check_pointer((void *) list,"json::JData::get_list() null pointer input list");
    json::JsonObj jbody;
    get_jbody("get_list",&jbody);
    json::JsonObj *obj;
    try
    {
         obj=new json::JsonObj();
         jbody.get_object(label,obj);
         list->set(obj,jtype);
    }
    catch(vmp::exception &x)
    {
         delete obj;
         json::primitive_error(label); 
    }
}

void JData::new_table(vmp::str label,json::JTable *table,vmp::str jtype)
{
    vmp::except_check_pointer((void *) table,"json::JData::new_table() null pointer input table");
    json::JsonObj jbody;
    get_jbody("new_table",&jbody);
    json::JsonObj *obj=new json::JsonObj();
    jbody.add_object_obj(label,obj);
    table->set(obj,jtype); 
}
        
void JData::set_table(vmp::str label,json::JTable *table,vmp::str jtype)
{
    vmp::except_check_pointer((void *) table,"json::JData::set_table() null pointer input table");
    if(table->table_ == 0)
        vmp::except_check_pointer((void *) table,"json::JData::set_table() input table not setting");    
    if((jtype != "") && (jtype != table->jdata_type()))
        vmp::except_s("json::JData::set_table() input table bad type");
    json::JsonObj jbody;
    get_jbody("set_table",&jbody);
    jbody.add_object_ex(label,table->table_);
}

void JData::get_table(vmp::str label,json::JTable *table,vmp::str jtype)
{
    vmp::except_check_pointer((void *) table,"json::JData::get_table() null pointer input table");
    json::JsonObj jbody;
    get_jbody("get_table",&jbody);
    json::JsonObj *obj;
    try
    {
         obj=new json::JsonObj();
         jbody.get_object(label,obj);
         table->set(obj,jtype);
    }
    catch(vmp::exception &x)
    {
         delete obj;
         json::primitive_error(label); 
    }
}

void JData::set_args(vmp::str label,vmp::vector<vmp::str> args)
{
    json::JsonObj jbody,obj;
    get_jbody("set_args",&jbody);
    jbody.add_object_array_strings(label,args,&obj);
}
        
vmp::vector<vmp::str> JData::get_args(vmp::str label)
{
    json::JsonObj jbody;
    vmp::vector<vmp::str> ret;
    get_jbody("get_args",&jbody);
    try
    {
        ret=jbody.get_object_array_strings(label);
    }
    catch(vmp::exception &x)
    {
        json::primitive_error(label);   
    }
    return ret;
}

void JData::set_vars(vmp::str label,vmp::Table<vmp::str,vmp::str> *vars)
{
    vmp::except_check_pointer((void *) vars,"json::JData::get_table() null pointer input vars");    
    json::JsonObj jbody,table,obj;
    vmp::str tmp;
    get_jbody("set_vars",&jbody);
    jbody.add_object_obj(label,&table);
    vmp::vector<vmp::str> keys=vars->all_keys();
    for(vmp_index i=0;i<keys.size();i++)
    {
        vars->search(keys[i],&tmp);
        table.add_object_str(keys[i],tmp,&obj);
    }
}
        
void JData::get_vars(vmp::str label,vmp::Table<vmp::str,vmp::str> *vars)
{
    vmp::except_check_pointer((void *) vars,"json::JData::get_table() null pointer input vars");
    json::JsonObj jbody,table;
    get_jbody("get_vars",&jbody);
    vars->clear();
    try
    {
        jbody.get_object(label,&table);
        vmp::vector<vmp::str> keys=table.keys();
        for(vmp_index i=0;i<keys.size();i++)
            vars->insert(keys[i],table.get_object_str(keys[i]));
    }
    catch(vmp::exception &x)
    {
        vars->clear();
        json::primitive_error(label);   
    }
}

void JData::set_custom(vmp::str label,json::JsonObj *obj)
{
    vmp::except_check_pointer((void *) obj,"json::JData::set_custom() null pointer input obj");
    if(!obj->isobject())
        vmp::except("json::JData::set_custom() input obj bad object");
    json::JsonObj jbody;
    get_jbody("set_custom",&jbody);
    jbody.add_object_ex(label,obj);
}

void JData::get_custom(vmp::str label,json::JsonObj *obj)
{
    vmp::except_check_pointer((void *) obj,"json::JData::get_custom() null pointer input obj");
    json::JsonObj jbody,custom,tmp;
    get_jbody("get_custom",&jbody);
    try
    {
        if(!obj->isobject())
            vmp::except("");
        jbody.get_object(label,&custom);
        vmp::vector<vmp::str> keys=custom.keys();
        for(vmp_index i=0;i<keys.size();i++)
        {
            custom.get_object(keys[i],&tmp);
            obj->add_object_ex(keys[i],&tmp);
        }
    }
    catch(vmp::exception &x)
    {
        json::primitive_error(label);   
    }
}

void JData::set_buf(vmp::str label,vmp::Buf *buf)
{
    vmp::except_check_pointer((void *) buf,"json::JData::set_buf() null pointer input buf");
    json::JsonObj jbody,obj;
    get_jbody("set_buf",&jbody);
    buf->index();
    jbody.add_object_str(label,buf->read_xstr(buf->size()),&obj);  
}

void JData::get_buf(vmp::str label,vmp::Buf *buf)
{
    vmp::except_check_pointer((void *) buf,"json::JData::get_buf() null pointer input buf");
    json::JsonObj jbody;
    get_jbody("get_buf",&jbody);
    buf->reset();
    try
    {
        vmp::str value=jbody.get_object_str(label);
        buf->write_xstr(value);
        buf->index();
    }
    catch(vmp::exception &x)
    {
        json::primitive_error(label);
    }
}

void JData::set_netaddress(vmp::str label,net::Address *address)
{
    vmp::except_check_pointer((void *) address,"json::JData::set_netaddress() null pointer input address");
    json::JsonObj jbody,obj,obj2;
    get_jbody("set_netaddress",&jbody);
    jbody.add_object_obj(label,&obj);
    obj.add_object_str("host",address->host(),&obj2);
    obj.add_object_str("service",address->service(),&obj2);   
}

void JData::get_netaddress(vmp::str label,net::Address *address)
{
    vmp::except_check_pointer((void *) address,"json::JData::get_netaddress() null pointer input address"); 
    json::JsonObj jbody,obj;
    get_jbody("get_netaddress",&jbody);
    try
    {
         jbody.get_object(label,&obj);
         vmp::str host=obj.get_object_str("host");
         vmp::str service=obj.get_object_str("service");
         address->set(host,service);
    }
    catch(vmp::exception &x)
    {
        json::primitive_error(label);
    }
}

void JData::set_macaddress(vmp::str label,net::MacAddress *mac)
{
    vmp::except_check_pointer((void *) mac,"json::JData::set_macaddress() null pointer input mac");
    json::JsonObj jbody,obj;
    get_jbody("set_macaddress",&jbody);
    jbody.add_object_str(label,mac->str_addr(),&obj);
}
        
void JData::get_macaddress(vmp::str label,net::MacAddress *mac)
{
    vmp::except_check_pointer((void *) mac,"json::JData::get_macaddress() null pointer input mac");
    json::JsonObj jbody;
    get_jbody("get_macaddress",&jbody);
    try
    {
        mac->set(jbody.get_object_str(label));
    }
    catch(vmp::exception &x)
    {
        json::primitive_error(label);
    }
}

void JData::set_packet(vmp::str label,packet::Packet *packet)
{
    vmp::except_check_pointer((void *) packet,"json::JData::set_packet() null pointer input packet");
    json::JsonObj jbody,obj,obj2;
    vmp::Buf buf;
    packet::PacketHelper helper;
    get_jbody("set_packet",&jbody);
    jbody.add_object_obj(label,&obj);
    obj.add_object_str("basetype",packet->type(),&obj2);
    helper.write(packet,&buf);
    buf.index();
    obj.add_object_str("packet",buf.read_xstr(buf.size()),&obj2);
}

packet::Packet *JData::get_packet(vmp::str label)
{
    packet::Packet *ret;
    json::JsonObj jbody,obj;
    vmp::Buf buf;
    packet::PacketHelper helper;
    get_jbody("get_packet",&jbody);
    try
    {
         jbody.get_object(label,&obj);
         vmp::str basetype=obj.get_object_str("basetype");
         vmp::str value=obj.get_object_str("packet");
         buf.write_xstr(value);
         ret=helper.read(basetype,&buf);
    }
    catch(vmp::exception &x)
    {
        json::primitive_error(label);
    }
    return ret;
}

}}

