/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 * Date last update: 27/06/2020
 */

#include "openssl0/openssl0.h"

namespace vampiria { namespace openssl { namespace pkg {

static vmp_bool init_=false;

void init()
{
    if(! init_)
    {
        ERR_load_crypto_strings();
        SSL_load_error_strings();
        OpenSSL_add_all_algorithms();
        ERR_load_BIO_strings();
        if(OPENSSL_init_ssl(0, NULL) < 0)
            vmp::except_s("openssl::pkg::init Could not initialize the library"); 
        init_=true;
    }
}

void end()
{
    if(init_)
    {
        ERR_free_strings();
        FIPS_mode_set(0);
        EVP_cleanup();
        CRYPTO_cleanup_all_ex_data();
        ERR_free_strings();
        init_=false;
    }
}

Ctx::Ctx()
{
    ctx_=0;
}
        
Ctx::~Ctx()
{
    ctx_=0;
}

openssl::pkg::Ctx *ctx_tls_generic()
{
    openssl::pkg::Ctx *ret=0;
    SSL_CTX *ctx=SSL_CTX_new(TLS_method());
    if(ctx == 0)
        vmp::except("openssl::pkg::ctx_tls_generic error '%s'",openssl::pkg::err_string().c_str());
    ret=new openssl::pkg::Ctx();
    ret->ctx_=ctx;
    return ret;
}

openssl::pkg::Ctx *ctx_tls_client()
{
    openssl::pkg::Ctx *ret=0;
    SSL_CTX *ctx=SSL_CTX_new(TLS_client_method());
    if(ctx == 0)
        vmp::except("openssl::pkg::ctx_tls_client() error '%s'",openssl::pkg::err_string().c_str());
    ret=new openssl::pkg::Ctx();
    ret->ctx_=ctx;
    return ret;
}

openssl::pkg::Ctx *ctx_tls_server()
{
    Ctx *ret;
    SSL_CTX *ctx=SSL_CTX_new(TLS_server_method());
    if(ctx == 0)
        vmp::except("openssl::pkg::ctx_tls_server() error '%s'",openssl::pkg::err_string().c_str());
    ret=new Ctx();
    ret->ctx_=ctx;
    return ret;
}

void ctx_pem_pkey_file(openssl::pkg::Ctx *ctx,vmp::str fpath)
{
    if((ctx == 0) || (ctx->ctx_ == 0))
        vmp::except_s("openssl::pkg::ctx_pem_pkey_file() error 'context input null'");
    if(SSL_CTX_use_PrivateKey_file(ctx->ctx_,fpath.c_str(),SSL_FILETYPE_PEM) <= 0)
        vmp::except("openssl::pkg::ctx_pem_pkey_file() error '%s'",openssl::pkg::err_string().c_str()); 
}

void ctx_pem_x509_file(openssl::pkg::Ctx *ctx,vmp::str fpath)
{
    if((ctx == 0) || (ctx->ctx_ == 0))
        vmp::except_s("openssl::pkg::ctx_pem_x509_file() error 'context input null'");
    if (SSL_CTX_use_certificate_file(ctx->ctx_,fpath.c_str(),SSL_FILETYPE_PEM) <= 0)
        vmp::except("openssl::pkg::ctx_pem__file() error '%s'",openssl::pkg::err_string().c_str());  
}

void ctx_free(openssl::pkg::Ctx *ctx)
{
    if(ctx != 0)
    {    
        if(ctx->ctx_ != 0)
            SSL_CTX_free(ctx->ctx_);
        delete ctx;    
    }  
}

Ssl::Ssl(openssl::pkg::Ctx *ctx)
{
    vmp::except_check_pointer((void *)ctx,"openssl::pkg::Ssl() null input context");
    ctx_=ctx;
    ssl_=SSL_new(ctx_->ctx_);
    socket_=-1;
    reset();
}

Ssl::~Ssl()
{
    close();
    //SSL_clear(ssl_);//bugs segmentation fault
    //SSL_free(ssl_); //bugs segmentation fault
    vmp::warning("SSL_free Bugs segmentation fault in some cases");
    ssl_=0;
    ctx_=0;
}

void Ssl::reset()
{
    close();
    blocking_=true;
    state_=openssl::pkg::SSL_STATE_INIT;
}

net::Socket Ssl::socket()
{
    return socket_;
}

void Ssl::block()
{
    blocking_=true;
}
        
void Ssl::noblock()
{
    blocking_=false;
}

vmp::str Ssl::type()
{
    return type_;
}

void Ssl::get_local_address(net::Address *result)
{
    net::socket_addrlocal(socket_,result);
}

void Ssl::get_peer_address(net::Address *result)
{
    net::socket_addrpeer(socket_,result);
}

vmp_int Ssl::sendData(vmp::Buf *buf)
{
    vmp_int ret;
    if(state_ != openssl::pkg::SSL_STATE_READY)
        vmp::except_s("openssl::pkg::Ssl::sendData() invalid state at the call point");
    vmp::except_check_pointer((void *)buf,"openssl::pkg::Ssl::sendData() null pointer input buffer");
    ret=SSL_write(ssl_,(void *)buf->pointer(),buf->size());
    if(ret < 0)
    {
        state_=openssl::pkg::SSL_STATE_RCLOSE;
        vmp::except_s(openssl::pkg::err_string().c_str());
    }
    return ret;
}

vmp_int Ssl::recvData(vmp::Buf *buf)
{
    if(state_ != openssl::pkg::SSL_STATE_READY)
        vmp::except_s("openssl::pkg::Ssl::recvData() invalid state at the call point");
    vmp::except_check_pointer((void *)buf,"openssl::pkg::Ssl::recvData() null pointer input buffer");
    vmp_int tmp,retval=0;
    buf->reset();
    while(1)
    {
        buf->newsize(buf->size()+vmp::MAXSIZEREAD);
        tmp= SSL_read(ssl_,buf->pointer(retval),vmp::MAXSIZEREAD);
        if(tmp == -1)
        {
             vmp_uint err= SSL_get_error(ssl_,tmp);
             if((err == SSL_ERROR_WANT_READ) || (err == SSL_ERROR_WANT_WRITE))
             {
                 if(retval==0)
                     return -1;
                 else
                     break;
             }
	     state_=openssl::pkg::SSL_STATE_RCLOSE;
	     buf->reset();
             vmp::except_s(openssl::pkg::err_string().c_str());
        }
        retval+=tmp;
        if((vmp_uint) tmp < vmp::MAXSIZEREAD)
            break;
    }
    if(retval == 0)
       state_=openssl::pkg::SSL_STATE_RCLOSE;
    else
       buf->newsize(retval);
    return retval;
}

void Ssl::key_update()
{
    if(state_ != openssl::pkg::SSL_STATE_READY)
        vmp::except_s("openssl::pkg::Ssl::key_update() invalid state at the call point");
    if(SSL_version(ssl_) <= TLS1_2_VERSION)
        SSL_renegotiate_abbreviated(ssl_);
    else
        SSL_key_update(ssl_,SSL_KEY_UPDATE_REQUESTED);
}

void Ssl::get_peer_cerificate(openssl::pkg::X509_Wrap *cout)
{
    vmp::except_check_pointer((void *)cout,"openssl::pkg::Ssl::get_peer_cerificate() null pointer input certificate");
    X509 *cert=SSL_get_peer_certificate(ssl_);
    vmp::except_check_pointer((void *)cert,"No certificate was presented by the peer or no connection was established");
    cout->set(cert);   
}

vmp_bool Ssl::shutdown()
{
    if(state_ == openssl::pkg::SSL_STATE_READY )
    {
        SSL_set_shutdown(ssl_,SSL_RECEIVED_SHUTDOWN);
        state_ = openssl::pkg::SSL_STATE_SHUTDOWN;
    }
    if(state_ == openssl::pkg::SSL_STATE_SHUTDOWN)
    {
	vmp_int ret=SSL_shutdown(ssl_) ;
	if(ret == 1)
	    return true;
	else if  (ret < 0)
	{
	     vmp_uint err= SSL_get_error(ssl_,ret);
             if(!((err == SSL_ERROR_WANT_READ) || (err == SSL_ERROR_WANT_WRITE)))
             {
                  state_=openssl::pkg::SSL_STATE_RCLOSE;
	          vmp::except_s(openssl::pkg::err_string().c_str());
	     }		     
	}
    }
    return false;    
}

void Ssl::close()
{
    if(state_ == openssl::pkg::SSL_STATE_CLOSE)
        return;
    state_=openssl::pkg::SSL_STATE_CLOSE;
    net::socket_close(&socket_);   
}

SslClient::SslClient(openssl::pkg::Ctx *ctx):openssl::pkg::Ssl(ctx)
{
     type_="client";
}

SslClient::~SslClient()
{
}

void SslClient::connect(net::Address *server)
{
    if((server == 0) || (server->iptype() == PF_UNSPEC) || (server->service() == "0"))
        vmp::except_s("openssl::pkg::SslClient::connect() invalid address connect");
    if(state_ == openssl::pkg::SSL_STATE_CLOSE)
        state_ = openssl::pkg::SSL_STATE_INIT;
    if(state_ != openssl::pkg::SSL_STATE_INIT)
        vmp::except_s("openssl::pkg::SslClient::connect() invalid state at the call point");
    socket_=net::socket_stream(server->iptype(),blocking_);
    if(net::socket_connect(socket_,server))
    {    
        SSL_set_fd(ssl_,socket_);
        state_=openssl::pkg::SSL_STATE_HANDSHAKE;
    }
    else
        state_=openssl::pkg::SSL_STATE_CONNECT;
}

vmp_bool SslClient::connect_check()
{
    if(state_ == openssl::pkg::SSL_STATE_HANDSHAKE)
        return true;
    if(state_ != SSL_STATE_CONNECT) 
        vmp::except_s("openssl::pkg::SslClient::connect_check() invalid state at the call point");
    if(net::socket_connect_check(socket_))
    {
        SSL_set_fd(ssl_,socket_);
        state_=openssl::pkg::SSL_STATE_HANDSHAKE;
        return true;
    }
    return false;
}

vmp_bool SslClient::ssl_connect()
{
    if(state_ != SSL_STATE_HANDSHAKE) 
        vmp::except_s("openssl::pkg::SslClient::ssl_connect() invalid state at the call point");
    vmp_int ret=SSL_connect(ssl_);
    if(ret == 1)
    {
        state_=openssl::pkg::SSL_STATE_READY;
        return true;
    }
    vmp_uint err= SSL_get_error(ssl_,ret);
    switch(err)
    {
        case SSL_ERROR_WANT_READ: break;
        case SSL_ERROR_WANT_WRITE: break;
        default: vmp::except_s(openssl::pkg::err_string().c_str());
    }
    return false;
}

void SslClient::get_peer_cerificate(openssl::pkg::X509_Wrap *cout)
{
    vmp::except_check_pointer((void *)cout,"openssl::pkg::Ssl::get_peer_cerificate() null pointer input certificate");
    STACK_OF(X509) *chain;
    chain=SSL_get_peer_cert_chain(ssl_);
    vmp::except_check_pointer((void *)chain,"No certificate was presented by the peer or no connection was established");
    vmp_size num = sk_X509_num(chain);
    openssl::pkg::X509_Wrap *tmp;
    for(vmp_index i=0;i<num;i++)
    {
        if(i == 0)
        {
            cout->set(sk_X509_value(chain,i));
            tmp=cout;
        }
        else
        {
            tmp->parent_=new X509_Wrap();
            tmp=tmp->parent_;
            tmp->set(sk_X509_value(chain,i));
        }
    }
}

SslServer::SslServer(vmp_int socket,openssl::pkg::Ctx *ctx):openssl::pkg::Ssl(ctx)
{
    type_="server";
    socket_=socket;
    SSL_set_fd(ssl_,socket);
    state_=openssl::pkg::SSL_STATE_HANDSHAKE;
}

SslServer::~SslServer()
{
}

vmp_bool SslServer::ssl_accept()
{
    if(state_ != SSL_STATE_HANDSHAKE) 
        vmp::except_s("openssl::pkg::SslServer::ssl_accept() invalid state at the call point");
    vmp_int ret=SSL_accept(ssl_);
    if(ret == 1)
    {
        state_=openssl::pkg::SSL_STATE_READY;
        return true;
    }
    vmp_uint err= SSL_get_error(ssl_,ret);
    switch(err)
    {
        case SSL_ERROR_WANT_READ: break;
        case SSL_ERROR_WANT_WRITE: break;
        default: vmp::except_s(openssl::pkg::err_string().c_str());
    }
    return false;
}

void sslserver_free_connection(openssl::pkg::SslServer *server)
{
    if(server != 0)
        delete server;    
}

SslListen::SslListen(openssl::pkg::Ctx *ctx):openssl::pkg::Ssl(ctx)
{
    type_="listen";
}
        
SslListen::~SslListen()
{
}

void SslListen::server(net::Address *local,vmp_uint backlog)
{
    if((local == 0) || (local->iptype() == PF_UNSPEC) || (local->service() == "0"))
        vmp::except("openssl::pkg::SslListen::server() invalid address listen");
    if(state_ != SSL_STATE_INIT)
        vmp::except_s("openssl::pkg::SslClient::server() invalid state at the call point");
    socket_=net::socket_stream(local->iptype(),blocking_);
    net::socket_bind(socket_,local);
    net::socket_listen(socket_,backlog);
    state_=SSL_STATE_LISTEN;
}

openssl::pkg::SslServer *SslListen::accept()
{
    if(state_ != SSL_STATE_LISTEN)
        vmp::except_s("openssl::pkg::SslListen::accept() invalid state at the call point");
    vmp_int ret=net::socket_accept(socket_,blocking_);
    if(ret == -1)
        return 0;
    openssl::pkg::SslServer *server=new openssl::pkg::SslServer(ret,ctx_);
    return server;    
}

void SslListen::get_peer_address(net::Address *result)
{
    vmp::except_s("Listen ssl has no remote peer address");
}

void SslListen::key_update()
{
    vmp::except_s("Listen ssl not key update operation");
}

void SslListen::get_peer_cerificate(openssl::pkg::X509_Wrap *cout)
{
    vmp::except_s("Listen ssl not peer certificate associated");
}

vmp_bool SslListen::shutdown()
{
    return true;
}
}}}

