/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 01/01/2020
 */

#include <regex>

#ifndef VAMPIRIA_VMP_UNICODE_H

#define VAMPIRIA_VMP_UNICODE_H 1

namespace vampiria { namespace vmp { namespace unicode {

//!Writes the format value(printf style) in the string
/*!
     @param pstr pointer to string
     @param fmt format string
     @param ... format args
     @return string size
*/
vmp_size str_write(vmp::str *pstr,const vmp_char *fmt,...);

//!Concatenate the format value(printf style) in the string
/*!
     @param pstr pointer to string
     @param fmt format string
     @param ... format args
     @return string size
*/
vmp_size str_cwrite(vmp::str *pstr,const vmp_char *fmt,...);

//! Split string
/*!
     @param str input string
     @param delimiter token value
     @return split strings
*/
vmp::vector<vmp::str> str_split(vmp::str str,vmp::str delimiter);

//! Creates a string concatenate strings in list, inserting the token between the strings,if delimiter=="" return charatters list
/*!
     @param list List of strings to be concatenated
     @param token String to be inserted between the strings to be concatenated
     @return The resulting string                        
*/
vmp::str str_join(vmp::vector<vmp::str> list,vmp::str token);

//! Check if the byte input is an integer
/*!
     @param byte input
     @return true ok,otherwise false 
*/
vmp_bool byte_isdigit(vmp_byte byte);

//! Check if the byte input is a hexadecimal integer
/*!
     @param byte input
     @return true ok,otherwise false 
*/
vmp_bool byte_isxdigit(vmp_byte byte);

//! Check if the string input is an integer
/*!
     @param str string input
     @return true ok,otherwise false 
*/
vmp_bool str_isdigit(vmp::str str);

//! Check if the string input is a hexadecimal integer
/*!
     @param str string input
     @return true ok,otherwise false 
*/
vmp_bool str_isxdigit(vmp::str str);

//! Check if the string input is an real
/*!
     @param str string input
     @return true ok,otherwise false 
*/
vmp_bool str_isreal(vmp::str str);

//! Converts the string input to int
/*!
     @param istr string input
     @return The converted value.(except error)
*/ 
vmp_int str_todigit(vmp::str istr);

//! Converts the string input to int in range min-max
/*!
     @param istr string input
     @param min value accepted
     @param max value accepted
     @return The converted value.(except error)
*/ 
vmp_int str_todigit_range(vmp::str istr,vmp_int min,vmp_int max);

//! Converts the string input to real
/*!
     @param dstr string input
     @return The converted value.(except error)
*/ 
vmp_real str_toreal(vmp::str dstr);

//! Converts the string input to real in range min-max
/*!
     @param dstr string input
     @param min value accepted
     @param max value accepted
     @return The converted value.(except error)
*/ 
vmp_real str_toreal_range(vmp::str dstr,vmp_real min,vmp_real max);

//! Converts a string of exadecimal values in the byte vector (2 characters 0-F 1 byte)
/*!
     @param xstr string input
     @return byte vector(except error)
*/
vmp::vector<vmp_byte> xstr_tobytes(vmp::str xstr);

//! Converts a bytes vector to a string of hexadecimal values
/*!
     @param bstr byte vector input
     @return The converted string value.
*/
vmp::str bytes_toxstr(vmp::vector<vmp_byte> bytes);

//! Converts a bytes vector to a string of hexadecimal values in human mode(the reverse function does not work)
/*!
     @param bstr byte vector input
     @param delimiter byte delimiter
     @return The converted string value in human mode.
*/
vmp::str bytes_toxstr_hm(vmp::vector<vmp_byte> bytes,vmp::str delimiter);

//! Converts a string in normal format to a string in hexadecimal format
/*!
     @param str input string
     @param delimiter byte delimiter
     @return The converted string value
*/
vmp::str str_toxstr(vmp::str str,vmp::str delimiter);

//!Convert a string from hexdecimal format to a string in normal format
/*!
     @param xstr input string hexadecimal format
     @param delimiter byte delimiter
     @return The converted string value,throw exception to error
*/
vmp::str xstr_tostr(vmp::str xstr,vmp::str delimiter);

//! Search substring sub in str
/*!
    @param str input string
    @param sub substring to search
    @return true if found sub in str otherwise false
*/
vmp_bool str_findsub(vmp::str str,vmp::str sub);

//! Check if a string has ascii code characters <= 0x20 or >= 0x7E(\ n, \ t DEL SPACE etc ..)
/*!
    @param str string input
    @return true If there are no characters <= 0x20 or >= 7E and word size > 0,otherwise false
*/
vmp_bool str_isword(vmp::str str);

//! Format the input string by replacing all characters with ascii code <= 0x20 or >= 0x7E with a space
/*!
    @param content string input
    @return formatted string
*/
vmp::str str_format(vmp::str content);

//! Format the input string by replacing all characters with ascii code <= 0x20 with a space or \n if the line size exceeds maxline 
/*!
    @param content string input
    @param maxline max line size
    @return formatted string
*/
vmp::str str_format_maxline(vmp::str content,vmp_size maxline);

//! Remove all characters with ascii code <= 0x20 or >= 0x7E at the end of the string
/*!
    @param content input string
    @return formatted string
*/
vmp::str str_format_end(vmp::str content);

//! Extracts all the characters in the string clist from the string str and returns them
/*!
    @param str input string
    @param clist char extract string
    @return characters found
*/
vmp::str str_extract_char(vmp::str str,vmp::str clist);

//!Check if there are regular expressions of types Basic POSIX grammar and case insensitive in a string and return them
/*!
    @param str string to analize
    @param str_regex regular espression to search
    @return the vector of all matched strings
*/
vmp::vector<vmp::str> str_regex_posix_icase(vmp::str str,vmp::str str_regex);

}}}

#endif


