/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 03/02/2020
 */

#ifndef VAMPIRIA_PYLIB_MODULE_H

#define VAMPIRIA_PYLIB_MODULE_H 1

namespace vampiria { namespace pylib {

//!Module c++ to python wrapper utils(see pylib::ObjBase)
class Module:public pylib::Objbase
{
    public:
	//! A constructor
        /*!
            @param ref references python wrapper class
            @param name Python class name
            @param prefix Module path in which the class is inserted
        */ 
        Module(pylib::Objref *ref,vmp::str name,vmp::str prefix="");
        
        //!A destructor
         ~Module();      

        //! Operator =
	const Module &operator=(Module &module); 
    
        //! Adds a submodule to the module
        /*!
            @param name submodule name
            @param ret submodule wrapper module object
        */
        Module add_submodule(vmp::str name);
    
        //! Adds a constant value to the module
	/*!
            @param name constant name
            @param value constant value
	*/
	template<typename T>
        void add_const(vmp::str name,T value)
        {
	    pylib::Objref *obj=pylib::Converter<T>(value);
            pylib::pymodule_addobj(ref(),name,obj);
	}
	
	//! Inserts a python function into the module following the code c++ of a stub
	/*!
	    @param name function name
	    @param stub stub (see pylib::Generic_def)
            @return void(except error)	
	*/
	template<class Stub_t>
        void def_with_stub(vmp::str name,Stub_t const &stub)
        {
            pylib::Def *def;
            py_function *p=new py_function(stub);
            try
            {
                def=(pylib::Def *)pyobj_getattr<pylib::Objref *>(ref(),name);
                def->insert_py(p);
            }
            catch(vmp::exception &x)
            {
                def=new pylib::Def(p,name,(*this));
            }
       }

       //! Inserts a python function into the module (wrapper c++ to python function)
       /*!
           @param name  function name
           @param fn function c++ to wrap
           @param retfree Converter return free arg(see pylib::Converter) 
           @return void(except error)	
       */
       template <class Fn>
       void def_(vmp::str name,Fn fn,vmp_bool retfree=true)
       {
           def_with_stub(name,pylib::Def_default<Fn>(fn,retfree));
       }

       //! Inserts a python class into the module following the code c++ of a stub
       /*!
	   @param name class name
	   @param stub constructor stubf(see pylib::Constructor)
           @return class wrapper(except error)
       */
       template<class Wrap,class Constructor_t>
       pylib::Class<Wrap> class_with_constructor(vmp::str name,Constructor_t const &constructor)
       {
           vmp::str cname,prefix=fullname();
           if(pyobj_getattr_obj(ref(),name) != 0)
               vmp::except("In Module '%s' declaretion duplicate class '%s'",prefix.c_str(),name.c_str());
           vmp::unicode::str_write(&cname,"<class %s>",name.c_str());
           pylib::Objref *cref=pylib::class_metatype_eval(name,prefix);
           pyobj_setattr<vmp_size>(cref,name,(vmp_size) sizeof(Wrap));
           pylib::type_info ctype=pylib::unwrap_type_id((Wrap*)0);
           RegClass *r=Registry::lookup(ctype);
           pylib::incref(cref);
           r->p_obj_=(pylib::Objtype *) cref;            
           pylib::pyobj_setattr<pylib::Objref *>(ref(),name,cref);
           py_function *p=new pylib::py_function(constructor);
           pylib::Class<Wrap> ret(cref,cname,prefix);
           new Def(p,"__init__",ret);
           return ret;
       }

       //! Inserts a python classs into the module (wrapper c++ to python class with constructor no parameter)
       /*!
           @param name  class name
           @return class wrapper(except error)
       */
       template<class Wrap>
       pylib::Class<Wrap> class_(vmp::str name)
       {
           return class_with_constructor<Wrap>(name,pylib::Constructor_default<Wrap>());
       } 

       //! Inserts a python class in the module that cannot be built dynamically
       /*!
           @param name  class name
           @return class wrapper(except error)
       */
       template<class Wrap>
       pylib::Class<Wrap> class_no_constructor(vmp::str name)
       { 
           return class_with_constructor<Wrap>(name,pylib::Constructor_no_allocated<Wrap>());
       }
};

//!Create the root module (used in the main macro)
/*!
    @param name root module name
    @return module wrapper 
*/
pylib::Module init_module(vmp::str name);

}}

#endif

