/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 12/02/2020
 */

#include "process.h"

namespace vampiria { namespace process {

static vmp::Pid rootpid_;

static void box_signal_run(vmp_int sig)
{
}

static void box_signal_init(vmp_int sig)
{
    if(rootpid_ != vmp::getpid_wrap())
        vmp::exit_ok();
}

BoxManager::BoxManager(process::BOX_FREE boxfree)
{
    rootprocess_=false;
    boxfree_=boxfree;
}

BoxManager::~BoxManager()
{
}

void BoxManager::add_box(process::Box *box,vmp_bool is_root,process::BOX_COM ctype)
{
    process::Box *tmp;
    vmp::except_check_pointer((void *)box,"process::BoxManager::add_box() null input box");
    if(box_.search(box->id_,&tmp))
        vmp::except("process::BoxManager::add_box() duplicate box id '%s'",box->id_.c_str());
    if(!box->is_setting())
        vmp::except("process::BoxManager::add_box() box id '%s' not set",box->id_.c_str());
    if(box->is_running())
        vmp::except("process::BoxManager::add_box() box id '%s' already running process",box->id_.c_str());
    if(is_root) 
    {   
        if(rootprocess_)
            vmp::except("process::BoxManager::add_box() box id '%s' duplicate root process",box->id_.c_str());
        rootprocess_=true;
    }
    box->set_input(is_root);
    box->set_output(ctype);
    if (box->run() == 0)
    {
        try
        {
            vmp::vector<process::Box *> vbox=box_.all_data();
            for(vmp_index i=0;i<vbox.size();i++)
            {
                vbox[i]->close_input_w();
                vbox[i]->close_output_r();    
            }
            box->routine();
        }
        catch(vmp::exception& x) 
        {
            vmp::exit_failure("Process pid %d error %s",box->pid_,x.what());
        }
        
    }
    box_.insert(box->id_,box);
}

void BoxManager::ctl_process()
{
    vmp::vector<vmp::str> sig;
    sig.push_back("sigterm");
    sig.push_back("sigint");
    sig.push_back("sigchld");
    vmp::signal_wrap(sig,process::box_signal_run);
    vmp::Select_FD select_fd;
    process::Box *tmp;
    vmp::vector<process::Box *> vbox=box_.all_data();
    while(vbox.size() > 0)
    {
        select_fd.reset();
        for(vmp_index i=0;i<vbox.size();i++)
        {
            vmp::Pid pid=vbox[i]->pid();
            if(vbox[i]->end())
            {
                 box_.cancel(vbox[i]->id_,&tmp);
                 if(boxfree_ != 0)
                     (*boxfree_)(pid,tmp);
                 vbox[i] = 0;
            }
            else
            {
                 select_fd.fd_add(vbox[i]->stdoutr_);
                 select_fd.fd_add(vbox[i]->stderrr_);
            }
        }
        try
        {
            select_fd.fd_monitor();
        }
        catch(vmp::exception &x)
        {
            break;
        }
        for(vmp_index i=0;i<vbox.size();i++)
            if(vbox[i] != 0)
               vbox[i]->read_data();  
        vbox.clear();
        vbox=box_.all_data();  
    }  
}

void BoxManager::close_all()
{
    vmp::vector<process::Box *> vbox=box_.all_data();
    for(vmp_index i=0;i<vbox.size();i++)
    {
        vbox[i]->kill("sigint");
        vmp::Pid pid=vbox[i]->pid_;
        vmp::waitpid_wrap(&pid);
        vbox[i]->close_input_w();
        vbox[i]->close_output_r();
        if(boxfree_ != 0)
            (*boxfree_)(pid,vbox[i]);
    }
    vbox.clear();
}

Box::Box(vmp::str id)
{
    id_=id;
    setting_=false;
    rootdir_="";
    routine_=0;
    output_=0;
    error_=0;
    end_=0;

    pid_ = -1;
    stdinr_=-1;
    stdinw_=-1;
    streamw_=0;
    stdoutr_=-1;
    stdoutw_=-1;
    stderrr_=-1;
    stderrw_=-1;
}
       
Box::~Box()
{
    setting_=false;
    rootdir_="";
    routine_=0;
    output_=0;
    error_=0;
    end_=0;
}

void Box::set_input(vmp_bool is_root)
{
    if(!is_root)
    {
        vmp::pipe_wrap(&stdinr_,&stdinw_);
        streamw_=vmp::fd_stream(stdinw_);   
    }
}

void Box::set_output(process::BOX_COM ctype)
{
    if(ctype == process::BOX_PIPE)
    {
        vmp::pipe_wrap(&stdoutr_,&stdoutw_);
        vmp::pipe_wrap(&stderrr_,&stderrw_);  
    }
    else if(ctype == process::BOX_PTS)
    {
        vmp::pty_wrap(&stdoutw_,&stdoutr_);
        vmp::pty_wrap(&stderrw_,&stderrr_);
    }
    vmp::fd_noblock(stdoutr_);
    vmp::fd_noblock(stderrr_);
}

vmp::Pid Box::run()
{
    pid_=vmp::fork_wrap();
    if(pid_ == 0)
    {
        vmp::fs::chdir_wrap(rootdir_);
        pid_=vmp::getpid_wrap();
        close_input_w();
        close_output_r();
        if(stdinr_ != -1)
            vmp::dup2_wrap(stdinr_,STDIN_FILENO);
        vmp::dup2_wrap(stdoutw_,STDOUT_FILENO);
        vmp::dup2_wrap(stderrw_,STDERR_FILENO);
        return 0;
    }
    close_input_r();
    close_output_w();
    return pid_;
}

vmp_bool Box::end()
{
    if((stdoutr_ == -1) && (stderrr_ == -1))
    {
        vmp::Pid pid=pid_;
        vmp_int status=vmp::waitpid_wrap(&pid_);
        if(end_ != 0)
            (*end_)(pid,this,status);
        return true;
    }
    return false;   
}

void Box::routine()
{
    if(routine_ != 0)
    {   
        routine_(this);
        vmp::exit_ok();
    }
    vmp::except_s("process::Box::routine() null pointer callback");
}

void Box::read_data()
{
    vmp::str data;
    vmp_int retcode;
    if(stdoutr_ != -1)
    {
        try
        {
            while((retcode=vmp::fd_read(stdoutr_,&data)) > 0)
                if(output_ != 0)
                    (*output_)(this,data);
            if(retcode == 0)
                vmp::fd_close(&stdoutr_);
        }
        catch(vmp::exception &x)
        {
        }
    }
    if(stderrr_ != -1)
    {
        try
        {        
            while((retcode=vmp::fd_read(stderrr_,&data)) > 0)
                if(error_ != 0)
                    (*error_)(this,data);
            if(retcode == 0)
                vmp::fd_close(&stderrr_);
        }
        catch(vmp::exception &x)
        {
        }
    }
    
}

void Box::set(vmp::str rootdir,process::BOX_ROUTINE routine,process::BOX_STD output,process::BOX_STD error,process::BOX_END end)
{
    setting_=true;
    rootdir_=rootdir;
    routine_=routine;
    output_=output;
    error_=error;
    end_=end;
}

vmp_bool Box::is_setting()
{
    return setting_;
}

vmp_bool Box::is_running()
{
     if(pid_ != -1)
         return true;
     return false;
}

void Box::close_input_r()
{
    vmp::fd_close(&stdinr_);
}
       
void Box::close_input_w()
{
    vmp::stream_close(&streamw_);
    vmp::fd_close(&stdinw_);
}
       
void Box::close_output_r()
{
    vmp::fd_close(&stdoutr_);
    vmp::fd_close(&stderrr_);
}
       
void Box::close_output_w()
{
    vmp::fd_close(&stdoutw_);
    vmp::fd_close(&stderrw_);
}

void Box::kill(vmp::str signal)
{
    if(pid_ != -1)
       vmp::kill_wrap(pid_,signal);
}

vmp::Pid Box::pid()
{
    return pid_;
}

vmp::str Box::rootdir()
{
    return rootdir_;
}

vmp::str Box::id()
{
    return id_;
}

void Box::write_data(vmp::str data)
{
    vmp::stream_write(streamw_,data);
}

process::BoxManager *BoxManager_new(process::BOX_FREE boxfree)
{
    rootpid_=vmp::getpid_wrap();
    BoxManager *ret=new BoxManager(boxfree);
    vmp::vector<vmp::str> sig;
    sig.push_back("sigterm");
    sig.push_back("sigint");
    vmp::signal_wrap(sig,process::box_signal_init);
    return ret;
}

void BoxManager_free(process::BoxManager *manager)
{
    if(manager != 0)
    {
        manager->close_all();
        delete manager;
    }
}

}}

