/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 17/11/2020
 */

#include "packet.h"

namespace vampiria { namespace packet {

DataSize::DataSize(vmp::str field,vmp_size datalen):packet::DataBytes(field)
{
    data_=0;
    if(datalen < 1 || datalen > 4)
        vmp::except_s("Internal error (packet::DataSize())");
    datalen_=datalen;
}

DataSize::~DataSize()
{
}

vmp::str DataSize::get(vmp::str field)
{
    vmp::str ret;
    vmp::unicode::str_write(&ret,"%u",data_);
    return ret;
}
        
void DataSize::set(vmp::str field,vmp::str data)
{
    try
    {
        data_=vmp::unicode::str_todigit_range(data,0,(vmp_int)(vmp::math::pow_wrap(2,(datalen_*8)) -1));
    }
    catch(vmp::exception &x)
    {
        vmp::except_s("Malformed data size");
    }
}
        
void DataSize::read(vmp::Buf *buf)
{
    data_=buf->read_size(datalen_);
}
        
void DataSize::write(vmp::Buf *buf)
{
    buf->write_size(data_,datalen_);
}
        
vmp_size DataSize::get_data()
{
    return data_;
}

void DataSize::set_data(vmp_size data)
{
    if(data > (vmp::math::pow_wrap(2,(datalen_*8)) -1))
        vmp::except_s("Data too large");
    data_=data;
}

DataHexSize::DataHexSize(vmp::str field,vmp_size datalen):packet::DataSize(field,datalen)
{
}
        
DataHexSize::~DataHexSize()
{
}

vmp::str DataHexSize::get(vmp::str field)
{
    vmp::str ret="";
    switch(datalen_)
    {
        case 1:
           vmp::unicode::str_write(&ret,"%02X",data_);
           break;
        case 2:
           vmp::unicode::str_write(&ret,"%04X",data_);
           break;
        case 3:
           vmp::unicode::str_write(&ret,"%06X",data_);
           break;
        case 4:
           vmp::unicode::str_write(&ret,"%08X",data_);
           break;
        default:
           break;
    }
    return ret;
}
        
void DataHexSize::set(vmp::str field,vmp::str data)
{
    try
    {
        if(data.size() != (datalen_ * 2))
            vmp::except_s("");
        vmp::vector<vmp_byte> bytes=vmp::unicode::xstr_tobytes(data);
        data_=0;
        for(vmp_index i=0;i<datalen_;i++)
        {    
            data_=data_ << 8;
            data_ += bytes[i];
        }
    }
    catch(vmp::exception &x)
    {
        vmp::except_s("Malformed data hex size");
    }
}

DataSelect::DataSelect(vmp::str field,vmp_size datalen):packet::DataSize(field,datalen)
{
}

DataSelect::~DataSelect()
{
    vmp::table_delete_alldata<vmp_size,vmp::str *>(code_);
    vmp::table_delete_alldata<vmp::str,vmp_size *>(strcode_);
}

vmp::str DataSelect::get(vmp::str field)
{
    vmp::str ret="";
    vmp::str *pret;
    if(code_.search(data_,&pret))
        ret=(*pret);
    else
    {
        switch(datalen_)
        {
            case 1:
                vmp::unicode::str_write(&ret,"%02X",data_);
                break;
            case 2:
                vmp::unicode::str_write(&ret,"%04X",data_);
                break;
            case 3:
                vmp::unicode::str_write(&ret,"%06X",data_);
                break;
            case 4:
                vmp::unicode::str_write(&ret,"%08X",data_);
                break;
            default:
                break;
       }
    }
    return ret; 
}
        
void DataSelect::set(vmp::str field,vmp::str data)
{
    vmp_size *pcode,code;
    if(strcode_.search(data,&pcode))
        code=(*pcode);
    else
    {
        try
        {
            if(data.size() != (2*datalen_))
                vmp::except_s("");
            code =0;
            vmp::vector<vmp_byte> bytes=vmp::unicode::xstr_tobytes(data);
            for(vmp_index i=0;i<datalen_;i++)
                code=(code << 8)+bytes[i]; 
        } 
        catch(vmp::exception &x)
        {
            vmp::except_s("Malformed Data Select");
        }

    }
    set_data(code);
}

void DataSelect::insert_code(vmp::str strcode,vmp_size code)
{
    if(code > (vmp::math::pow_wrap(2,(datalen_*8)) -1))
        vmp::except_s("Code too large");
    vmp::str *str=new vmp::str();
    (*str)=strcode;
    vmp_size *c=new vmp_size();
    (*c)=code;
    strcode_.insert(strcode,c);
    code_.insert(code,str);
       
}

}}
