/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 13/11/2020
 */

#include "vmp.h"
#include "net.h"

#include "packet/datatypes.h"

#ifndef VAMPIRIA_PACKET_H

#define VAMPIRIA_PACKET_H 1

namespace vampiria { namespace packet {

//!Packet type raw packet identifier
const vmp::str P_RAW="raw";

//!Packet type ethernet identifier
const vmp::str P_ETHERNET="ethernet";

//!Packet type arp identifier
const vmp::str P_ARP="arp";
	
//!Packet type ipv4 identifier	
const vmp::str P_IPV4="ipv4";
	
//!Packet type udp identifier
const vmp::str P_UDP="udp";
	
//!Packet type tcp identifier
const vmp::str P_TCP="tcp";
	
class PacketHelper;

//!Abstract class for implementing packages
class Packet
{
    private:
        friend PacketHelper;
    protected:
        vmp::vector<packet::DataTypes *> data_;/*!<List of implemented data in byte order(see packet::DataTypes)*/
        vmp::Table<vmp::str,vmp_index> search_;/*!<Internal usage*/
        vmp::str type_;/*!< Header type*/
        Packet *next_;/*!<Next packet header*/
        Packet *prev_;/*!<Prev packet header*/
        vmp_bool subtype_;/*!<Used for variable data headers; if true the package is the subtype of the previous package. (A package can only have one subtype)*/
        vmp::str master_;/*!<If it is a subtype it indicates the name of the main package*/

        /*!Called in derived constructors to set the package as a subtype of the master*/
        /*!
            @param master master type
            @sa subtype_
            @sa master_ 
        */
        void set_subtype(vmp::str master);
        
        //!Insert data into the package
        /*!
            @param bytes data in byte order
            @return except error
        */  
        void insert_data(packet::DataTypes *bytes);
    public:
        //! A constructor
        /*!
            @param type header type
            @param subpacket is subpacket?
            @sa subpacket_
        */
        Packet(vmp::str type);

        //! A destructor
        virtual ~Packet();

        //! Gets header type
        /*!
            @sa type_
            @return header type
        */
        vmp::str type();
        
        //! Check if the header is a subtype of master
        /*!
            @param master master type
            @return boolean response
            @sa master_
            @sa subtype_
        */
        vmp_bool subtype(vmp::str master);

        //!Read header data from a buffer
        /*!
            @param buf input buf
        */
        virtual void read(vmp::Buf *buf);

        //!Writes the header data to a buffer
        /*!
            @param buf output buf
        */
        virtual void write(vmp::Buf *buf);

        //!Writes header data used for packet checksum in transport layer
        /*!
            @param buf input buf
        */
        virtual void pseudo_header(vmp::Buf *buf);

        //!Returns the data in string format associated with the field
        /*!
            @param field input field
            @return assocaited data(except error)
        */
        virtual vmp::str get(vmp::str field);

        //!Set the data associated with the field(string format)
        /*!
            @param field input field
            @param data  input data
        */ 
        virtual void set(vmp::str field,vmp::str data);

        //!Returns a human-readable string associated with the header type
        /*!
            @return human-readable string
        */
        virtual vmp::str print();
        
        //!Return next type in string mode(default "")
        /*!
            @return next type
        */
        virtual vmp::str next_type();
      
        //!Returns next packet header
        /*!
            @return next packet header
        */
        Packet *next();

        //!Returns next prev header
        /*!
            @return prev packet header
        */
        Packet *prev();
        
        //!Operator to concatenate a next header to the packet
        /*!
            @param next next header
            @return pointer to this packet
        */
        Packet *operator/(Packet *next);

        //!Look for a type header starting from this header
        /*!
            @param type packet type
            @return Header packet associated with the type(except not found)
        */
        Packet *operator[](vmp::str type);

        //! Returns the data type of index i
        /*!
             @param i index data
             @sa data_
             @return data type associated or 0 in case of error
        */
        packet::DataTypes *data(vmp_index i);

        //! Writes the payload of the header in raw format (next packet data) to the buffer
        /*!
            @param buf output buffer
        */
        void payload(vmp::Buf *buf);
};

//!Free memory packet
/*!
     @param packet first packet header
*/
void packet_free(packet::Packet *packet);

//!Package building function
typedef packet::Packet *(*PACKBUILD)();

//!Package management utility
class PacketHelper
{
    private:
        PACKBUILD packdefault_;/*!<Default package building function*/
        vmp::Table<vmp::str,PACKBUILD> packlist_;/*!<Internal usage*/
    public:

        //!A constructor
        PacketHelper();

        //!A destructor
        ~PacketHelper();

        //!Insert a new packet header
        /*!
            @param type header type
            @param builder package building function
        */
        void insert_packet(vmp::str type,packet::PACKBUILD builder);

        //!Clone a header by entering a new name as a construction reference
        /*!
            @param type header type
            @param clone new reference name
            @return except error
        */
        void clone_type(vmp::str type,vmp::str clone);

        //!Reads an entire packet from the buffer starting from a basic type
        /*!
            @param basetype basic type
            @param buf read buffer
            @return new packet(except error)
        */
        packet::Packet *read(vmp::str basetype,vmp::Buf *buf);

        //!Writes an entire packet to a buffer
        /*!
            @param packet packet to write
            @param buf output buffer
            @return except error
        */
        void write(packet::Packet *packet,vmp::Buf *buf);

        //!Clone input packet header (not whole packet)
        /*!
            @param packet header packet to clone
            @param header cloned(except error)
        */
        packet::Packet *clone_hdr(packet::Packet *packet);
	
	//!Returns a human-readable string associated with the packet.(All headers starting from the current package)
        /*!
             @param packet input packet
             @return human-readable string
        */
	vmp::str print_all(packet::Packet *packet);
};

//! Create default arp fixed header
/*!
    header field:<BR>
    "htype" harware type(see packet::DataArpHtype)<BR>
    "ptype" protocol type (see packet::DataEtherType)<BR>
    "hlen" hardware len<BR>
    "plen" protocol len<BR>
    "oper" operation (request,reply etc...)
*/
packet::Packet *Arp_D();

//! Returns the name associated with an arp subpackage  based on harwdare type and protocol type.
//! The package format is package::P_ARP_XXXX_YYYY where XXXX is a hexadecimal code htype and YYYY is a hexadecimal code ptype
/*!
    @param htype hardware type (see packet::DataArpHtype)
    @param ptype protocl type (see packet::DataEtherType)
    @return name subpackage
*/ 
vmp::str arpsub_str_generic(vmp_uint16 htype,vmp_uint16 ptype);

//! Create subpacket header arp ethernet ipv4
/*!
    header field:<BR>
    "sha" source mac address<BR>
    "spa" source ipv4 address<BR>
    "dha" destination mac address<BR>
    "dpa" destination protocl address 
*/
packet::Packet *Arp_Ethernet_Ipv4_S();

//! Create default arp ethernet ipv4 header
/*!
    header field: fixed arp header + arp header ethernet
*/ 
packet::Packet *Arp_Ethernet_Ipv4_D();

//! Create an arp ethernet ipv4 header
packet::Packet *Arp_Ethernet_Ipv4(vmp::str oper,vmp::str sha,vmp::str spa,vmp::str tha,vmp::str tpa);

//! Create default ethernet header
/*!
    header field:<BR>
    "dstmac" destination mac address<BR>
    "srcmac" source mac address<BR>
    "ethertype" next header type (see packet::DataEtherType())
*/
packet::Packet *Ethernet_D();

//! Create a ethernet header(see packet::Ethernet_D())
packet::Packet *Ethernet(vmp::str dstmac,vmp::str srcmac,vmp::str ethertype);

//! Create deafult ipv4 header(see ipv4 header doc)
/*!
    header field:<BR>
    "version" value is 4<BR>
    "ihl" header length in words (in header base ihl=5)<BR>
    "precedence" ["routine","priority","immediate","flash","flash_override","critic/ecp","internetwork_control","network_control"]<BR>
    "delay" ["normal","low"] <BR>
    "throughput" ["normal","high"] <BR>
    "relibility" ["normal","high"] <BR>
    "tos6" ["unset","set"]<BR>
    "tos7" ["unset","set"]<BR>
    "total_length" lenght header + length data<BR>
    "identification" header idendentification<BR>
    "flags0" ["unset","set"]<BR>
    "flags_df" ["no","yes"]<BR>
    "flags_mf" ["last_fragment","more_fragments"]<BR>
    "frag_offset" fragment offset<BR>
    "ttl" time to live<BR>
    "protocol" next header type(see packet::DataIpProto)<BR>
    "header_checksum" checksum value<BR>
    "source" source ipv4 address<BR>
    "destination" destination ipv4 address<BR>
    "options" ipv4 options data in raw mode
*/
packet::Packet *Ipv4_D();

//! Create a ethernet header(see packet::Ethernet_D())
packet::Packet *Ipv4(vmp::str total_length,vmp::str identification,vmp::str ttl,vmp::str protocol,vmp::str source,vmp::str destination,vmp::str options);

//! Check integrity ipv4 header.
/*!
    @param packet input packet
    @return true if the integrity of the ipv4 header is verified, otherwise false.
            If the packet does not contain an ipv4 header, throw an exception 
*/
vmp_bool Ipv4_integrity_check(packet::Packet *packet);

//! Adjust ipv4 checksum and length
//! If there are problems with the pseudo headers  or If the packet does not contain an ipv4 header, throw an exception
/*!
     @param packet input packet
*/ 
void Ipv4_adjust_check(packet::Packet *packet);

//! Create default raw header
/*!
    header field:<BR>
    "data" data string
*/
packet::Packet *Raw_D();

//! Create a raw header(see packet::Raw_D())
packet::Packet *Raw(vmp::str data);

//! Create default tcp header(see tcp doc)
/*!
    header field<BR>
    "srcport" source port<BR>
    "dstport" destination port<BR>
     "sequence_n" sequence number<BR>
     "ack_n" acknowledgment number<BR>
     "offset" header length in words (in header base offset=5)<BR>
     "reserved" data reserved 4 bits rapresented in hexa mode<BR>
     "cwr" Congestion Window Reduced ["unset","set"] <BR>
     "ece"  Explicit Congestion Notification ["unset","set"] <BR>
     "urg" urgent pointer  is valid ["unset","set"] <BR>
     "ack" Acknowledgment number is valid ["unset","set"] <BR>
     "psh" no buffering data ["unset","set"]<BR>
     "rst" reset data ["unset","set"] <BR>
     "syn" initial sequence number  ["unset","set"] <BR>
     "fin"  connection close ["unset","set"] <BR>
     "window" windows size received<BR>
     "checksum" udp checksum<BR>
     "urgent_p" urgent pointer<BR>
     "options"  tcp advanced options data in raw mode
*/
packet::Packet *Tcp_D();

//! Create a Tcp header
packet::Packet *Tcp(vmp::str srcport,vmp::str dstport,vmp::str sequence_n,vmp::str ack_n,vmp::str cwr,
                    vmp::str ece,vmp::str urg,vmp::str ack,vmp::str psh,vmp::str rst,vmp::str syn,vmp::str fin,
                    vmp::str window,vmp::str urgent_p,vmp::str options);

//!Check if the packet with tcp header is well structured by the checksum field that identifies the integrity of the packet is correct.
/*!
    @param packet input packet
    @return true is verified,otherwise false.
	    If the packet does not contain an tcp header, throw an exception 
*/
vmp_bool Tcp_integrity_check(packet::Packet *packet);

//! Adjust tcp checksum
//! If there are problems with the pseudo headers  or If the packet does not contain an tcp header, throw an exception
/*!
     @param packet input packet
*/ 
void Tcp_adjust_check(packet::Packet *packet);

//! Create deafult udp header(see udp doc)
/*!
     header field<BR>
     "srcport" source port<BR>
     "dstport" destination port<BR>
     "length" header + data length<BR>
     "checksum" udp checksum<BR>
*/
packet::Packet *Udp_D();

//! Create a udp header
packet::Packet *Udp(vmp::str srcport,vmp::str dstport);

//!Check if the packet with udp header is well structured by checking if the length field has the right value and the checksum field that identifies the integrity of the packet is correct.
/*!
    @param packet input packet
    @return true is verified,otherwise false.
            If the packet does not contain an udp header, throw an exception 
*/
vmp_bool Udp_integrity_check(packet::Packet *packet);

//! Fix the udp header by setting the length field and calculating the checksum that identifies the integrity of the package is correct.
//! If there are problems with the pseudo headers  or If the packet does not contain an udp header, throw an exception 
/*!
     @param packet input packet
*/
void Udp_adjust_check(packet::Packet *packet);

}}

#endif

#include "packet/arp.h"
#include "packet/ethernet.h"
#include "packet/ip.h"
#include "packet/raw.h"
#include "packet/tl.h"


