/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 * Date last update: 10/06/2020
 */

#ifndef VAMPIRIA_NET_UDP_H

#define VAMPIRIA_NET_UDP_H 1

namespace vampiria { namespace net {

const vmp_size UDP_MSIZE_DEFAULT = 508;/*!<Udp max size packet default value*/

//!Udp connection class
class Udp
{
    private:
       Socket socket_;/*!<Socket references*/
       vmp_bool blocking_;/*!< is blocking?*/
       vmp_size maxpsize_;/*!< max packet size*/
    public:
       //! A Constructor
       Udp();

       //! A Destructor
       ~Udp();

       //! Reset Udp connection
       void reset();

       //! Returns socket references
       /*!
           @sa socket_
           @return socket refernces
       */
       net::Socket socket();
       
       //! Setting socket in blocking mode.Must be done before the connection is made
       /*!
           @sa blocking_
       */
       void block();

       //! Setting socket in non blocking mode.Must be done before the connection is made
       /*!
           @sa blocking_
       */
       void noblock();

       //! Setting max packet size.Must be done before the connection is made
       /*!
           @param size new max packet size
           @sa maxpsize_
       */
       void set_maxpsize(vmp_size size);

       //! Returns max packet size
       /*!
           @return max packet size
           @sa maxpsize_
       */
       vmp_size maxpsize();

       //!Gets socket address local
       /*!
           @param result address result
           @return void (except error)
           @sa socket_
       */
       void get_local_address(net::Address *result);
       
       //! Create Udp listen server
       /*!
           @param local address to bind udp connection
           @return void (except error)
       */  
       void server(net::Address *local);

       //! Create udp client ipv4
       /*! 
           @return void (except error)
       */
       void client4();
       
       //! Create udp client ipv6
       /*! 
           @return void (except error)
       */
       void client6();

       //! Send datagram packet
       /*!
           @param buf data to send
           @param address remote address to send data
           @return size packet to send(except error)
       */
       vmp_int sendTo(vmp::Buf *buf,net::Address *peer);
       
       //! Recv datagram packet
       /*!
           @param buf received data
           @param address remote address from recv data
           @return size packet to receive,-1 no data received from non blocking socket(except error)
       */
       vmp_int recvFrom(vmp::Buf *buf,net::Address *peer);

       //! Close udp connection
       void close();
};

//!Udp class event(see framework event)
class EventUdp:public event::Event
{
     private:
       //!Init udp event
       /*!
           @param maxpsize max packet size
           @param recv see net::EVTCBRECVFROM 

       */
       void evt_udp_init(vmp_size maxpsize,net::EVTCBRECVFROM recv);
       
       //! Reset Udp event
       void evt_udp_reset();
     public:
       //! A Constructor
       EventUdp();

       //! A Destructor
       ~EventUdp();

       net::Udp udp_;/*!<Udp connection*/
       net::EVTCBRECVFROM recv_;/*!<Recv from callback assocaited*/
       net::Address local_;/*! Udp local address*/
       vmp_bool getaddr_;/*! For internal use*/
       
       //!Create a new event udp server
       /*!
           @param  ui user interface (see Class UI)
           @param local address to listen server connection
           @param maxpsize max packet size udp connection
           @param recv function performed when data is received
           @param  close function performed at the end of the event
           @return the new cell associated with the event
       */
       event::Cell *evt_udp_server(event::UI *ui,net::Address *local,vmp_size maxpsize,net::EVTCBRECVFROM recv,event::EVTCB close);
       
       //!Create a new event udp client ipv4
       /*!
           @param  ui user interface (see Class UI)
           @param maxpsize max packet size udp connection
           @param recv function performed when data is received
           @param  close function performed at the end of the event
           @return the new cell associated with the event
       */
       event::Cell *evt_udp_client4(event::UI *ui,vmp_size maxpsize,net::EVTCBRECVFROM recv,event::EVTCB close);
       
       //!Create a new event udp client ipv6
       /*!
           @param  ui user interface (see Class UI)
           @param maxpsize max packet size udp connection
           @param recv function performed when data is received
           @param  close function performed at the end of the event
           @return the new cell associated with the event
       */
       event::Cell *evt_udp_client6(event::UI *ui,vmp_size maxpsize,net::EVTCBRECVFROM recv,event::EVTCB close);
       
       //! Send datagram packet
       /*!
           @param buf data to send
           @param address remote address to send data
           @return void(except error)
       */
       void evt_udp_sendTo(vmp::Buf *buf,net::Address *peer);
       
       //! Returns max packet size
       /*!
           @return max packet size
       */
       vmp_size evt_udp_maxpsize();
       
       //!Gets socket address local
       /*!
          @return address local 
       */
       net::Address *evt_udp_localaddr();

       //! Close event
       void evt_udp_close();
       
       //! Free event
       void evt_udp_free();
};

//!Udp User interface
class UdpUI:public event::UI
{
    private:
       vmp::utils::Storage<net::EventUdp> uref_;/*!<Storage event*/
       vmp_size maxpsize_;/*!<Max packet size udp connection*/
       net::EVTCBRECVFROM revent_;/*!<Recv data callback*/
       event::EVTCB cevent_;/*!<Close event callback*/
    public:
       
       //!A Constructor
       /*!
            @param manager input manager
       */
       UdpUI(event::Manager *manager);
       
       //! A Destructor
       ~UdpUI();

       //!Virtual class implemented
       vmp::str identity(event::Cell *cell);

       //!Virtual class implemented
       void close_event(event::Cell *cell);
       
       //!Virtual class implemented
       void free_ref(event::Cell *cell);

       //!Set Udp event
       /*!
           @param revent recv data callback
           @param cevent close event callback
           @param maxpsize max packet size 
       */
       void set_event(net::EVTCBRECVFROM revent,event::EVTCB cevent,vmp_size maxpsize=net::UDP_MSIZE_DEFAULT);

       //!Create udp server event
       /*!
            @param local listen address udp connection
            @return cell associated with event(except error)
       */
       event::Cell *new_server(net::Address *local);

       //!Create udp client ipv4 event
       /*!
            @return cell associated with event(except error)
       */
       event::Cell *new_client4();
       
       //!Create udp client ipv6 event
       /*!
            @return cell associated with event(except error)
       */
       event::Cell *new_client6();

       //! Send datagram packet
       /*!
           @param cell cell associated
           @param buf data to send
           @param peer remote address to send data
           @return void (except error)
       */
       void sendTo(event::Cell *cell,vmp::Buf *buf,net::Address *peer);

       //! Returns max packet size
       /*!
           @param cell cell associated
           @return max packet size
       */ 
       vmp_size maxpsize(event::Cell *cell);
       
       //!Gets socket address local
       /*!
           @param cell cell associated
           @return address local 
       */
       net::Address *localaddr(event::Cell *cell);
};

}}

#endif

