/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 * Date last update: 10/06/2020
 */

#ifndef VAMPIRIA_NET_TCP_H

#define VAMPIRIA_NET_TCP_H 1

namespace vampiria { namespace net {

//!Virtual class generic tcp connection
class Tcp
{
    protected:
        Socket socket_;/*!<Socket references*/
        vmp_bool blocking_;/*!< is blocking?*/
        vmp_bool ready_;/*!Internal use*/
        vmp::str type_;/*! Tcp type connection(setting in derivated class)*/
    public:
        //! A Constructor
        Tcp();

        //!A Destructor
        virtual ~Tcp();

        //! Reset connection
        void reset();

        //! Returns socket references
        /*!
           @sa socket_
           @return socket references
        */
        net::Socket socket();
        
        //! Setting socket in blocking mode.Must be done before the connection is made
        /*!
           @sa blocking_
        */
        void block();
        
        //! Setting socket in non blocking mode.Must be done before the connection is made
        /*!
           @sa blocking_
        */
        void noblock();
        
        //! Returns connection type
        /*!
            @sa type_
            @return connection type
        */
        vmp::str type();
 
        //!Gets socket address local
        /*!
            @param result address result
            @return void (except error)
            @sa socket_
        */
        void get_local_address(net::Address *result);
        
        //!Gets socket address peer
        /*!
            @param result address result
            @return void (except error)
            @sa socket_
        */
        virtual void get_peer_address(net::Address *result);
        
        //! Send stream packet
        /*!
            @param buf data to send
            @return size packet to send,0 socket close(except error)
        */
        vmp_int sendData(vmp::Buf *buf);
        
        //! Recv stream packet
        /*!
           @param buf received data
           @return size packet to receive,0 close connection,-1 no data received from non blocking socket(except error)
        */
        vmp_int recvData(vmp::Buf *buf);
  
        //! Close tcp connection
        void close();
};

//! Tcp client connection
class TcpClient:public net::Tcp
{
    public:
        //! A Constructor
        TcpClient();
        
        //! A Destructor
        ~TcpClient();

        //! Connect tcp client
        /*!
            @param server remote server to connect
            @return void (except error)
        */
        void connect(net::Address *server);
        
        //! Check if connection is active(for non blocking socket)
        /*!
            @return true if connection is active,otherwise false
        */
        vmp_bool connect_check();
};

//! Tcp server connection established(see net::TcpListen)
class TcpServer:public net::Tcp
{
    public:
        //! A Constructor
        /*!
            @param socket socket coonection associated
        */
        TcpServer(vmp_int socket);

        //! A Destructor
        ~TcpServer();
};

//! Free tcp connection server created by net::TcpListen
/*!
     @param server tcp connection server
*/
void tcpserver_free_connection(net::TcpServer *server);

//!Tcp listen server
class TcpListen:public net::Tcp
{
    public:
        //!A Constructor
        TcpListen();

        //! A Destructor
        ~TcpListen();

        //! virtual function implemented(return except)
        void get_peer_address(net::Address *result);

        //! Listen tcp server
        /*!
            @param local listen address
            @param backlog the maximum length to which the  queue  of pending  connections
        */  
        void server(net::Address *local,vmp_uint backlog);
        
        //! Accept tcp connection
        /*!
            @return net::TcpServer if accepted connection(free memory with net::tcpserver_free_connection),0 otherwise(no blocking mode)  
        */
        net::TcpServer *accept();
};

//!Tcp class event(see framework event)
class EventTcp:public event::Event
{
    private:
       //! Reset Tcp event
       void evt_tcp_reset();
 
    public:
       //! A Constructor
       EventTcp();

       //! A Destructor
       ~EventTcp();

       event::EVTCB connect_;/*!<Tcp client connect callback.Called when a connection is complete.*/
       net::EVTCBACCEPT acptevent_;/*!<Tcp server accept callback.Called when a connection is accepted*/
       event::EVTCB cchild_;/*!< Close child event callback. Used for close accepted connection server*/
       net::EVTCBRECV recv_;/*!< Recv data stream.Called when socket receive data*/
       net::Address local_;/*!< Local address socket*/
       net::Address remote_;/*!< Remote address socket*/

       net::Tcp *tcp_;/*!<Tcp connection associated*/
       
       //!Create a new event tcp client
       /*!
           @param  ui user interface (see Class UI)
           @param  server remote server address
           @param connect function performed when coonect is complete
           @param recv function performed when data is received
           @param  close function performed at the end of the event
           @return the new cell associated with the event
       */
       event::Cell *evt_tcp_client(event::UI *ui,net::Address *server,event::EVTCB connect,net::EVTCBRECV recv,event::EVTCB close);
       
       //!Create a new event tcp server
       /*!
           @param  ui user interface (see Class UI)
           @param local address to listen server connection
           @param backlog the maximum length to which the  queue  of pending  connections
           @param acptevent function performed when connection is accepted
           @param svlcevent function performed when close listen socket
           @param recv function performed when rececive data from socket
           @param svcevent function performed when close server connection
           @return the new cell associated with the event
       */
       event::Cell *evt_tcp_server(event::UI *ui,net::Address *local,vmp_uint backlog,net::EVTCBACCEPT acptevent,event::EVTCB svlcevent,net::EVTCBRECV recv,event::EVTCB svcevent);

       //! Send stream packet
       /*!
           @param buf data to send
           @return void(except error)
       */
       void evt_tcp_send(vmp::Buf *buf);

       //!Gets socket address local
       /*!
           @return address local 
       */
       net::Address *evt_tcp_localaddr();

       //!Gets socket address peer
       /*!
           @return address peer
       */
       net::Address *evt_tcp_peeraddr();
       
       //! Gets tcp connection type
       /*!
           @return connection type ("client","listen","server")
       */
       vmp::str evt_tcp_type();
       
       //! Close event
       void evt_tcp_close();
       
       //! Free event
       void evt_tcp_free();
};

//!Tcp User interface
class TcpUI:public event::UI
{
    private:
       vmp::utils::Storage<net::EventTcp> tref_;/*!<Storage event*/

       event::EVTCB      connect_;/*!<Tcp client connect callback.Called when a connection is complete.*/
       net::EVTCBRECV    crecv_;/*!< Recv data stream.Called when socket receive data client type*/
       event::EVTCB      cclose_;/*! Close client callback*/

       net::EVTCBACCEPT  acptevent_;/*!<Tcp server accept callback.Called when a connection is accepted*/
       event::EVTCB      svlcevent_;/*!<Function performed when close listen event*/       

       net::EVTCBRECV    svrevent_;/*!< Recv data stream.Called when socket receive data server type*/
       event::EVTCB      svcevent_;/*! Close server callback*/

    public:
       
       //!A Constructor
       /*!
            @param manager input manager
       */
       TcpUI(event::Manager *manager);
       
       //! A Destructor
       ~TcpUI();

       //!Virtual class implemented
       vmp::str identity(event::Cell *cell);
       
       //!Virtual class implemented
       void close_event(event::Cell *cell);
       
       //!Virtual class implemented
       void free_ref(event::Cell *cell);
       
       //!Virtual class implemented(for server connection)
       event::Event *child_event_new(event::Cell *cell);

       //!Set client event
       /*!
           @param connect connect callback
           @param crecv  receive data stream
           @param cclose close client callback
       */
       void set_event_client(event::EVTCB connect,net::EVTCBRECV crecv,event::EVTCB cclose);
       
       //! Set server event
       /*!
           @param acptevent server accept callback
           @param svlcevent close listen event
           @param revent receive data stream
           @param svcevent close connection server
       */
       void set_event_server(net::EVTCBACCEPT acptevent,event::EVTCB svlcevent,net::EVTCBRECV revent,event::EVTCB svcevent);

       //! Add tcp client event. Callback events assigned with set_event_client
       /*!
           @param raddress remote server address
           @return event cell (except error)
       */
       event::Cell *new_client(net::Address *raddress);

       //! Add tcp server listen event. Callback events assigned with set_event_server
       /*!
           @param local address listener
           @param backlog max connection server
           @return event cell (except error)
       */
       event::Cell *new_server(net::Address *local,vmp_size backlog);

       //! Send packet
       /*!
           @param cell event cell
           @param buf packet buffer
	   @return except error
       */
       void send(event::Cell *cell,vmp::Buf *buf);

       //!Gets socket address local
       /*!
           @param cell cell associated
           @return address local 
       */
       net::Address *localaddr(event::Cell *cell);

       //!Gets socket address peer
       /*!
           @param cell cell associated
           @return address local 
       */
       net::Address *peeraddr(event::Cell *cell);
};

}}

#endif

