/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 06/04/2022
*/

#ifndef VAMPIRIA_JSON_JDATA_H

#define VAMPIRIA_JSON_JDATA_H 1

namespace vampiria { namespace json {

class JList;
class JTable;

//! Internal usage
void primitive_error(vmp::str label);

//! Internal usage
void set_primitive_error(vmp::str label);

//! JData class
/*!
    Used for creating and reading data in json.
    The root object is a json(object type) object that contains the fields jtype(json data type), 
    jkey(json data key) and jbody (a json object that contains the primitives that contain the data).<BR>
    example:{ jtype:"dataname",jkey:"ciao3",jbody:{"value":3,"str":"ciao,"extra":"extra data"}}
*/
class JData
{
    private:
        json::JsonObj *root_;/*!<json data object references*/

        //! Internal usage
        void get_jbody(vmp::str fname,json::JsonObj *ret);

        //! Internal usage        
        void check_init(vmp::str fname);
    public:

        //! A constructor
        JData();

        //! A destructor
        ~JData();
        
        //! Reset json data object
        /*!
            @ref root_
        */
        void reset();
        
        //! Create the basic structure for a data json, used before entering the data in the construction of a new data(jbody is empty)
        /*!
            @param root json object where the data is created
            @param jtype jdata value
            @param jkey  jdata key value
            @return void or except in case of failure
        */
        void new_data(json::JsonObj *root,vmp::str jtype,vmp::str jkey);

        //! Set the structure with an existing json data object
        /*!
            @param root json object that contains the data
            @return void or except in case of failure
        */
        void set(json::JsonObj *root);

        //! Return the root json data object
        /*!
            @ref root_
            @return json object root or except in case of failure
        */
        json::JsonObj *root();

        //! Return json data type
        /*!
            @ref root_
            @return json data type or except in case of failure
        */
        vmp::str jtype();
        
        //! Return json data key
        /*!
            @ref root_
            @return json data type or except in case of failure
        */
        vmp::str jkey();

        //! Insert a text primitive in jbody
        /*!
            @param label primitive label
            @param value primitive value
            @return void or except in case of failure
        */
        void set_text(vmp::str label,vmp::str value);

        //! Gets a text primitive in jbody
        /*!
            @param label primitive label
            @return primitive value or except in case of failure
        */
        vmp::str get_text(vmp::str label);

        //! Insert an integer primitive in jbody
        /*!
            @param label primitive label
            @param value primitive value
            @return void or except in case of failure
        */
        void set_integer(vmp::str label,vmp_int value);

        //! Gets an integer primitive in jbody
        /*!
            @param label primitive label
            @return primitive value or except in case of failure
        */
        vmp_int get_integer(vmp::str label);

        //! Insert an integer primitive in jbody(between min and max)
        /*!
            @param label primitive label
            @param min minimum value of the data
            @param max maximum value of the data
            @param value primitive value
            @return void or except in case of failure
        */
        void set_integer_range(vmp::str label,vmp_int min,vmp_int max,vmp_int value);

        //! Gets an integer primitive in jbody(between min and max)
        /*!
            @param label primitive label
            @param min minimum value of the data
            @param max maximum value of the data
            @return primitive value or except in case of failure
        */
        vmp_int get_integer_range(vmp::str label,vmp_int min,vmp_int max);

        //! Insert a real primitive in jbody
        /*!
            @param label primitive label
            @param value primitive value
            @return void or except in case of failure
        */
        void set_real(vmp::str label,vmp_real value);

        //! Gets a real primitive in jbody
        /*!
            @param label primitive label
            @return primitive value or except in case of failure
        */
        vmp_real get_real(vmp::str label);

        //! Insert a real primitive in jbody(between min and max)
        /*!
            @param label primitive label
            @param min minimum value of the data
            @param max maximum value of the data
            @param value primitive value
            @return void or except in case of failure
        */        
        void set_real_range(vmp::str label,vmp_real min,vmp_real max,vmp_real value);

        //! Gets a real primitive in jbody(between min and max)
        /*!
            @param label primitive label
            @param min minimum value of the data
            @param max maximum value of the data
            @return primitive value or except in case of failure
        */
        vmp_real get_real_range(vmp::str label,vmp_real min,vmp_real max);
        
        //! Create a empty list primitive in the jbody.
        /*!
            @param label primitive label
            @param list output list object(to manage the list)
            @param jtype type of json data entered in the list(jtype="" generic list not typed)
            @return void or except in case of failure
        */
        void new_list(vmp::str label,json::JList *list,vmp::str jtype="");

        //! Insert a list primitive in the jbody.
        /*!
            @param label primitive label
            @param list input list to insert
            @param jtype type of json data entered in the list(jtype="" generic list not typed)
            @return void or except in case of failure
        */        
        void set_list(vmp::str label,json::JList *list,vmp::str jtype="");
        
        //! Gets a list primitive in the jbody.
        /*!
            @param label primitive label
            @param list output list object(to manage the list)
            @param jtype type of json data entered in the list(jtype="" generic list not typed)
            @return void or except in case of failure
        */  
        void get_list(vmp::str label,json::JList *list,vmp::str jtype="");

        //! Create a empty table primitive in the jbody.
        /*!
            @param label primitive label
            @param table output table object(to manage the table)
            @param jtype type of json data entered in the data object in table(jtype="" generic table not typed)
            @return void or except in case of failure
        */
        void new_table(vmp::str label,json::JTable *table,vmp::str jtype="");

        //! Create a table primitive in the jbody.
        /*!
            @param label primitive label
            @param table input table to insert
            @param jtype type of json data entered in the data object in table(jtype="" generic table not typed)
            @return void or except in case of failure
        */
        void set_table(vmp::str label,json::JTable *table,vmp::str jtype="");
        
        //! Gets a table primitive in the jbody.
        /*!
            @param label primitive label
            @param table output table object(to manage the table)
            @param jtype type of json data entered in the data object in table(jtype="" generic table not typed)
            @return void or except in case of failure
        */
        void get_table(vmp::str label,json::JTable *table,vmp::str jtype="");
        
        //! Insert an args primitive in the jbody(string vector).
        /*!
            @param label primitive label
            @param value primitive value
            @return void or except in case of failure
        */
        void set_args(vmp::str label,vmp::vector<vmp::str> args);

        //! Gets an args primitive in jbody(string vector).
        /*!
            @param label primitive label
            @return primitive value or except in case of failure
        */        
        vmp::vector<vmp::str> get_args(vmp::str label);
        
        //! Insert an vars primitive in the jbody(table<string,string>).
        /*!
            @param label primitive label
            @param vars pointer primitive value
            @return void or except in case of failure
        */
        void set_vars(vmp::str label,vmp::Table<vmp::str,vmp::str> *vars);

        //! Gets an vars primitive in the jbody(table<string,string>).
        /*!
            @param label primitive label
            @param vars pointer to output table
            @return void or except in case of failure
        */        
        void get_vars(vmp::str label,vmp::Table<vmp::str,vmp::str> *vars);
        
        //! Insert a json object primitive in the jbody(custom json object).
        /*!
            @param label primitive label
            @param obj json object to insert
            @return void or except in case of failure
        */
        void set_custom(vmp::str label,json::JsonObj *obj);

        //! Gets a json object primitive in the jbody(custom json object).
        /*!
            @param label primitive label
            @param obj json object output
            @return void or except in case of failure
        */        
        void get_custom(vmp::str label,json::JsonObj *obj);

        //! Insert a buffer primitive in the jbody.
        /*!
            @param label primitive label
            @param buf pointer input buffer
            @return void or except in case of failure
        */
        void set_buf(vmp::str label,vmp::Buf *buf);

        //! Gets a buffer primitive in the jbody.
        /*!
            @param label primitive label
            @param buf pointer output buffer
            @return void or except in case of failure
        */       
        void get_buf(vmp::str label,vmp::Buf *buf);

        //! Insert a netaddress primitive in the jbody.
        /*!
            @param label primitive label
            @param address pointer input netaddress
            @return void or except in case of failure
        */        
        void set_netaddress(vmp::str label,net::Address *address);

        //! Gets a netaddress primitive in the jbody.
        /*!
            @param label primitive label
            @param address pointer output netaddress
            @return void or except in case of failure
        */ 
        void get_netaddress(vmp::str label,net::Address *address);

        //! Insert a macaddress primitive in the jbody.
        /*!
            @param label primitive label
            @param mac pointer input macaddress
            @return void or except in case of failure
        */   
        void set_macaddress(vmp::str label,net::MacAddress *mac);

        //! Gets a macaddress primitive in the jbody.
        /*!
            @param label primitive label
            @param mac pointer output macaddress
            @return void or except in case of failure
        */ 
        void get_macaddress(vmp::str label,net::MacAddress *mac);

        //! Insert a packet primitive in the jbody.
        /*!
            @param label primitive label
            @param packet pointer input packet
            @return void or except in case of failure
        */ 
        void set_packet(vmp::str label,packet::Packet *packet);
        
        //! Gets a packet primitive in the jbody(use use packet::packet_free(packet) to free memory packet).
        /*!
            @param label primitive label
            @return packet or except in case of failure
        */ 
        packet::Packet *get_packet(vmp::str label);
};

}}

#endif

