/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 22/12/2018
 */

#include "main.h"

void main_signal_init(vmp_int sig)
{
    vmp::io_forced_unlock(); 	
    Vampiria *vmp=vmp::Instance::get_global<Vampiria>();
    vmp::vector<vmp::str> kprocess=vmp->process_.all_keys();
    for(vmp_index i=0;i<kprocess.size();i++)
        vmp->free_process(kprocess[i]);
    exit_ok();
}

void process_free(vmp::Pid pid,process::Box *box)
{
    Process *process=(Process *) box;
    Vampiria *vmp=vmp::Instance::get_global<Vampiria>();
    vmp->free_process(process->id());
}

void run_process(process::Box *box)
{
    Process *process=(Process *) box;
    try
    {
        process->module_->run();
    }
    catch(vmp::exception &x)
    {
        vmp::exit_failure("Process id '%s' fatal error %s",process->id().c_str(),x.what());
    }
    vmp::exit_ok();
}

void output_process(process::Box *box,vmp::str data)
{
    Process *process=(Process *) box;
    for(vmp_index i=0;i < process->stdout_.size();i++)
        if(process->stdout_[i] != 0)
             process->stdout_[i]->write(data);
}

void error_process(process::Box *box,vmp::str data)
{
    Process *process=(Process *) box;
    for(vmp_index i=0;i < process->stderr_.size();i++)
        if(process->stderr_[i] != 0)
             process->stderr_[i]->write(data);
}

void end_process(vmp::Pid pid,process::Box *box,vmp_int status)
{
    Vampiria *vmp=vmp::Instance::get_global<Vampiria>();
    Process *process=(Process *) box;
    
    //Free stdout write 
    for(vmp_index i=0;i<process->stdout_.size();i++)
        process->del_stdout(i);
     
    //Free stderr write 
    for(vmp_index i=0;i<process->stderr_.size();i++)
        process->del_stderr(i);
    
    for(vmp_index i=0;i<process->exports_.size();i++)
    {
        try
        {
            vmp::vector<vmp::str> data=vmp::input_file((vmp::str)(process->exports_[i].second));
            vmp::str idata=vmp::unicode::str_join(data,"\n");
            vmp::output_file((vmp::str) (process->exports_[i].first),false,"%s\n",idata.c_str());
            vmp::debug("Exports file %s->%s",process->exports_[i].second.c_str(),process->exports_[i].first.c_str());
        }
        catch(vmp::exception &x)
        {
            vmp::warning("%s",x.what());
        }
    }
    for(vmp_index i=0;i<process->kill_.size();i++)
    {
        vmp::str name=process->kill_[i].first;
        Process *pkill=vmp->get_process(name);
        if(pkill != 0)
        {
            pkill->kill(process->kill_[i].second);
            vmp::debug("Process (%s) send kill (%s) process (%s)",process->id().c_str(),process->kill_[i].second.c_str(),process->kill_[i].first.c_str());
        }
    }
    vmp::debug("Terminated Process id '%s' pid '%d' with status '%d'",process->id().c_str(),pid,status);      
}

Vampiria::Vampiria()
{
    root_="";
    pid_=vmp::getpid_wrap();
    mode_=UNDEF;
    args_="";
    manbox_=0;
    proot_=0;
    envp_=0;
}
       
Vampiria::~Vampiria()
{
}

vmp::str Vampiria::get_root()
{
    if(root_ == "")
    {
        vmp::str tmp;
        vmp::unicode::str_write(&tmp,"%u/",pid_);
        tmp=vmp::env::vampiria_cache(tmp);    
        vmp::fs::mkdir_wrap(tmp,0700);
        root_=tmp;
    }
    return root_;
}

Process *Vampiria::add_process(xml::Tag *ptag,vmp::str id,vmp_bool init,process::BOX_COM comtype)
{
    Process *process=get_process(id);
    if(process != 0)
        ptag->parser_error("Duplicate process id='%s' prev in line %u",id.c_str(),process->ptag_->get_line());
    if(init && (proot_ != 0))
        ptag->parser_error("Duplicate init process prev in line %u",proot_->ptag_->get_line());
    vmp::str rootdir=get_root();
    vmp::unicode::str_write(&rootdir,"%s%s/",get_root().c_str(),id.c_str());
    vmp::fs::mkdir_wrap(rootdir,0700);
    process=new Process(id);
    process->envp_=(*envp_);
    process->ptag_=ptag;
    process->rootdir_=rootdir;
    if(init)
       proot_=process;
    process->is_root_=init;
    process->comtype_=comtype;
    process->set(rootdir,run_process,output_process,error_process,end_process);
    process_.insert(id,process);
    vmp::debug("Set Process id '%s' Root Directory '%s'",id.c_str(),rootdir.c_str());
    return process;
}

Process *Vampiria::get_process(vmp::str id)
{
    Process *ret;
    if(process_.search(id,&ret))
       return ret;             
    return 0;
}

void Vampiria::free_process(vmp::str id)
{
   try
   { 
       Process *ret;
       process_.cancel(id,&ret);
       delete ret;
   }
   catch(vmp::exception &x)
   {
   }  
}

void Vampiria::destroy()
{
    process_.clear();
    mode_=UNDEF;
    args_="";
    proot_=0;
    if(root_ != "")
    {
        vmp::fs::rmdir_wrap(root_);
        vmp::debug("Remove root directory '%s'",root_.c_str());
        root_=""; 
    }
    if(envp_ != 0)
        delete envp_;
    vmp::debug("Exit Vampiria");
}

void exit_ok()
{
    Vampiria *vmp=vmp::Instance::get_global<Vampiria>();
    vmp->destroy();
    vmp::exit_ok();
}

void exit_failure(vmp::str error)
{
    Vampiria *vmp=vmp::Instance::get_global<Vampiria>();
    vmp->destroy();
    vmp::exit_failure("%s",error.c_str());
}

vmp_int main(vmp_int argc,vmp_char *argv[],vmp_char *envp[])
{
    vmp::str error;
    Vampiria *vmp=new Vampiria();
    
    vmp::Instance::set_global<Vampiria>(vmp);
    vmp::init("vampiria",false);
    
    //init signal setting pre box manager
    vmp::vector<vmp::str> sig;
    sig.push_back("sigterm");
    sig.push_back("sigint");
    vmp::signal_wrap(sig,main_signal_init);

    vmp->envp_=new vmp::args::Envp(envp);
    try
    {
        vmp::args::GetOpt opt;
        opt.set_opt('h',opt_helper,false);
        opt.set_opt('i',opt_info,false);
        opt.set_opt('v',opt_version,false);
        opt.set_opt('l',opt_modlist,false);
        opt.set_opt('m',opt_modtopic,true);
        opt.set_opt('p',opt_rootpath,false);
        opt.set_opt('c',opt_check,true);
        opt.set_opt('d',opt_debug,true);
        opt.set_opt('r',opt_run,true);
        opt.parse(argc,argv);
        if(vmp->mode_ == UNDEF)
            vmp::except_s("Invalid input arguments!!!");
    }
    catch(vmp::exception &x)
    {
        vmp::unicode::str_write(&error,"%s\n%s",x.what(),usage().c_str());
        exit_failure(error);
    }
    vmp::debug("Init Vampiria with pid=%d",vmp->pid_);
    vmp::debug("Parsing xml file '%s'",vmp->args_.c_str());		
    try
    {
        vmp->parser_.add_tag<VampiriaTag>("vampiria");
        vmp->parser_.add_tag<ProcessTag>("process");
        vmp->parser_.add_tag<PutenvTag>("putenv");
        vmp->parser_.add_tag<PreloadTag>("preload");
        vmp->parser_.add_tag<ImportTag>("import");
        vmp->parser_.add_tag<ModuleTag>("module");
        vmp->parser_.add_tag<StderrTag>("stderr");
        vmp->parser_.add_tag<StdoutTag>("stdout");
        vmp->parser_.add_tag<FileoutTag>("fileout");
        vmp->parser_.add_tag<ProcessoutTag>("processout");
        vmp->parser_.add_tag<ExportTag>("export");
        vmp->parser_.add_tag<KillTag>("kill");
        vmp->parser_.build(vmp->args_,vmp::env::software_path(name_,VAMPIRIA_DTDPATH));
        vmp->proot_=0;
        vmp->parser_.destroy();
        vmp->modules_.reset();
        if(vmp->mode_ == CHECK)
        {
            vmp::info("Vampiria check '%s' [OK]",vmp->args_.c_str());                
            exit_ok();        
        }
    }
    catch(vmp::exception &x)
    {
        vmp->parser_.destroy();
        vmp->modules_.reset();
        error=vmp::unicode::str_format_end(x.what());
        exit_failure(error);
    }
    try 
    {
        vmp->manbox_=process::BoxManager_new(process_free);
        vmp::vector<Process *> process=vmp->process_.all_data();
        for(vmp_index i=0;i<process.size();i++)
        {
            vmp::debug("Run process id '%s'",process[i]->id().c_str());
            vmp->manbox_->add_box(process[i],process[i]->is_root_,process[i]->comtype_);
            process[i]->verify_input();
        }
        vmp::debug("Enter process routine"); 
        vmp->manbox_->ctl_process();
        vmp::debug("Exit process routine"); 
        process::BoxManager_free(vmp->manbox_);
    }
    catch(vmp::exception &x) 
    {
        process::BoxManager_free(vmp->manbox_);
        vmp::unicode::str_write(&error,"%s",x.what());
        exit_failure(error);
    }
    exit_ok();
    return 0;
}

