/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 18/12/2019
 */

#include "vmp.h"

namespace vampiria { namespace vmp {

#define BUF_SIZE_INIT    32
#define BUF_EOB_ERROR "End Of Buf Error"

vmp::str buf_eob_strerror()
{
    return BUF_EOB_ERROR;
}

Buf::Buf()
{
    maxsize_=BUF_SIZE_INIT;         
    mem_=(vmp_byte *) vmp::malloc_wrap(maxsize_);
    reset();        
}

Buf::~Buf()
{
    vmp::free_wrap((void **) &mem_);
}

void Buf::expand(vmp_size size)
{
    vmp_uchar *tmp;        
    if(size > maxsize_)
    {
        do
            maxsize_ *= 2;                
        while(size > maxsize_);
        tmp=(vmp_byte *) vmp::malloc_wrap(maxsize_);
        vmp::bzero_wrap((void *)tmp,maxsize_);                
        vmp::memcpy_wrap((void *)tmp,(void *)mem_,size_);
        vmp::free_wrap((void **)&mem_);
        mem_=tmp;
    }
    if(size > size_)
        size_=size;
}

void Buf::newsize(vmp_size size)
{
    if(size > size_)        
        expand(size);
    else if(size < size_)
    {        
        vmp::bzero_wrap((void *)(&(mem_[size])),(size_-size));
        size_=size;
    }
}

vmp_size Buf::size()
{
    return size_;
}

vmp_size Buf::size_reading()
{
    return (size_-index_);
}

void Buf::index(vmp_index point)
{
    if(point < size_)
        index_=point;
    else
        index_=size_;
}

vmp_index Buf::get_index()
{
    return index_;
}

vmp_bool Buf::eob()
{
    if(index_ == size_)
        return true;
    return false;
}
         
void Buf::reset()
{
    vmp::bzero_wrap((void *)mem_,maxsize_);        
    size_=0;
    index();
}
                
vmp_byte *Buf::pointer(vmp_index point)
{
    if(point < size_)        
        return &(mem_[point]);
    return 0;
}

vmp_bool Buf::operator==(Buf &buffer)
{
    if(size_ == buffer.size())    
    {
        for(vmp_index i=0;i<size_;i++)
            if(mem_[i] != (*(buffer.pointer(i))))
                return false;
        return true;        
    }
    return false;
}

vmp_bool Buf::operator!=(Buf &buffer)
{
    if((*this) == buffer)
        return false;
    return true;
}

vmp_byte &Buf::operator[](vmp_index index)
{
    if(index >= size_)
        vmp::except_s("Overflow Buffer");
    return mem_[index];
}

Buf& Buf::operator=(Buf buffer)
{
    newsize(buffer.size());
    memcpy(mem_,buffer.pointer(),buffer.size());
    index(buffer.index_);    
    return (*this);
}

void Buf::write_byte(vmp_byte value)
{
    vmp_index nindex=index_+1;
    expand(nindex);
    mem_[index_]=value;
    index(nindex);
}

vmp_byte Buf::read_byte()
{
    vmp_byte ret=0x00;
    if(eob())
        vmp::except("%s read byte",BUF_EOB_ERROR);
    ret=mem_[index_];
    index(index_+1);
    return ret;
}
 
void Buf::write_array(vmp_byte *value,vmp_size size)
{
    vmp_index nindex=index_+size;        
    expand(nindex);        
    for(vmp_index i=0;i<size;i++)
        mem_[i+index_]=value[i];
    index(nindex);    
}

void Buf::read_array(vmp_byte *value,vmp_size size)
{
    for(vmp_index i=0;i<size;i++)
    {
        try
        {
            value[i]=read_byte();
        }
        catch(vmp::exception &x)
        {
            vmp::except("%s read array size %u",BUF_EOB_ERROR,size);
        }
    }
}

void Buf::write_vector(vmp::vector<vmp_byte> value)
{
    vmp_index nindex=index_+value.size();        
    expand(nindex);        
    for(vmp_index i=0;i<value.size();i++)
        mem_[i+index_]=value[i];
    index(nindex); 
}

vmp::vector<vmp_byte> Buf::read_vector(vmp_size size)
{
    vmp::vector<vmp_byte> ret;
    for(vmp_index i=0;i<size;i++)
    {
        try
        {
            ret.push_back(read_byte());
        }
        catch(vmp::exception &x)
        {
            vmp::except("%s read vector size %u",BUF_EOB_ERROR,size);
        }
    }
    return ret; 
}

void Buf::write_size(vmp_size value,vmp_uint dim)
{
    vmp_uint d;
    vmp_index nindex;
    if(dim == 0)
        return;
    else if(dim > 4)
        d=4;
    else
        d=dim;
    nindex=index_+d;
    expand(nindex);        
    for(vmp_index i=index_;i<nindex;i++)
        mem_[i]=(vmp_byte)((value >> ((nindex-i-1)*8)) & 0xFF);
    index(nindex);    
}
                
vmp_size Buf::read_size(vmp_uint dim)
{
    vmp_size ret=0;
    vmp_uint d;
    vmp_index nindex;
    if(dim == 0)
        return ret;
    else if(dim > 4)
        d=4;
    else
        d=dim;
    nindex=index_+d;
    for(vmp_index i=index_;i<nindex;i++)
    {
        if(i<size_)
           ret+=(vmp_size)(mem_[i] << ((nindex-i-1)*8));       
        else
           vmp::except("%s read size",BUF_EOB_ERROR);;
    }
    index(nindex);    
    return ret;       
} 

void Buf::write_str(vmp::str value)
{
    vmp_size nindex=(vmp_size) (value.size() + index_);
    expand(nindex);        
    for(vmp_index i=0;i<value.size();i++)
        mem_[i+index_]=value[i];                
    index(nindex);       
}
                
vmp::str Buf::read_str(vmp_size size)
{
    vmp::str ret="";     
    vmp_index i,nindex;
    for(i=0,nindex=index_;i<size && nindex<size_;i++)
        ret+=mem_[nindex++]; 
    if(i != size)
        vmp::except("%s read string size %d",BUF_EOB_ERROR,size);
    index(nindex);     
    return ret;
} 

void Buf::write_xstr(vmp::str value)
{
    vmp::vector<vmp_byte> bytes;
    try
    {
        bytes=vmp::unicode::xstr_tobytes(value);
    }
    catch(vmp::exception &x)
    {
        vmp::except("write xstr %s",x.what());
    }
    write_vector(bytes);
}

vmp::str Buf::read_xstr(vmp_size size)
{
    vmp::vector<vmp_byte> bytes;
    try
    {
        bytes=read_vector(size);
    }
    catch(vmp::exception &x)
    {
        vmp::except("%s read xstr size %u",BUF_EOB_ERROR,size);
    }
    return vmp::unicode::bytes_toxstr(bytes);
}

vmp::str Buf::read_xstr_hm(vmp_size size,vmp::str delimiter)
{
    vmp::vector<vmp_byte> bytes;
    try
    {
        bytes=read_vector(size);
    }
    catch(vmp::exception &x)
    {
        vmp::except("%s read xstr size %u",BUF_EOB_ERROR,size);
    }
    return vmp::unicode::bytes_toxstr_hm(bytes,delimiter);
}

void Buf::cat(Buf *value)
{
    index(size_);   
    write_buf(value);
}

void Buf::cathead(Buf *value)
{
    index();
    vmp_size s=size_;
    vmp_byte tmp[s];
    read_array(tmp,s);
    reset();
    write_buf(value);
    write_array(tmp,s);    
}

void Buf::write_buf(Buf *value)
{
    vmp_size s=value->size();        
    vmp_index nindex=s+index_;
    expand(nindex);
    vmp::memcpy_wrap(pointer(index_),value->pointer(),s);
    index(nindex);  
}

void Buf::read_buf(Buf *value,vmp_size size)
{
    vmp_index nindex=index_+size;
    if(nindex > size_)
        vmp::except("%s read buf size %d",BUF_EOB_ERROR,size);
    value->newsize(size);        
    vmp::memcpy_wrap(value->pointer(),pointer(index_),size);        
    index(nindex);
}

void Buf::cut(vmp_index init,vmp_size len)
{
    vmp_index p1,p2;
    for(p2=init,p1=0;(p2 < (init+len) && p2 < size_);p2++,p1++)
        mem_[p1]=mem_[p2];        
    newsize(p1);
    index();        
}

}
}

