/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 *Date last update: 10/06/2020
 */

#include "net.h"

namespace vampiria { namespace net {

Udp::Udp()
{
    socket_=-1;
    reset();
}
       
Udp::~Udp()
{
    reset();
}

void Udp::reset()
{
    close();
    maxpsize_=net::UDP_MSIZE_DEFAULT;
    blocking_=true;
}

Socket Udp::socket()
{
    return socket_;
}

void Udp::block()
{
    blocking_=true;
}
       
void Udp::noblock()
{
    blocking_=false;
}

void Udp::set_maxpsize(vmp_size size)
{
    if(size > 0)
        maxpsize_=size;
    else
        maxpsize_=net::UDP_MSIZE_DEFAULT;
}
       
vmp_size Udp::maxpsize()
{
    return maxpsize_;
}

void Udp::get_local_address(net::Address *result)
{
    net::socket_addrlocal(socket_,result);
}

void Udp::server(net::Address *local)
{
    if(socket_ != -1)
        close();
    if((local == 0) || (local->iptype() == PF_UNSPEC))
        vmp::except_s("net::Udp::server invalid local address"); 
    try
    {
        socket_=socket_datagram(local->iptype(),blocking_);
        socket_bind(socket_,local);
    }
    catch(const std::exception& x)
    {
        close();
        vmp::except_s(x.what());
    }
}

void Udp::client4()
{
    if(socket_ != -1)
        close();
    try
    {
        socket_=socket_datagram(AF_INET,blocking_);
    }
    catch(const std::exception& x)
    {
        close();
        vmp::except_s(x.what());
    }
}

void Udp::client6()
{
    if(socket_ != -1)
        close();
    try
    {
        socket_=socket_datagram(AF_INET6,blocking_);
    }
    catch(const std::exception& x)
    {
        close();
        vmp::except_s(x.what());
    }
}

vmp_int Udp::sendTo(vmp::Buf *buf,net::Address *peer)
{
    vmp_int ret;
    vmp::except_check_pointer((void *)buf,"net::Udp::sendTo() null pointer input buffer");
    vmp::except_check_pointer((void *)peer,"net::Udp::sendTo() invalid peer address");
    if(buf->size() > maxpsize_)
        vmp::except("net::Udp::sendTo large packet size [max=%d]",maxpsize_);
    if((ret=sendto(socket_,(void *)buf->pointer(),buf->size(),0,peer->addr(),peer->addrsize()))==-1)
       vmp::except_errno();
    fsync(socket_);
    return ret; 
}

vmp_int Udp::recvFrom(vmp::Buf *buf,Address *peer)
{
    vmp::except_check_pointer((void *)buf,"net::Udp::recvFrom() null pointer input buffer");
    buf->reset();
    vmp::except_check_pointer((void *)peer,"net::Udp::recvFrom() null pointer peer address");
    vmp_char tmp[maxpsize_];
    struct sockaddr_storage peeraddr;
    socklen_t peeraddr_len=sizeof(struct sockaddr_storage);
    vmp::bzero_wrap(tmp,maxpsize_);
    vmp_int ret=recvfrom(socket_,(void *) &tmp,maxpsize_,0,(net::Sockaddr *) &peeraddr, &peeraddr_len);
    if(ret == 0)
       return 0;
    else if(ret == -1)
    {
        if((errno == EAGAIN) || (errno == EWOULDBLOCK
) || (errno == EINTR))
            return -1;
        vmp::except_errno();
    }
    peer->set((net::Sockaddr *)&peeraddr,peeraddr_len);
    buf->write_array((vmp_byte *) tmp,(vmp_size) ret);
    buf->index();
    return ret;
}

void Udp::close()
{
    socket_close(&socket_);
} 

void udp_ev(event::Cell *cell)
{
    vmp_bool exec=false;
    event::Manager *manager=cell->get_manager();
    net::EventUdp *event=cell->event<EventUdp>();
    vmp::Buf buf;
    net::Address peer; 
    vmp_int ret;
    manager->lock();
    try
    {
        ret=event->udp_.recvFrom(&buf,&peer);
        if(ret == 0)
            manager->cell_close(cell,event::SUCCESS);
        else if(ret > 0)
            exec=manager->cell_update(cell);
    }
    catch(vmp::exception &x)
    {
        manager->cell_close(cell,event::ERROR,x.what());
    }
    manager->unlock();
    if(exec)
        event->recv_(cell,&buf,&peer);
    buf.reset();
    peer.reset();
}

EventUdp::EventUdp():event::Event()
{
}
       
EventUdp::~EventUdp()
{
}

void EventUdp::evt_udp_init(vmp_size maxpsize,net::EVTCBRECVFROM recv)
{
    if(recv == 0)
        recv_=net::empty_recvfrom_ev;
    else
        recv_=recv;
    udp_.set_maxpsize(maxpsize);
}

void EventUdp::evt_udp_reset()
{
    recv_=0;
    udp_.reset();
    local_.reset();
}

event::Cell *EventUdp::evt_udp_server(event::UI *ui,net::Address *local,vmp_size maxpsize,net::EVTCBRECVFROM recv,event::EVTCB close)
{
    evt_udp_init(maxpsize,recv);
    if((local == 0) || (local->iptype() == PF_UNSPEC) || (local->service() == "0"))
        vmp::except("net::evt_udp_server invalid address bind");
    local_.copy(local);
    try
    {
        udp_.server(&local_);
    }
    catch(vmp::exception &x)
    {
        evt_udp_reset();
        vmp::except_s(x.what());
    }
    getaddr_=false;
    return evt_new(ui,udp_.socket(),udp_ev,close); 
}

event::Cell *EventUdp::evt_udp_client4(event::UI *ui,vmp_size maxpsize,net::EVTCBRECVFROM recv,event::EVTCB close)
{
    evt_udp_init(maxpsize,recv);
    try
    {
        udp_.client4();
        udp_.get_local_address(&local_);
    }
    catch(vmp::exception &x)
    {
        evt_udp_reset();
        vmp::except_s(x.what());
    }
    getaddr_=true;
    return evt_new(ui,udp_.socket(),udp_ev,close); 
}
       
event::Cell *EventUdp::evt_udp_client6(event::UI *ui,vmp_size maxpsize,net::EVTCBRECVFROM recv,event::EVTCB close)
{
    evt_udp_init(maxpsize,recv);
    try
    {
        udp_.client6();
        udp_.get_local_address(&local_);
    }
    catch(vmp::exception &x)
    {
        evt_udp_reset();
        vmp::except_s(x.what());
    }
    getaddr_=true;
    return evt_new(ui,udp_.socket(),udp_ev,close); 
}

void EventUdp::evt_udp_sendTo(vmp::Buf *buf,net::Address *peer)
{
     event::Manager *manager=cell_->get_manager();
     if(manager->cell_update(cell_))
     {
         try
         {
             if(udp_.sendTo(buf,peer) == 0)
                 manager->cell_close(cell_,event::SUCCESS);
             if(getaddr_)
             {
                 udp_.get_local_address(&local_);
                 getaddr_=false;
             }
         }
         catch(vmp::exception &x)
         {
             manager->cell_close(cell_,event::ERROR,x.what());
         }
     }
}

vmp_size EventUdp::evt_udp_maxpsize()
{
    return udp_.maxpsize();
}

net::Address *EventUdp::evt_udp_localaddr()
{
    return &local_;
}

void EventUdp::evt_udp_close()
{
    udp_.close();
    evt_close();
}

void EventUdp::evt_udp_free()
{
    evt_udp_reset();
    evt_free();
}

UdpUI::UdpUI(event::Manager *manager):event::UI(manager)
{
    set_event(0,0);
}

UdpUI::~UdpUI()
{
}

vmp::str UdpUI::identity(event::Cell *cell)
{
    net::EventUdp *uevent=cell->event<net::EventUdp>();
    vmp::str ret="";
    net::Address *addr=uevent->evt_udp_localaddr();
    vmp::unicode::str_write(&ret,"Udp event [%s:%s][maxpsize=%d]",addr->host().c_str(),addr->service().c_str(),uevent->evt_udp_maxpsize());
    return ret;
}
       
void UdpUI::close_event(event::Cell *cell)
{
    net::EventUdp *uevent=cell->event<net::EventUdp>();
    uevent->evt_udp_close();
}
       
void UdpUI::free_ref(event::Cell *cell)
{
    net::EventUdp *uevent=cell->event<net::EventUdp>();
    uevent->evt_udp_free();
    uref_.free(uevent);
}

void UdpUI::set_event(net::EVTCBRECVFROM revent,event::EVTCB cevent,vmp_size maxpsize)
{
    revent_=revent;
    cevent_=cevent;
    maxpsize_=maxpsize;
}

event::Cell *UdpUI::new_server(net::Address *local)
{
    event::Cell *cell=0;
    manager_->lock();
    net::EventUdp *udp=uref_.get();
    try
    {
        cell=udp->evt_udp_server(this,local,maxpsize_,revent_,cevent_);
    }
    catch(vmp::exception &x)
    {
        uref_.free(udp);
        manager_->unlock();
        vmp::except("net::UdpUI::add_server() '%s'",x.what());
    }
    manager_->unlock();
    return cell;
}

event::Cell *UdpUI::new_client4()
{
    event::Cell *cell=0;
    manager_->lock();
    net::EventUdp *udp=uref_.get();
    try
    {
        cell=udp->evt_udp_client4(this,maxpsize_,revent_,cevent_);
    }
    catch(vmp::exception &x)
    {
        uref_.free(udp);
        manager_->unlock();
        vmp::except("net::UdpUI::add_client4() '%s'",x.what());
    }
    manager_->unlock();
    return cell;
}

event::Cell *UdpUI::new_client6()
{
    event::Cell *cell=0;
    manager_->lock();
    net::EventUdp *udp=uref_.get();
    try
    {
        cell=udp->evt_udp_client6(this,maxpsize_,revent_,cevent_);
    }
    catch(vmp::exception &x)
    {
        uref_.free(udp);
        manager_->unlock();
        vmp::except("net::UdpUI::add_client6() '%s'",x.what());
    }
    manager_->unlock();
    return cell;
}

void UdpUI::sendTo(event::Cell *cell,vmp::Buf *buf,net::Address *peer)
{
    net::EventUdp *uevent=cell->event<net::EventUdp>();
    event::Manager *manager=cell->get_manager();
    manager->lock();
    try
    {
        uevent->evt_udp_sendTo(buf,peer);
    }
    catch(vmp::exception &x)
    {
        manager->unlock();
        vmp::except("event::UdpUI::sendTo() '%s'",x.what());
    }
    manager->unlock();
}

vmp_size UdpUI::maxpsize(event::Cell *cell)
{
    net::EventUdp *uevent=cell->event<net::EventUdp>();
    event::Manager *manager=cell->get_manager();
    manager->lock();
    vmp_size ret=uevent->evt_udp_maxpsize();
    manager->unlock();
    return ret;
}

net::Address *UdpUI::localaddr(event::Cell *cell)
{
    net::EventUdp *uevent=cell->event<net::EventUdp>();
    event::Manager *manager=cell->get_manager();
    manager->lock();
    net::Address *ret=uevent->evt_udp_localaddr();
    manager->unlock();
    return ret;
}

}}

