#!/usr/bin/python

#This program is free software; you can redistribute it and/or modify
#it under the terms of the GNU General Public License version 3 as
#published by the Free Software Foundation;

#This program is distributed in the hope that it will be useful,
#but WITHOUT ANY WARRANTY; without even the implied warranty of
#MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#GNU General Public License for more details.

#You should have received a copy of the GNU General Public License
#along with this program; if not, write to the Free Software
#Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#Author: Marco Guastella alias Vasta
#Web page:<www.ragnu.it>
#Email: <vasta@ragnu.it>
#Date last Update : 30/03/2019

import vmp
from vmp_gui0 import popup as popup

import tkinter as tk

##default label text widget
label_="None"

##popup title fileselect widget
fileselect_title_='Select File path'

##popup title dirselect widget
dirselect_title_='Select Dir path'

##popup default init select dir
initdir_='~'

##default width widget value
width_=10

##default height widget value
height_=5

##default text value widget
text_entry_=''

##default text value widget
text_text_=[]

##Widget label
class Label:
  ##A Constructor
  #
  #@param frame frame associated
  #@param text label text
  #@param bg background color
  #@param fg foreground color
  #@param maxline maximum number of characters in a line
  def __init__(self,frame,text=label_,bg=None,fg=None,maxline=None):
    self.text=None
    self.label=tk.Label(frame.ref,text=self.text,bg=bg,fg=fg,justify=tk.LEFT)
    self.set_text(text,maxline)
  
  ##Set text label
  #
  #@param text label text
  #@param maxline maximum number of characters in a line  
  def set_text(self,text,maxline=None):
    self.text=text
    if maxline == None:
      self.label.configure(text=text)
    else:
      ftext=vmp.unicode.str_format_maxline(self.text,maxline)
      self.label.configure(text=ftext)

  ##Returns widget
  def get_widget(self):
    return self.label

  ##Returns Label text
  def get_value(self):
    return self.text

##Widget label
class Button:
  ##A Constructor
  #
  #@param frame frame associated
  #@param text button text
  #@param callback performed when the button is clicked (def coomand(self))
  def __init__(self,frame,text=label_,command=None):
    self.text=''    
    self.button=tk.Button(frame.ref,text=self.text,command=command)
    self.set_text(text)
  
  ##Set text label
  #
  #@param text button text
  def set_text(self,text):
    self.text=text
    self.button.configure(text=self.text)

  ##Enable button
  #
  #@param text button text
  def enable(self,text=None):
    if text is not None:
      self.text=text    
    self.button.configure(text=self.text,state='normal')

  ##Disable button
  #
  #@param text button text
  def disable(self,text=None):
    if text is not None:
      self.text=text      
    self.button.configure(text=self.text,state='disabled')  
  
  ##Returns widget
  def get_widget(self):
    return self.button

  ##Returns button text
  def get_value(self):
    return self.text

##Widget checkbutton
class CheckButton:
  ##A Constructor
  #
  #@param frame frame associated
  #@param active is active after construction?
  #@param label label text
  #@param bg background color
  def __init__(self,frame,active=False,label=None,bg=None):
    self.value=tk.BooleanVar()
    self.checkbutton=tk.Checkbutton(frame.ref,bg=bg,bd=0,activebackground=bg,variable=self.value,onvalue=True,offvalue=False,text=label)
    if active:
      self.checkbutton.select()
  
  ##Returns widget 
  def get_widget(self):
    return self.checkbutton
   
  ##Returns boolean value
  def get_value(self):
    return self.value.get()

##Widget options
class Options:
  ##A Constructor
  #
  #@param frame frame associated
  #@param labels list labels value
  def __init__(self,frame,labels):
    self.value=tk.StringVar()
    self.value.set(labels[0])    
    self.options=tk.OptionMenu(frame.ref,self.value,*(labels))

  ##Returns widget 
  def get_widget(self):
    return self.options

  ##Returns the selected value
  def get_value(self):
    return self.value.get()

##Widget entry
class Entry:
  ##A Constructor
  #
  #@param frame frame associated
  #@param size entry size
  #@param bg background color
  #@param fg foreground color
  #@param fbg background color of the frame containing the entry
  #@param value default text value.(list string,one for line)
  #@param scroll insert scrollbar?
  #@param fselect insert file select button?
  #@param dselect insert directory select button?
  #@param rst insert reset button?
  #@param initdir initial directory for file select and directory select action
  #@show  clear text?
  def __init__(self,frame,size=width_,bg=None,fg=None,fbg=None,value=text_entry_,scroll=False,fselect=False,dselect=False,rst=False,initdir=initdir_,show=True): 
    self.container=frame.ref
    self.initdir=initdir
    self.frame=tk.Frame(self.container,padx=0,pady=0,bg=fbg)    
    self.value=tk.StringVar()
    self.value.set(value)    
    if scroll:    
      xScroll=tk.Scrollbar(self.frame,orient=tk.HORIZONTAL)
      xScroll.grid(row=1, column=0,sticky=tk.E+tk.W)
      if show:
        entry=tk.Entry(self.frame,xscrollcommand=xScroll.set,bg=bg,fg=fg,width=size,textvariable=self.value)
      else:
        entry=tk.Entry(self.frame,xscrollcommand=xScroll.set,bg=bg,fg=fg,width=size,textvariable=self.value,show='*')
      entry.grid(row=0,column=0)
      xScroll['command']=entry.xview
    else:
      if show:
        entry=tk.Entry(self.frame,bg=bg,fg=fg,width=size,textvariable=self.value)
      else:
        entry=tk.Entry(self.frame,bg=bg,fg=fg,width=size,textvariable=self.value,show='*')
      entry.grid(row=0,column=0)    
    i=1
    if fselect:
      sbutton=tk.Button(self.frame,text='f',padx=0,pady=0,command=self.select_file)
      sbutton.grid(row=0,column=i)
      i=i+1
    if dselect:
      sbutton=tk.Button(self.frame,text='d',padx=0,pady=0,command=self.select_dir)
      sbutton.grid(row=0,column=i)
      i=i+1
    if rst:
      sbutton=tk.Button(self.frame,text='r',padx=0,pady=0,command=self.reset)
      sbutton.grid(row=0,column=i)
    
  ##Returns widget 
  def get_widget(self):
    return self.frame
  
  ##Returns entry value
  def get_value(self):
    return self.value.get()

  ##Callback select an file and write the path in the entry
  def select_file(self):
    filepath=popup.ask_openfilename(self.container,fileselect_title_,self.initdir)    
    if filepath is not None:
      self.value.set("{0}{1}".format(self.value.get(),filepath))      
  
  ##Callback select a directory and write the path in the entry
  def select_dir(self):
    dirpath=popup.ask_directory(self.container,dirselect_title_,self.initdir)    
    if dirpath is not None:      
      self.value.set("{0}{1}".format(self.value.get(),dirpath))

  ##Callback reset entry text
  def reset(self):
    self.value.set('')

##Widget Text
class Text:
  ##A Constructor
  #
  #@param frame frame associated
  #@param sizew witdh size text entry
  #@param sizeh height size text entry
  #@param bg background color
  #@param fg foreground color
  #@param fbg background color of the frame containing the entry
  #@param value default text entry
  #@param scroll insert scrollbar?
  #@param fselect insert file select button?
  #@param dselect insert directory select button?
  #@param paste insert paste file button?
  #@param rst insert reset button?
  #@param initdir initial directory for file select and directory select action
  def __init__(self,frame,sizew=width_,sizeh=height_,bg=None,fg=None,fbg=None,value=text_text_,scroll=False,fselect=False,dselect=False,paste=False,rst=False,initdir=initdir_):
    self.container=frame.ref 
    self.initdir=initdir
    self.frame=tk.Frame(self.container,padx=0,pady=0,bg=fbg)

    if scroll:  
      xScroll=tk.Scrollbar(self.frame,orient=tk.HORIZONTAL)
      xScroll.grid(row=1, column=0,sticky=tk.E+tk.W)      
      yScroll=tk.Scrollbar(self.frame,orient=tk.VERTICAL)
      yScroll.grid(row=0,column=1,rowspan=1,sticky=tk.N+tk.S)
      self.text=tk.Text(self.frame,wrap=tk.NONE,bg=bg,fg=fg,width=sizew,height=sizeh,xscrollcommand=xScroll.set,yscrollcommand=yScroll.set)
      self.text.grid(row=0,column=0)
      xScroll['command'] = self.text.xview      
      yScroll['command'] = self.text.yview
    else:    
      yScroll=tk.Scrollbar(self.frame,orient=tk.VERTICAL)
      yScroll.grid(row=0,column=1,rowspan=1,sticky=tk.N+tk.S)
      self.text=tk.Text(self.frame,wrap=tk.NONE,bg=bg,fg=fg,width=sizew,height=sizeh,yscrollcommand=yScroll.set)
      self.text.grid(row=0, column=0)
      yScroll['command'] = self.text.yview  
    
    i=0    
    for line in value:
      if i == 0:      
        self.text.insert(tk.END,"{0}".format(line))
      else:
        self.text.insert(tk.END,"\n{}".format(line))      
      i=i+1
    
    i=2
    if fselect:
      sbutton=tk.Button(self.frame,text='f',padx=0,pady=0,command=self.select_file)
      sbutton.grid(row=0,column=i)
      i=i+1
    if dselect:
      sbutton=tk.Button(self.frame,text='d',padx=0,pady=0,command=self.select_dir)
      sbutton.grid(row=0,column=i)
      i=i+1
    if paste:
      sbutton=tk.Button(self.frame,text='p',padx=0,pady=0,command=self.paste_file)
      sbutton.grid(row=0,column=i)
      i=i+1    
    if rst:
      sbutton=tk.Button(self.frame,text='r',padx=0,pady=0,command=self.reset)
      sbutton.grid(row=0,column=i)

  ##Returns widget
  def get_widget(self):
    return self.frame
  
  ##Returns text value
  def get_value(self):
    ret=self.text.get('0.0',tk.END).split('\n')
    if (len(ret) == 2) and (ret[0] == '') and (ret[1] == ''):
      return []
    return ret[0:len(ret)-1]
  
  ##Callback select an file and write the path in the text entry
  def select_file(self):
    filepath=popup.ask_openfilename(self.container,fileselect_title_,self.initdir)    
    if filepath is not None:      
      self.text.insert(tk.END,"{0}".format(filepath))
      
  ##Callback select a directory and write the path in the text entry      
  def select_dir(self):
    dirpath=popup.ask_directory(self.container,dirselect_title_,self.initdir)    
    if dirpath is not None:    
      self.text.insert(tk.END,"{0}".format(dirpath))

  ##Callback paste the contents of a file
  def paste_file(self):
    filepath=popup.ask_openfilename(self.container,fileselect_title_,self.initdir)    
    if filepath is not None:
      rfile=vmp.input_file(filepath)
      i=0
      for line in rfile:     
        if i == 0:
          self.text.insert(tk.END,"{0}".format(line))
        else:
          self.text.insert(tk.END,"\n{0}".format(line))
        i=i+1
  
  ##Callback reset text value
  def reset(self):
    self.text.delete('0.0',tk.END)

##Widget listbox
class ListBox:
  ##A Constructor
  #
  #@param frame frame associated
  #@param sizew witdh size listbox
  #@param sizeh height size listbox
  #@param bg background color
  #@param fg foreground color
  #@param fbg background color of the frame containing the listbox
  #@param listdata listdata value(listdata is tuple)
  def __init__(self,frame,sizew=width_,sizeh=height_,bg=None,fg=None,fbg=None,listdata=()):
    self.frame=tk.Frame(frame.ref,bg=fbg)
    yScroll=tk.Scrollbar(self.frame,orient=tk.VERTICAL)
    yScroll.grid(row=0,column=1,rowspan=2,sticky=tk.N+tk.S)
    xScroll=tk.Scrollbar(self.frame,orient=tk.HORIZONTAL)
    xScroll.grid(row=1, column=0,sticky=tk.E+tk.W)
    self.listbox=tk.Listbox(self.frame,xscrollcommand=xScroll.set,yscrollcommand=yScroll.set,bg=bg,fg=fg,width=sizew,height=sizeh)
    self.listbox.grid(row=0, column=0)
    xScroll['command'] = self.listbox.xview
    yScroll['command'] = self.listbox.yview    
    self.update_data(listdata)

  ##reset list box
  def reset(self):
    self.listbox.delete(0,tk.END)
  
  ##Update listdata value(listdata is tuple)
  def update_data(self,listdata):
    self.reset()
    self.listdata=listdata    
    for data in self.listdata:      
      self.listbox.insert(tk.END,data)  

  ##Returns widget
  def get_widget(self):
    return self.frame
    
  ##Returns selected value,None is value is not selected
  def get_value(self):
    try:
      ret=self.listbox.selection_get()
      for l in self.listdata:
        if l == ret:
          return ret
    except:
      pass
    return None

##Widget Editor
class Editor:
  ##A Constructor
  #
  #@param frame frame associated
  #@param sizew witdh size listbox
  #@param sizeh height size listbox
  #@param bg background color
  #@param fg foreground color
  #@param fbg background color of the frame containing the listbox
  #@param text init text
  def __init__(self,frame,sizew=width_,sizeh=height_,bg=None,fg=None,fbg=None,text=''):
    self.frame=tk.Frame(frame.ref,bg=fbg)
    xScroll=tk.Scrollbar(self.frame,orient=tk.HORIZONTAL)
    xScroll.grid(row=1, column=0,sticky=tk.E+tk.W)      
    yScroll=tk.Scrollbar(self.frame,orient=tk.VERTICAL)
    yScroll.grid(row=0,column=1,rowspan=1,sticky=tk.N+tk.S)
    self.text=tk.Text(self.frame,wrap=tk.NONE,bg=bg,fg=fg,width=sizew,height=sizeh,xscrollcommand=xScroll.set,yscrollcommand=yScroll.set)
    self.text.grid(row=0, column=0)
    xScroll['command'] = self.text.xview      
    yScroll['command'] = self.text.yview
    self.set_value(text)

  ##Returns widget
  def get_widget(self):
    return self.frame

  ##Set text value
  #
  #@param text text to set
  def set_value(self,text):
    self.text.delete('@0,0',tk.END)
    self.text.insert(tk.END,text)
  
  ##Returns text editor
  def get_value(self):
    return self.text.get('@0,0',tk.END)


