#!/usr/bin/python

#This program is free software; you can redistribute it and/or modify
#it under the terms of the GNU General Public License version 3 as
#published by the Free Software Foundation;

#This program is distributed in the hope that it will be useful,
#but WITHOUT ANY WARRANTY; without even the implied warranty of
#MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#GNU General Public License for more details.

#You should have received a copy of the GNU General Public License
#along with this program; if not, write to the Free Software
#Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#Author: Marco Guastella alias Vasta
#Web page:<www.ragnu.it>
#Email: <vasta@ragnu.it>
#Date last Update : 15/06/2020

import vmp
import vmp_xterm0 as xterm
import vmp_gui0   as gui

import config       as cfg 
import gvmp_parser  as prs

def fatal_error(frame,error):
  gui.popup.showerror(title=cfg.frame_fatal_error_,error=error,parent=frame)
  frame.close()
  vmp.exit_failure(error)
  
class ModInfo(gui.Frame):
   def __init__(self):
     gui.Frame.__init__(self,bg=cfg.frame_bg_,title=cfg.frame_title_+cfg.lcmd_imodules_,
                        padx=cfg.frame_padx_,pady=cfg.frame_pady_,ipadx=cfg.frame_ipadx_,ipady=cfg.frame_ipady_)
     self.add_widget_w(gui.widget.Label(self,text=cfg.l_imodules_,bg=cfg.frame_bg_),0,0)
     self.entry=gui.widget.Entry(self,size=cfg.frame_entry_size_,scroll=True)
     self.add_widget_w(self.entry,0,1)
     self.add_widget_c(gui.widget.Button(self,text=cfg.frame_button_run_,command=self.run),1,0,cspan=2)
  
   def run(self):
     value=self.entry.get_value()
     xterm.shell(cfg.cmd_imodules_(value),"{0} {1} {2}".format(cfg.main_helpmenu_,cfg.lcmd_imodules_,value))
      
class GuiMask(gui.Frame):
  def __init__(self,maskname):
    self.maskname=maskname
    title="{0}{1}{2}".format(cfg.frame_title_,cfg.mask_subtitle_,maskname)
    gui.Frame.__init__(self,bg=cfg.frame_bg_,title=title,posx=cfg.main_posx_,posy=cfg.main_posy_,
                       ipadx=cfg.mask_ipadx_,ipady=cfg.mask_ipady_,exit=self.exit)
    self.set_scrollbar(scrollmax=cfg.main_height_)
    self.parser=None
    self.tagvars={}
    self.pycode=[]
    self.root=None
    self.path=None
    self.x=0
    self.y=0 
 
  def row(self):
    self.x=self.x+1
    self.y=0

  def column(self,advance=1):
    self.y=self.y+advance

  def create_data(self,filepath):
    try:
      self.parser=prs.parse(filepath,self)
      self.fpath="{0}{1}_{2}.xml".format(cfg.path_auto_,self.maskname,int(vmp.time.time_wrap()))      
      self.filelabel=gui.widget.Label(self,text="{0}{1}".format(cfg.mask_titlepath_,self.fpath),bg=cfg.frame_bg_)      
      self.add_widget_w(self.filelabel,self.x,self.y,cspan=cfg.mask_grid_width_)      
      self.row()    
      self.cbutton=gui.widget.Button(self,text=cfg.mask_cbutton_,command=self.change_runlocation)      
      self.add_widget_w(self.cbutton,self.x,self.y)
      self.column()    
      self.sbutton=gui.widget.Button(self,text=cfg.mask_sbutton_,command=self.save)
      self.add_widget_w(self.sbutton,self.x,self.y)
      self.column()
      self.debug=gui.widget.CheckButton(self,label=cfg.mask_debug_,bg=cfg.frame_bg_)
      self.add_widget_w(self.debug,self.x,self.y)          
      self.column()
      self.rbutton=gui.widget.Button(self,cfg.mask_rbutton_,command=self.run)
      self.add_widget_w(self.rbutton,self.x,self.y)
    except Exception as e:
      vmp.except_s(str(e))
      
  def add_vartag(self,entry):
    if entry.var in self.tagvars:
       entry.parser_error("duplicate var {0} (prev {1})".format(entry.var,self.tagvars[entry.var].get_line()))
    self.tagvars[entry.var]=entry
     
  def enable_button(self,wait):
    if wait is not 0:
      vmp.time.vsleep(wait)
    self.cbutton.enable()    
    self.sbutton.enable()
    self.rbutton.enable()
  
  def disable_button(self):
    self.cbutton.disable()    
    self.sbutton.disable()
    self.rbutton.disable()
      
  def vampiria_out(self):
    ret=True
    var={}
    try:
      for tvar in self.tagvars:
        self.tagvars[tvar].update_value()
        var[tvar]=self.tagvars[tvar].get_data()
      for c in self.pycode:
        var[c.var]=c.exec_code(var)
      prs.write_parser(self.root,self.fpath,var)
    except Exception as e:
      gui.popup.showerror(title=cfg.common_inputerr_,error=str(e),parent=self)
      ret=False      
    var={}
    return ret

  def save(self):
    self.disable_button()    
    if self.vampiria_out() == True:    
      self.enable_button(cfg.mask_sleep_run_)
    else:
      self.enable_button(0)
    
  def run(self):
    self.disable_button()    
    if self.vampiria_out() == True:
      try:
        xterm.shell(cfg.cmd_exec_(self.fpath,self.debug.get_value()),cfg.lcmd_exec(self.fpath,self.debug.get_value()))
      except Exception as e:
        gui.popup.showerror(title=cfg.lrun_error_,error=str(e),parent=self) 
      self.enable_button(cfg.mask_sleep_run_)
    else:
      self.enable_button(0)

  def change_runlocation(self):
    filepath=gui.popup.ask_saveasfilename(parent=self,title=cfg.mask_cbutton_,initdir=cfg.path_static_,filetypes=[("XML","*.xml")])
    if not filepath == None:
      self.fpath=filepath
      self.filelabel.set_text("{0}{1}".format(cfg.mask_titlepath_,self.fpath))

  def exit(self):
    self.tagvars={}
    if self.parser is not None:
      self.parser.destroy()
      self.parser=None
    self.root=None
    self.close()

class StaticMask(gui.Frame):
  def __init__(self):
    gui.Frame.__init__(self,bg=cfg.frame_bg_,title=cfg.frame_title_+cfg.static_subtitle_,
                       posx=cfg.main_posx_,posy=cfg.main_posy_,ipadx=cfg.static_ipadx_,
                       ipady=cfg.static_ipady_,exit=self.exit)
    self.set_scrollbar(scrollmax=cfg.main_height_)
    self.fpath=None
    self.fname=None
    self.flabel=gui.widget.Label(self,bg=cfg.frame_bg_) 
    self.add_widget_c(self.flabel,0,0,cspan=cfg.static_grid_width_)
    self.bopen=gui.widget.Button(self,text=cfg.static_bopen_,command=self.eopen)
    self.add_widget_c(self.bopen,1,0)
    self.bsave=gui.widget.Button(self,text=cfg.static_bsave_,command=self.save)
    self.add_widget_c(self.bsave,1,1)
    self.bsaveas=gui.widget.Button(self,text=cfg.static_bsaveas_,command=self.saveas)
    self.add_widget_c(self.bsaveas,1,2)
    self.bdelete=gui.widget.Button(self,text=cfg.static_bdelete_,command=self.edelete)
    self.add_widget_c(self.bdelete,1,3)
    self.editor=gui.widget.Editor(self,sizew=cfg.static_editor_width_,sizeh=cfg.static_editor_height_)
    self.add_widget_c(self.editor,2,0,cspan=cfg.static_grid_width_)
    self.bcheck=gui.widget.Button(self,text=cfg.static_bcheck_,command=self.echeck)
    self.add_widget_c(self.bcheck,3,0)
    self.debug=gui.widget.CheckButton(self,label=cfg.static_bdebug_,bg=cfg.frame_bg_)
    self.add_widget_c(self.debug,3,1)
    self.brun=gui.widget.Button(self,text=cfg.static_brun_,command=self.erun)
    self.add_widget_c(self.brun,3,2)
    self.refresh_button()

  def refresh_button(self):
    if self.fpath is None:
      self.bsave.disable()
      self.bsaveas.disable()
      self.bdelete.disable()
      self.bcheck.disable()
      self.brun.disable()
      self.flabel.set_text("{0}empty".format(cfg.static_titlepath_))
    else:
      self.flabel.set_text("{0}{1}".format(cfg.static_titlepath_,self.fname))
      self.bsave.enable()
      self.bsaveas.enable()
      self.bdelete.enable()
      self.bcheck.enable()
      self.brun.enable()

  def eopen(self):
    f=gui.popup.ask_openfilename(self,cfg.static_run_ask_,cfg.path_static_,filetypes=[("XML","*.xml")])
    if f is not None:
      try:
        self.fpath=f
        split=vmp.unicode.str_split(self.fpath,"/")
        self.fname=split[len(split)-1]
        fstr=vmp.input_file_s(self.fpath)
        self.editor.set_value(fstr)
        self.refresh_button()
      except:
        pass
    
  def edelete(self):
    result=gui.popup.askquestion(title=cfg.static_bdelete_,question=cfg.frame_confrm_delete_msg_(self.fname))
    if result:
      try:
        vmp.fs.unlink_wrap(self.fpath)
        self.fpath=None
        self.fname=None
        self.editor.set_value('')
        self.refresh_button()
      except Exception as e:
        gui.popup.showerror(title=cfg.frame_delete_error_title_(self.fname),error=str(e),parent=self)       

  def save(self):
    if self.fpath is not None:
      fstr=self.editor.get_value()
      try:
        vmp.output_file(self.fpath,True,fstr)
        return True
      except Exception as e:
        gui.popup.showerror(title=cfg.lsave_error_,error=str(e),parent=self)
    return False    
  
  def saveas(self):
    result=gui.popup.ask_saveasfilename(self,cfg.static_save_ask_,cfg.path_static_,filetypes=[("XML","*.xml")])
    if result is not None:
      self.fpath=result
      split=vmp.unicode.str_split(self.fpath,"/")
      self.fname=split[len(split)-1]
      self.save()
      self.refresh_button()
    
  def echeck(self):
    cexec=True
    if not self.save():
      cexec=gui.popup.askquestion(title=cfg.lsave_error_,question=cfg.lsave_question_error_(self.fname))  
    if cexec:
      try:
        xterm.shell(cfg.cmd_check_(self.fpath),cfg.lcmd_check_(self.fname))
      except Exception as e:
        gui.popup.showerror(title=cfg.lcheck_error_,error=str(e),parent=self)

  def erun(self):
    cexec=True
    if not self.save():
      cexec=gui.popup.askquestion(title=cfg.lsave_error_,question=cfg.lsave_question_error_(self.fname))
    try:
      xterm.shell(cfg.cmd_exec_(self.fpath,self.debug.get_value()),cfg.lcmd_exec(self.fpath,self.debug.get_value()))
    except Exception as e:
      gui.popup.showerror(title=cfg.lrun_error_,error=str(e),parent=self) 

  def exit(self):
    self.close()

class CustomMask(gui.Frame):
  def __init__(self):
    gui.Frame.__init__(self,bg=cfg.frame_bg_,title=cfg.frame_title_+cfg.custom_subtitle_,
                       posx=cfg.main_posx_,posy=cfg.main_posy_,ipadx=cfg.custom_ipadx_,
                       ipady=cfg.custom_ipady_,exit=self.exit)
    self.set_scrollbar(scrollmax=cfg.main_height_)
    self.fpath=None
    self.fname=None
    self.flabel=gui.widget.Label(self,bg=cfg.frame_bg_) 
    self.add_widget_c(self.flabel,0,0,cspan=cfg.custom_grid_width_)
    self.bopen=gui.widget.Button(self,text=cfg.custom_bopen_,command=self.eopen)
    self.add_widget_c(self.bopen,1,0)
    self.bsave=gui.widget.Button(self,text=cfg.custom_bsave_,command=self.save)
    self.add_widget_c(self.bsave,1,1)
    self.bsaveas=gui.widget.Button(self,text=cfg.custom_bsaveas_,command=self.saveas)
    self.add_widget_c(self.bsaveas,1,2)
    self.bdelete=gui.widget.Button(self,text=cfg.custom_bdelete_,command=self.edelete)
    self.add_widget_c(self.bdelete,1,3)
    self.editor=gui.widget.Editor(self,sizew=cfg.custom_editor_width_,sizeh=cfg.custom_editor_height_)
    self.add_widget_c(self.editor,2,0,cspan=cfg.custom_grid_width_)
    self.brun=gui.widget.Button(self,text=cfg.custom_brun_,command=self.erun)
    self.add_widget_c(self.brun,3,1,cspan=2)
    self.refresh_button()

  def refresh_button(self):
    if self.fpath is None:
      self.flabel.set_text("{0}empty".format(cfg.custom_titlepath_))
      self.bsave.disable()
      self.bsaveas.disable()
      self.bdelete.disable()
      self.brun.disable()
    else:
      self.flabel.set_text("{0}{1}".format(cfg.custom_titlepath_,self.fname))
      self.bsave.enable()
      self.bsaveas.enable()
      self.bdelete.enable()
      self.brun.enable()

  def eopen(self):
    f=gui.popup.ask_openfilename(self,cfg.custom_run_ask_,cfg.path_custom_,filetypes=[("XML","*.xml")])
    if f is not None:
      try:
        self.fpath=f
        split=vmp.unicode.str_split(self.fpath,"/")
        self.fname=split[len(split)-1]
        fstr=vmp.input_file_s(self.fpath)
        self.editor.set_value(fstr)
        self.refresh_button()
      except:
        pass

  def save(self):
    if self.fpath is not None:
      fstr=self.editor.get_value()
      try:
        vmp.output_file(self.fpath,True,fstr)
        return True
      except Exception as e:
        gui.popup.showerror(title=cfg.lsave_error_,error=str(e),parent=self)
    return False  

  def saveas(self):
    result=gui.popup.ask_saveasfilename(self,cfg.custom_save_ask_,cfg.path_custom_,filetypes=[("XML","*.xml")])
    if result is not None:
      self.fpath=result
      split=vmp.unicode.str_split(self.fpath,"/")
      self.fname=split[len(split)-1]
      self.save()
      self.refresh_button()

  def edelete(self):
    result=gui.popup.askquestion(title=cfg.custom_bdelete_,question=cfg.frame_confrm_delete_msg_(self.fname))
    if result:
      try:
        vmp.fs.unlink_wrap(self.fpath)
        self.fpath=None
        self.fname=None
        self.editor.set_value('')
        self.refresh_button()
      except Exception as e:
        gui.popup.showerror(title=cfg.frame_delete_error_title_(self.fname),error=str(e),parent=self)
  
  def ereplace(self):
    if self.save():
      try:
        doc_replaced(self.fpath,cfg.tagroot_,cfg.path_dtd_)
        fstr=vmp.input_file_s(self.fpath)
        self.editor.set_value(fstr)
        self.refresh_button()
      except:
        gui.popup.showerror(title=cfg.custom_replace_error_,error=str(e),parent=self)
      

  def erun(self):
    wmask=GuiMask(cfg.custom_project_title_(self.fname))    
    try:
      wmask.create_data(self.fpath)
      wmask.active()            
    except Exception as e:
      gui.popup.showerror(title=cfg.mask_error_title_,error=str(e),parent=self) 
      wmask.exit()
      del wmask

  def exit(self):
    self.close()

class TopFrame(gui.Frame):
  def __init__(self):
    gui.Frame.__init__(self,top=True,bg=cfg.frame_bg_,title=cfg.frame_title_+cfg.main_subtitle_,
                       width=cfg.main_width_,height=cfg.main_height_,posx=cfg.main_posx_,posy=cfg.main_posy_,
                       resize=cfg.main_resize_,padx=cfg.frame_padx_,pady=cfg.frame_pady_,
                       ipadx=cfg.frame_ipadx_,ipady=cfg.frame_ipady_,exit=self.exit)
    filemenu=gui.Menu(cfg.main_filemenu_,(gui.menu_cmd(cfg.lcmd_customized_open_,self.customized_open),gui.menu_cmd(cfg.lcmd_static_open_,self.static_open),gui.menu_separator(),gui.menu_cmd(cfg.lcmd_exit_,self.exit)))
    helpmenu=gui.Menu(cfg.main_helpmenu_,(gui.menu_cmd(cfg.lcmd_info_,self.cmd_info),gui.menu_cmd(cfg.lcmd_version_,self.cmd_version),gui.menu_cmd(cfg.lcmd_path_,self.cmd_path),gui.menu_cmd(cfg.lcmd_modules_,self.cmd_modules),gui.menu_cmd(cfg.lcmd_imodules_,self.cmd_imodules)))
    self.set_scrollbar(scrollmax=cfg.main_height_)
    self.set_menu((filemenu,helpmenu))
    self.data=vmp.fs.listendir(cfg.path_project_)
    self.data.sort()
    if vmp.getuid_wrap() == 0:
      self.add_widget_c(gui.widget.Label(self,text=cfg.main_root_,bg=cfg.frame_bg_),0,0,cspan=2)
    else:
      self.add_widget_c(gui.widget.Label(self,text=cfg.main_noroot_,bg=cfg.frame_bg_),0,0,cspan=2)
    self.add_widget_w(gui.widget.Label(self,text=cfg.main_lgroup_project_,bg=cfg.frame_bg_),1,0)
    self.ggui=gui.widget.ListBox(self,sizew=cfg.main_wbox_,sizeh=cfg.main_hbox_,listdata=self.data)
    self.add_widget_w(self.ggui,1,1)
    self.bproject=gui.widget.Button(self,cfg.main_lbutton_gopen_,self.open_dirproject)
    self.add_widget_c(self.bproject,2,1)    
    self.add_widget_w(gui.widget.Label(self,text=cfg.main_label_project_,bg=cfg.frame_bg_),3,0)  
    self.gui=gui.widget.ListBox(self,sizew=cfg.main_wbox_,sizeh=cfg.main_hbox_,listdata=())
    self.add_widget_w(self.gui,3,1)
    self.brun=gui.widget.Button(self,cfg.main_lbutton_grun_,self.run_project)
    self.add_widget_c(self.brun,4,1)

  def customized_open(self):
     mask=CustomMask()
     mask.active()

  def static_open(self):
     mask=StaticMask()
     mask.active()
     
  def cmd_info(self):
     xterm.shell(cfg.cmd_info_,"{0} {1}".format(cfg.main_helpmenu_,cfg.lcmd_info_)) 

  def cmd_version(self):
     xterm.shell(cfg.cmd_version_,"{0} {1}".format(cfg.main_helpmenu_,cfg.lcmd_version_))
     
  def cmd_path(self):
     xterm.shell(cfg.cmd_path_,"{0} {1}".format(cfg.main_helpmenu_,cfg.lcmd_path_))
     
  def cmd_modules(self):
     xterm.shell(cfg.cmd_modules_,"{0} {1}".format(cfg.main_helpmenu_,cfg.lcmd_modules_))
     
  def cmd_imodules(self):
     ModInfo().active()
     
  def exit(self):
    self.close()
    vmp.exit_ok()

  def open_dirproject(self):
    try:
      self.group=self.ggui.get_value()
      if self.group is None:
        gui.popup.showerror(title=cfg.main_ggui_error_title_,error=cfg.main_ggui_error_,parent=self)
      else:
        ldata=vmp.fs.listendir("{0}{1}/".format(cfg.path_project_,self.group))
        files=[]
        for filename in ldata:
          f=filename.split('.')
          if len(f) == 2 and f[1] == 'xml':      
             files.append(f[0])  
        files.sort()
        self.gui.update_data(tuple(files))
    except Exception as e:
      fatal_error(self,str(e))
      

  def run_project(self):
    name=self.gui.get_value()
    if  name is not None:
      filepath="{0}{1}/{2}.xml".format(cfg.path_project_,self.group,name)
      wmask=GuiMask(name)    
      try:
        wmask.create_data(filepath)
        wmask.active()            
      except Exception as e:
        gui.popup.showerror(title=cfg.mask_error_title_,error=str(e),parent=self) 
        wmask.exit()
        del wmask
    else:
      gui.popup.showerror(title=cfg.main_gui_error_title_,error=cfg.main_gui_error_msg_,parent=self)     

def sigint(sig,frame):
  global top
  top.exit()

def vmp_main():
  global top
  vmp.signal_py(("sigint",),sigint)
  if not vmp.fs.isdir(cfg.path_home_):
    vmp.fs.mkdir_wrap(cfg.path_home_,cfg.dir_priv_)
    vmp.fs.mkdir_wrap(cfg.path_custom_,cfg.dir_priv_)
    vmp.fs.mkdir_wrap(cfg.path_static_,cfg.dir_priv_)
    vmp.fs.mkdir_wrap(cfg.path_auto_,cfg.dir_priv_)
  top=TopFrame()
  top.active()
  vmp.exit_ok()
    
    
