/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 04/04/2020
 */

#include <openssl/ssl.h>
#include <openssl/err.h>
#include <openssl/x509v3.h>
#include <openssl/bn.h>
#include <openssl/asn1.h>
#include <openssl/evp.h>
#include <openssl/x509.h>
#include <openssl/x509_vfy.h>
#include <openssl/pem.h>
#include <openssl/bio.h>


#include "vmp.h"
#include "net.h"

#include "openssl/lib.h"
#include "openssl/pkey.h"
#include "openssl/x509.h"

#ifndef VAMPIRIA_OPENSSL_PKG_H

#define VAMPIRIA_OPENSSL_PKG_H 1

namespace vampiria { namespace openssl { namespace pkg {

//Initialized ssl lib
void init();

//Free structure ssl lib
void end();

//!Class used as container of context openssl(see SSL_CTX *)
class Ctx
{
    public:
        //! A constructor
        Ctx();

        //! A destructor
        ~Ctx();

        SSL_CTX *ctx_;/*!< Context openssl*/ 
};

//! Create new tls generic context
/*!
    @return new context
*/
openssl::pkg::Ctx *ctx_tls_generic();

//! Create new tls generic client context
/*!
    @return new context
*/
openssl::pkg::Ctx *ctx_tls_client();

//! Create new tls generic server context
/*!
    @return new context
*/
openssl::pkg::Ctx *ctx_tls_server();

//!Set the context for using the private key of a pem file
/*!
    @param ctx input context
    @param fpath file path private key pem
*/
void ctx_pem_pkey_file(openssl::pkg::Ctx *ctx,vmp::str fpath);

//!Set the context for using the certificate x509 of a pem file
/*!
    @param ctx input context
    @param fpath file path pem file x509
*/
void ctx_pem_x509_file(openssl::pkg::Ctx *ctx,vmp::str fpath);

//! Free memory context
/*!
    @param ctx context to free
*/
void ctx_free(openssl::pkg::Ctx *ctx);

typedef enum ssl_state {SSL_STATE_INIT,SSL_STATE_CONNECT,SSL_STATE_HANDSHAKE,SSL_STATE_READY,SSL_STATE_LISTEN,SSL_STATE_SHUTDOWN,SSL_STATE_RCLOSE,SSL_STATE_CLOSE} SSL_STATE;/*!<Ssl connection state for internal usage*/

//!Virtual class generic ssl connection
class Ssl
{
    protected:
        net::Socket socket_;/*!<Socket references*/
        vmp_bool blocking_;/*!< is blocking?*/
        vmp::str type_;/*!Ssl type connection(setting in derivated class)*/
        openssl::pkg::Ctx *ctx_;/*!Connection context*/
        SSL *ssl_;/*! Ssl handler*/
        openssl::pkg::SSL_STATE state_;/*!Connection state*/
    public:
        //! A Constructor
        /*!
            @param ctx ssl context associated
        */
        Ssl(openssl::pkg::Ctx *ctx);

        //!A Destructor(BUGS in SSL_FREE AND SSL_CLEAR)
        virtual ~Ssl();

        //! Reset connection
        void reset();

        //! Returns socket references
        /*!
           @sa socket_
           @return socket references
        */
        net::Socket socket();
        
        //! Setting socket in blocking mode.Must be done before the connection is made
        /*!
           @sa blocking_
        */
        void block();
        
        //! Setting socket in non blocking mode.Must be done before the connection is made
        /*!
           @sa blocking_
        */
        void noblock();
        
        //! Returns connection type
        /*!
            @sa type_
            @return connection type
        */
        vmp::str type();
 
        //!Gets socket address local
        /*!
            @param result address result
            @return void (except error)
            @sa socket_
        */
        void get_local_address(net::Address *result);
        
        //!Gets socket address peer
        /*!
            @param result address result
            @return void (except error)
            @sa socket_
        */
        virtual void get_peer_address(net::Address *result);
        
        //! Send stream packet
        /*!
            @param buf data to send
            @return size packet to send,0 socket close(except error)
        */
        vmp_int sendData(vmp::Buf *buf);
        
        //! Recv stream packet
        /*!
           @param buf received data
           @return size packet to receive,0 close connection,-1 no data received from non blocking socket(except error)
        */
        vmp_int recvData(vmp::Buf *buf);
  
        //!Renegotiate the session key with the peer
        /*!
           @return if error throew exception
        */
        virtual void key_update();

        //!Gets the peer's certificate
        /*!
            @param cout certificate output
            @return if error throew exception
        */
        virtual void get_peer_cerificate(openssl::pkg::X509_Wrap *cout);
	
	//! Shuotdown ssl connection
        /*!
            @return true if shoutdown is complete,otherwise false
        */
        virtual vmp_bool shutdown();

        //! Close ssl connection
        void close();
};

//! Ssl client connection
class SslClient:public openssl::pkg::Ssl
{
    public:
        //! A Constructor
        /*!
            @param ctx ssl context associated
        */
        SslClient(openssl::pkg::Ctx *ctx);
        
        //! A Destructor
        ~SslClient();

        //! Connect tcp client
        /*!
            @param server remote server to connect
            @return void (except error)
        */
        void connect(net::Address *server);
        
        //! Check if connection is active(for non blocking socket)
        /*!
            @return true if connection is active,otherwise false
        */
        vmp_bool connect_check();

        //! Runs the ssl handshake
        /*!
            @return true if true if the handshake is complete,otherwise false.If error throw exception
        */
        vmp_bool ssl_connect();
   
        //!Virtual function
        void get_peer_cerificate(openssl::pkg::X509_Wrap *cout);
};

//! Ssl server connection established(see openssl::pkg::SslListen)
class SslServer:public openssl::pkg::Ssl
{
    public:
        //! A Constructor
        /*!
            @param socket socket coonection associated
        */
        SslServer(vmp_int socket,openssl::pkg::Ctx *ctx);

        //! A Destructor
        ~SslServer();

        //! Check if connection is active(for non blocking socket)
        /*!
            @return true if connection is active,otherwise false.If error throw exception
        */
        vmp_bool ssl_accept();
};

//! Free ssl connection server created by openssl::pkg::Ssl
/*!
     @param server ssl connection server
*/
void sslserver_free_connection(openssl::pkg::SslServer *server);

//!Ssl listen server
class SslListen:public openssl::pkg::Ssl
{
    public:
        //!A Constructor
        SslListen(openssl::pkg::Ctx *ctx);

        //! A Destructor
        ~SslListen();

        //! virtual function implemented(return except)
        void get_peer_address(net::Address *result);

        //! Listen ssl server
        /*!
            @param local listen address
            @param backlog the maximum length to which the  queue  of pending  connections
        */  
        void server(net::Address *local,vmp_uint backlog);
        
        //! Accept tcp connection
        /*!
            @return openssl::pkg::SslServer if accepted connection(free memory with net::tcpserver_free_connection),0 otherwise(no blocking mode)  
        */
        openssl::pkg::SslServer *accept();

        //! virtual function implemented(return except)
        void key_update();
        
        //! virtual function implemented(return except)
        void get_peer_cerificate(openssl::pkg::X509_Wrap *cout);

        //! virtual function implemented(return true)
        vmp_bool shutdown();
};

}}}

#endif

#include "openssl/sslevent.h"
#include "openssl/sslui.h"

