/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 * Date last update: 04/07/2020
 */

#include "openssl0/openssl0.h"

namespace vampiria { namespace openssl { namespace pkg {

X509_Wrap::X509_Wrap()
{
    x509_=0;
    parent_=0;
}
        
X509_Wrap::~X509_Wrap()
{
    reset();
}

void X509_Wrap::reset()
{
    if(parent_ != 0)
        delete parent_;
    if(x509_ != 0)
        X509_free(x509_);
    x509_=0;
    parent_=0;    
}

void X509_Wrap::pointer_except()
{
    vmp::except_check_pointer((void *)x509_,"openssl::pkg::X509_Wrap() certificate pointer not setting");
}

void X509_Wrap::set(X509 *x509)
{
    x509_=x509;
}

X509_Wrap *X509_Wrap::parent()
{
    return parent_;
}

vmp_int X509_Wrap::version()
{
    pointer_except();
    return X509_get_version(x509_)+1;
}

void X509_Wrap::serial(vmp::Buf *bout)
{
    vmp::except_check_pointer((void *)bout,"openssl::pkg::X509_Wrap::serial() output buffer null pointer");
    bout->reset();
    pointer_except();
    ASN1_INTEGER *asn1=X509_get_serialNumber(x509_);
    if (asn1 == 0)
        vmp::except_s("openssl::pkg::X509_Wrap::serial() error getting serial number from certificate");
    for(vmp_int i=0;i < asn1->length; i++)
        bout->write_byte((vmp_byte)asn1->data[i]);
    bout->index();
}

vmp::str X509_Wrap::signature_info()
{
    pointer_except();
    vmp_int nid=X509_get_signature_nid(x509_);
    if (nid == NID_undef)
        vmp::except_s("openssl::pkg::X509_Wrap::signature_info() error getting signature info from certificate");
    const vmp_char *buf=OBJ_nid2ln(nid);
    vmp::str ret=buf;
    return ret;
}

vmp::str X509_Wrap::issuer()
{
    pointer_except();
    vmp::str ret=X509_NAME_oneline(X509_get_issuer_name(x509_),0,0);
    return ret;
}

vmp::str X509_Wrap::validity_before()
{
    pointer_except();
    vmp::Buf buf;
    ASN1_TIME *time = X509_get_notBefore(x509_);
    BIO *b = BIO_new(BIO_s_mem());
    try
    {
        if(!ASN1_TIME_print(b,time))
            vmp::except_s("");
        openssl::pkg::bio_to_buf(b,&buf);
    }
    catch(vmp::exception &x)
    {
        buf.reset();
        BIO_free(b);
        vmp::except_s("openssl::pkg::X509_Wrap::validity_before() error getting time");
    }
    vmp::str ret=buf.read_str(buf.size());
    buf.reset();
    BIO_free(b);
    return ret;
}

vmp::str X509_Wrap::validity_after()
{
    pointer_except();
    vmp::Buf buf;
    ASN1_TIME *time = X509_get_notAfter(x509_);
    BIO *b = BIO_new(BIO_s_mem());
    try
    {
        if(!ASN1_TIME_print(b,time))
            vmp::except_s("");
        openssl::pkg::bio_to_buf(b,&buf);
    }
    catch(vmp::exception &x)
    {
        buf.reset();
        BIO_free(b);
        vmp::except_s("openssl::pkg::X509_Wrap::validity_after() error getting time");
    }
    vmp::str ret=buf.read_str(buf.size());
    buf.reset();
    BIO_free(b);
    return ret;
}

vmp::str X509_Wrap::subject()
{
    pointer_except();
    vmp::str ret=X509_NAME_oneline(X509_get_subject_name(x509_),0,0);
    return ret;
}

void X509_Wrap::pubkey(openssl::pkg::PKey *kout)
{
    pointer_except();
    vmp::except_check_pointer((void *)kout,"openssl::pkg::X509_Wrap::pubkey() input key null pointer");
    EVP_PKEY *pkey=X509_get0_pubkey(x509_);
    vmp::except_check_pointer((void *)pkey,"openssl::pkg::X509_Wrap::pubkey() error getting public key");
    kout->set(pkey);
}

void X509_Wrap::id_issuer(vmp::Buf *out)
{
    vmp::except_check_pointer((void *)out,"openssl::pkg::X509_Wrap::id_issuer() output buffer null pointer");
    out->reset();
    pointer_except();
    ASN1_OCTET_STRING *keyid= (ASN1_OCTET_STRING *)X509_get0_authority_key_id(x509_);
    if(keyid == 0)
        vmp::except_s("openssl::pkg::X509_Wrap::id_issuer() error getting key id issuer");
    vmp_int l=ASN1_STRING_length(keyid);
    out->write_array((vmp_byte *)ASN1_STRING_get0_data(keyid),l);
    out->index();
}

void X509_Wrap::id_subject(vmp::Buf *out)
{
    vmp::except_check_pointer((void *)out,"openssl::pkg::X509_Wrap::id_subject() output buffer null pointer");
    out->reset();
    pointer_except();
    ASN1_OCTET_STRING *keyid= (ASN1_OCTET_STRING *)X509_get0_subject_key_id(x509_);
    if(keyid == 0)
        vmp::except_s("openssl::pkg::X509_Wrap::id_subject error getting key id subject");
    vmp_int l=ASN1_STRING_length(keyid);
    out->write_array((vmp_byte *)ASN1_STRING_get0_data(keyid),l);
    out->index();
}

vmp::str X509_Wrap::extensions()
{
    vmp::str ret;
    pointer_except();
    const STACK_OF(X509_EXTENSION) *exts = NULL;
    exts=X509_get0_extensions(x509_);
    BIO *b=BIO_new(BIO_s_mem());
    X509V3_extensions_print(b,0,exts,0,0);
    vmp::Buf buf;
    try
    {
        openssl::pkg::bio_to_buf(b,&buf);
    }
    catch(vmp::exception &x)
    {
        BIO_free(b);
        vmp::except("openssl::pkg::X509_Wrap::extensions() not extensions data found");
    }
    ret=buf.read_str(buf.size());
    BIO_free(b);
    return vmp::unicode::str_format_end(ret);
}

void X509_Wrap::signature_data(vmp::Buf *out)
{
    vmp::except_check_pointer((void *)out,"openssl::pkg::X509_Wrap::signature_data() output buffer null pointer");
    out->reset();
    pointer_except();
    const ASN1_BIT_STRING *sig=0;
    const X509_ALGOR *alg=0;
    X509_get0_signature(&sig,&alg,x509_);
    out->write_array(sig->data,sig->length);
    out->index(); 
}

void X509_Wrap::fingerprint_sha1(vmp::Buf *out)
{
    vmp::except_check_pointer((void *)out,"openssl::pkg::X509_Wrap::fingerprint_sha1() output buffer null pointer");
    out->reset();
    pointer_except();
    const EVP_MD *digest = EVP_sha1();
    out->newsize(20);
    vmp_uint len;
    if((X509_digest(x509_,digest,out->pointer(),&len) == 0) || (len != 20))
       vmp::except_s("openssl::pkg::X509_Wrap::fingerprint_sha1() error getting sha1 fingerprint");
}

void X509_Wrap::fingerprint_sha256(vmp::Buf *out)
{
    vmp::except_check_pointer((void *)out,"openssl::pkg::X509_Wrap::fingerprint_sha256() output buffer null pointer");
    out->reset();
    pointer_except();
    const EVP_MD *digest = EVP_sha256();
    out->newsize(32);
    vmp_uint len;
    if((X509_digest(x509_,digest,out->pointer(),&len) == 0) || (len != 32))
       vmp::except_s("openssl::pkg::X509_Wrap::fingerprint_sha1() error getting sha256 fingerprint");
}

}}}

