/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 08/12/2020
 */

#include "vmp.h"
#include <libxml++/libxml++.h>

#ifndef VAMPIRIA_XML_H

#define VAMPIRIA_XML_H 1

namespace vampiria { namespace xml {

//! Tag virtual class
/*! 
    Class from which the classes for the construction of objects tag
*/
class Tag
{
    private:
        vmp::str filepath_;/*!<Xml file path to read data tag*/
        vmp_uint line_;/*!< Line of tags in the xml file*/
        vmp::str name_; /*!< Tag Name*/
        vmp::str text_; /*!< Data type text Tag*/
        vmp::Table<vmp::str,vmp::str> attrs_;/*!<Hash table of objects attribute(attr_name,attr_value)*/
          
        Tag *parent_;/*!< Parent tag*/
        vmp::vector<Tag *> childs_; /*!< Child list tag*/  

        vmp_uint priority_;/*!Priority of the tag in the construction of the objects for the execution of Vampiria*/
                
        vmp_uint order_;/*!The number of tags that is read*/
    
        //!Init value data Tag
        void init();
    
        //!Function to be implemented in the derived class for call being destroyed. (not mandatory)
        virtual void destroy_impl();
    public:
        //! A constructor		
        Tag();
		
        //! A virtual destructor
        virtual ~Tag();

        //! Set xml file path to read data tag
        /*!
            @param filepath file path to setting
            @sa filepath_
        */
        void set_filepath(vmp::str filepath);
        
        //! Returns xml file path to read data tag
        /*!
            @return file path to read tag
            @sa filepath_
        */
	vmp::str get_filepath();

        //! Set line of tags in the xml file
	/*!
	    @param line line of tag to setting
	    @sa line_
	*/
	void set_line(vmp_uint line);
        
	//! Returns line of tags in the xml file
	/*!
	    @return line line of tag in the xml file
	    @sa line_
	*/
	vmp_uint get_line();

        //! Set tag name
	/*!
	    @param name tag name to setting
	    @sa name_
	*/
	void set_name(vmp::str name);
        
	//! Get tag name
	/*!
	    @return tag name 
	    @sa name_
	*/
	vmp::str get_name();

        //! Set text data
	/*!
	    @param text to setting
	    @sa text_
	*/
	void set_text(vmp::str text);
	
        //! Returns text data
	/*!
	     @return text data
	     @sa text_
	*/
	vmp::str get_text();

        //! Adds an attribute in tag
        /*!
            @param name attribute name
            @param value attribute value
            @sa attrs_                
        */                
        void add_attr(vmp::str name,vmp::str value);
		
        //! Returns an attibute value 
        /*!
            @param name attribute name
            @return the attribute value if it exists otherwise ""                        
            @sa attrs_
        */
        vmp::str get_attr(vmp::str name);
        
        //! Gets an attibute name list
        /*!
            @return the attribute name list                        
            \sa attrs_
        */
        vmp::vector<vmp::str> get_attrs_name();

        //! Set parent Tag
	/*!
	    @param parent parent tag
	    @sa parent_
	*/
	void set_parent(Tag *parent);
        
	//! Returns parent tag
	/*!
	    @return parent tag or 0 if root tag
	    @sa parent_
	*/
	Tag *get_parent();

        //! Add an child tag
	/*!
	     @param child child tag
	     @sa childs_
	*/
	void add_child(Tag *child);
        
	//!Returns the child to the index position
	/*!
	    @param index index position
	    @return the child to the index position or 0 if out range index
            @sa childs_	    
	*/
	Tag *get_child(vmp_index index);
        
	//! Returns the number of children in the tag
	/*!
	     @return children number
	     @sa childs_
	*/
	vmp_size get_child_number();

        //!Sets the tag's construction priority value. A higher value indicates that the tag is built first. If the priority values are the same, consider the order in which they are read. (See set_order ()). defaulf 0.
        /*!
            @param priority priority value to set
            @sa priority_
        */
        void set_priority(vmp_uint priority);

        //!Returns the priority of the tag
        /*!
            @return priority tag
            @sa priority_
        */
        vmp_uint get_priority();

        //!Sets the number of tags that is read
        /*!
            @param order value order to set
            @sa order_
        */
        void set_order(vmp_uint order);
       
        //! Returns the number order tag is read
        /*!
            @return order read value
            @sa order_
        */
        vmp_uint get_order();

        //!Throws an exception of parser error with the associated tag error
        /*!
              @param fmt format error string
              @param ... args list format string
        */
        void parser_error(const vmp_char *fmt,...);

        //!Throws an exception of parser error with the associated tag error
        /*!
            @param error error to except
        */
        void parser_error_s(vmp::str error);
        
        //! Destroy data Tag(call destroy_impl())		
        void destroy();

        //! Function to be implemented in the derived class for call during the construction of the tag. (not mandatory)
        virtual void build();

};

//!Tag base empty tag
class BaseTag:public Tag
{
    public:
        //!A Constructor
        BaseTag();

        //! A Destructor
        ~BaseTag();
};

//! Utility class to handle the priority tag in the construction of objects
class PriorityTag
{
    public:
        //! A constructor                
        PriorityTag();

        //! A destructor                
        ~PriorityTag();

        //! Operator to compare tag which has more or less of the other priorities                
        /*!
            @param t1 tag 1
            @param t2 tag 2
            @return true if t2 has higher priority than t1 otherwise false(hight numbers have higher priority)                
        */                
        vmp_bool operator()(xml::Tag *&t1,xml::Tag *&t2);
};

//! Xml Abstract Dom Parser
/*! Parsing xml file and builds data structures from the input file
*/
class DomParser
{
    private:
        vmp_uint gorder_;/*!Used internally to establish the reading order of the tags*/
        xml::Tag *root_;/*!<Root xml document tag*/
        vmp::priority_queue<Tag *,vmp::vector<Tag *>,xml::PriorityTag> ptag_;/*!<Priority tag build queue*/ 
        
        //!Recursive function used to build the tag tree
        /*!
            @param node library xml++ node
            @param parent parent tag
            @param xmlfile xml file processed
            @return root tag recursive
        */
        xml::Tag *evaluate(const xmlpp::Node *node,xml::Tag *parent,vmp::str filexml);
        
        //!Implemented in the derived class, it is used to create the tag
        /*!
            @param name tag name
            @return created tag
        */
        virtual xml::Tag *create_tag(vmp::str name)=0;
        
        //!Implemented in the derived class, it is used to free the memory allocated in the derived class
        virtual void destroy_impl()=0;
        
        //!Recursive function that serves to free up the memory of the tags. Call xml::tag->destroy()
        /*!
            @param tag input tag
        */
        void destroy_tag(xml::Tag *tag);
        
        //! Init data
        void init();
    public:
        //! A Constructor
        DomParser();

        //! A Destructor
        virtual ~DomParser();
        
        //!Returns root tag
        /*!
            @return root tag
            @sa root_
        */
        xml::Tag *get_root();
        
        //! Build the xml parser from xml file.Evaluate data(evaluate()) and run for each tag xml::tag->build() based on priority
        /*!
            @param filexml xml file path
            @param dtd external dtd path
        */
        void build(vmp::str filexml,vmp::str dtd);
        
        //! Destroy the xml parser structure call destroy_tag() and destroy_impl()
        void destroy();
};	

//!Utility for creating xml files, creating tags.
class WTag
{
    private:
        vmp::str name_; /*!< Name Tag*/
        vmp::str text_; /*!< Data type text Tag*/
        vmp::Table<vmp::str,vmp::str> attrs_;/*!<Hash table of objects attribute(attr_name,attr_value)*/  
        WTag *parent_;/*!< Parent tag*/
        vmp::vector<WTag *> childs_; /*!< Child list tag*/
        vmp_uint space_;/*!<Space left when writing the tag to the file */

    public:
        //! A Constructor
        /*!
            @param name tag name
            @param space space left when writing the tag to the file
        */
        WTag(vmp::str name,vmp_uint space);
        
        //! A Destructor
        ~WTag();

        //! Set text data tag
        /*!
            @param text text to set
            @sa text_
        */
        void set_text(vmp::str text);
        
        //! Adds an attribute hash table attr_
        /*!
            @param name attribute name
            @param value attribute value
            @sa attr_                
        */                
        void add_attr(vmp::str name,vmp::str value);
	
        //!Create a child tag and return it.
        /*!
            @param name child name
            @return child wtag
        */	
        WTag *add_child(vmp::str name);
        
        //! Set Parent tag
        /*!
            @param parent parent tag
            @sa parent_
        */
        void set_parent(WTag *parent);
        
        //! Returns parent tag
        /*!
            @return parent tag,0 if root tag
            @sa parent_
        */
        WTag *get_parent();

        //! Convert tag data to string format and add it to data input. Call recursively.
        /*!
            @param data data string
        */ 
        void write(vmp::str *data);
};

//!Utility for creating xml files. Parser data
class WParser
{
    private:
        xml::WTag *root_;/*!<Root Tag*/
        vmp::str rootname_;/*! root name xml tag*/
    public:
        //! A Constructor
        WParser();
        
        //! A Destructor
        ~WParser();

        //! Init write parser
        /*!
            @param rootname tag root name
            @return root tag created
            @sa root_
            @sa rootname_
       */
       xml::WTag *init(vmp::str rootname);
        
       //! Writes the parser data to a file
       /*!
           @param filepath file path
       */
       void write(vmp::str filepath);
};

}}

#endif

#include "xml/parser.h"


