/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 18/12/2019
 */

#include <map>

#ifndef VAMPIRIA_VMP_TABLE_H

#define VAMPIRIA_VMP_TABLE_H 1

namespace vampiria { namespace vmp {

//!Table type
template<class KEY,class DATA>
class Table
{
    private:
        typedef typename std::map<KEY,DATA >::iterator IteratorType;/*!< Iterator type*/
        std::map<KEY,DATA> data_;/*!< Map data*/
    public:
        //! A constructor
        Table(){}

        //! A destructor
        ~Table() {clear();}
       
        //!Table clear don't free memory pointer
        void clear(){data_.clear();}

        //!Table size
        /*!
             @return size data
             @sa data_        
        */
        vmp_size size() const {return data_.size();}

        //! Search a value in the table                
        /*!
             @param key key table
             @param pret data output
             @sa data_
             @return true ok,otherwise false
        */
        vmp_bool search(const KEY key,DATA *pret)
        {
            IteratorType it;
            it=data_.find(key);
            if(it != data_.end())
            {
                (*pret) = it->second;        
                return true;
            }
            return false;
        }

        //! Insert a value in the table, if the key already exists return except  
        /*!
             @param key key table
             @param data data value
             @sa data_
             @return void.(excpet error)
        */
        void insert(KEY key,DATA data)
        {
             DATA tmp;        
             if(!search(key,&tmp))
                 data_.insert(std::pair<KEY,DATA>(key,data));
             else
                 vmp::except_s("Table insert error key already");
        }

        //! Insert a value in the table. If the key already set data with old value otherwise set data pointer to 0. 
        /*!
             @param key key table
             @param pdata pointer data value
             @sa data_
            
        */
        void update(KEY key,DATA *pdata)
        {
             DATA tmp;        
             if(!search(key,&tmp))
             {
                 data_.insert(std::pair<KEY,DATA>(key,*pdata));
                 (*pdata)=0;
             }
             else
             {
                 DATA *tmp=pdata;
                 cancel(key,pdata);
                 data_.insert(std::pair<KEY,DATA>(key,(*tmp)));
             }
        }

        //! Cancel a value in the table                
        /*!
            @param key key table
            @param pdata data output
            @sa data_
            @return void except error
        */
        void cancel(const KEY key,DATA *pdata)
        {
            if(search(key,pdata))    
                data_.erase(key);
            else
                vmp::except_s("Table cancel error key not Found");    
        }

        //! Returns all table keys
        /*!
             @sa data_            
             @return table keys        
        */
        vmp::vector<KEY> all_keys()
        {
            vmp::vector<KEY> ret;
            IteratorType it;
            for(IteratorType it=data_.begin();it != data_.end();it++)
                ret.push_back(it->first);
            return ret;
        }

        //! Returns all data tables
        /*!
            @sa data_            
            @return table data        
        */
        vmp::vector<DATA> all_data()
        {
            vmp::vector<DATA> ret;
            IteratorType it;
            for(IteratorType it=data_.begin();it != data_.end();it++)
                ret.push_back(it->second);
            return ret;
        }

        //! Returns all pair (key,data) tables
        /*!
            @sa data_
            @return vector pair (key,data)
        */
        vmp::vector<vmp::pair<KEY,DATA>> all_pair_values()
        {
            vmp::vector<vmp::pair<KEY,DATA>> ret;
            IteratorType it;
            for(IteratorType it=data_.begin();it != data_.end();it++)
                ret.push_back(vmp::pair<KEY,DATA>(it->first,it->second));
            return ret;
        } 
};

//!Reset the table and free up data storage space. The DATA type must be a pointer and allocated with vmp::malloc_wrap
/*!
    @param table input table
*/
template<class KEY,class DATA>
void table_free_alldata(vmp::Table<KEY,DATA> table)
{
    vmp::vector<DATA> data=table.all_data();
    for(vmp_index i=0;i<data.size();i++)
        vmp::free_wrap((void **)&data[i]);
    table.clear();    
}

//!Reset the table and delete up data storage space. The DATA type must be a pointer and allocated with new
/*!
    @param table input table
*/
template<class KEY,class DATA>
void table_delete_alldata(vmp::Table<KEY,DATA> table)
{
    vmp::vector<DATA> data=table.all_data();
    for(vmp_index i=0;i<data.size();i++)
        delete data[i];
    table.clear();    
}

//! multi table type
template<class KEY,class DATA>
class MTable
{
     private:
         typedef typename std::multimap<KEY,DATA >::iterator IteratorType;/*!< Iterator type*/        
         std::multimap<KEY,DATA> data_;/*!< Multimap data*/
     public:
         //! A constructor                
         MTable() {}

         //! A destructor
         ~MTable() {clear();}

         //!Table clear don't free dinamic memory
         void clear() {data_.clear();}

         //!Table size
         /*!
              @return size data
              @sa data_        
         */
         vmp_size size() const {return data_.size();}
  
         //! insert a value in the table
         /*!
              @param key key table
              @param data data value
              \sa data_
         */                  
         void insert(KEY key,DATA data)
         {
              data_.insert(vmp::pair<KEY,DATA>(key,data));
         }

         //! search values in the table                
         /*!
              @param key key table
              \sa data_
              \return values 
         */                   
         vmp::vector<DATA> search(const KEY key)
         {
              vmp::vector<DATA> ret;    
              IteratorType it;
              vmp::pair<IteratorType,IteratorType> pit;      
              pit=data_.equal_range(key);
              for(it=pit.first;it != pit.second;++it)
                  ret.push_back(it->second);        
              return ret;
         }
         
         //! cancel values in the table                
         /*!
              @param key key table
              \sa data_
              \return values 
         */  
         vmp::vector<DATA> cancel(const KEY key)
         {
              vmp::vector<DATA> ret=search(key);        
              if(ret.size() != 0)    
                  data_.erase(key);
              return ret;
         }   

         //! returns all table keys
         /*!
              \sa data_            
              \return table keys        
         */ 
         vmp::vector<KEY> all_keys()
         {
              vmp::vector<KEY> ret;
              vmp_bool v;
              IteratorType it;
              for(IteratorType it=data_.begin();it != data_.end();it++)
              {
                   v=false;                
                   for(vmp_index i=0;i<ret.size() && !v;i++)
                        if(ret[i] == it->first)
                             v=true; 
                   if(!v)                
                        ret.push_back(it->first);
              }    
              return ret;
         }

         //! returns all data tables
         /*!
             \sa data_            
             \return table data        
         */  
         vmp::vector<DATA> all_data()
         {
              vmp::vector<DATA> ret;
              IteratorType it;
              for(IteratorType it=data_.begin();it != data_.end();it++)
                  ret.push_back(it->second);
              return ret;
         }    
};

//!Reset the multi table and delete up data storage space. The DATA type must be a pointer allocated with vmp::malloc_wrap
/*!
    @param mtable input multi table
*/
template<class KEY,class DATA>
void mtable_free_alldata(vmp::MTable<KEY,DATA> mtable)
{
    vmp::vector<DATA> data=mtable.all_data();
    for(vmp_index i=0;i<data.size();i++)
        vmp::free_wrap((void **)&data[i]);
    mtable.clear();    
}

//!Reset the multi table and delete up data storage space. The DATA type must be a pointer allocated with new
/*!
    @param mtable input multi table
*/
template<class KEY,class DATA>
void mtable_delete_alldata(vmp::MTable<KEY,DATA> mtable)
{
    vmp::vector<DATA> data=mtable.all_data();
    for(vmp_index i=0;i<data.size();i++)
        delete data[i];
    mtable.clear();    
}
}}

#endif

