/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 13/11/2020
 */

#ifndef VAMPIRIA_PACKET_DATATYPES_H

#define VAMPIRIA_PACKET_DATATYPES_H 1

namespace vampiria { namespace packet {

//!Abstract class for implementing packet data in byte order
class DataTypes
{
    public:
        //! A constructor 
        DataTypes();

        //! A destructor
	virtual ~DataTypes();

        //!Returns the fields used by the data
        /*!
            @return fields list 
        */
        virtual vmp::vector<vmp::str> fields()=0;
        
        //!Returns the data in string format associated with the field
        /*!
            @param field input field
            @return associated data(except error)
        */
        virtual vmp::str get(vmp::str field)=0;

        //!Set the data associated with the field(string format)
        /*!
            @param field input field
            @param data  input data
        */        
        virtual void set(vmp::str field,vmp::str data)=0;

        //!Read data from a buffer
        /*!
            @param buf input buf
        */
        virtual void read(vmp::Buf *buf)=0;

        //!Writes the data to a buffer
        /*!
            @param buf output buf
        */
        virtual void write(vmp::Buf *buf)=0;

        //!Returns a human-readable string associated with the data type
        /*!
            @return human-readable string
        */
        virtual vmp::str print()=0;
};

//!Abstract class for implementing packet data(single field in byte order)
class DataBytes:public packet::DataTypes 
{
    private:
        vmp::str field_;/*!<Data field*/
    public:
	//!A constructor
        /*!
            @param field data field
        */
        DataBytes(vmp::str field);

        //! A destructor
        virtual ~DataBytes();

        //! Virtual function.Returns only one item
        /*!
            @sa field_
        */
        vmp::vector<vmp::str> fields();

        //! Virtual function
        virtual vmp::str print();
};

//!Abstract class to implement data in bits format(see packet::Databits)
class Bitwise
{
    private:
        vmp::str field_;/*!<Data field*/
        vmp_byte init_;/*!<Initial bit where the data in DataBytes are found*/
        vmp_size size_;/*!<number of data bits*/
    public:

        //! A constructor
        /*!
            @param field data filed
            @param init initial bit where the data in DataBytes are found
            @param size number of data bits
        */
        Bitwise(vmp::str field,vmp_byte init,vmp_byte size);

        //!A destructor
        virtual ~Bitwise();

        //!Returns field
        vmp::str field();
        
        //! Returns initial bit where the data in DataBytes are found
        vmp_byte init();
        
        //! Returns number of data bits 
        vmp_byte size();

        //! Gets the data value
        /*!
            @param data data raw
            @return data value
        */
        virtual vmp::str get(vmp_size data)=0;
        
        //! Sets the data value
        /*!
            @param data data raw
            @return void or except to mistake
        */
        virtual vmp_size set(vmp::str data)=0;
        
        //!Returns a human-readable string associated with the data type
        /*!
            @param data data raw
            @return human-readable string
        */
        virtual vmp::str print(vmp_size data);
};

//!Base class in databytes to implement bitwise data(see packet::Bitwise)
class DataBits:public packet::DataTypes
{
    private:
        vmp_size data_;/*!<Data used for division into bits*/
        vmp_size datalen_;/*!<Number of bytes used by the data(values from 1 to 4)*/
        vmp::vector<packet::Bitwise *> bits_;/*!<List of implemented data in bit order(see packet::Bitwise)*/
        vmp::Table<vmp::str,vmp_index> search_;/*!<Internal usage*/

        //!Returns the data corresponding to the Bitwise in input
        /*!
            @param bits bitwise in input
            @sa data_
            @return data corresponding to the bitwise
        */
        vmp_size get_bits_value(packet::Bitwise *bits);

        //!Setting the data corresponding to the bitwise in input
        /*!
            @param bits bitwise in input
            @param data data to setting
            @sa data_
            @return void or except by mistake
        */
        void set_bits_value(packet::Bitwise *bits,vmp_size data);
    public:
	//! A constructor
        /*!
            @param datalen data length value(values from 1 to 4)
            @sa datalen_
        */
        DataBits(vmp_size datalen);

        //! A destructor
        ~DataBits();

        //! Insert Bitwise
        /*!
            @param bits bitwise to insert
            @sa bits_
            @sa search_
            @return void or except by mistake
        */
        void insert_data(packet::Bitwise *bits);

        //! Virtual function. Returns al Bitwise fields,
        /*!
            @sa bits_
        */
        vmp::vector<vmp::str> fields();

        //! Virtual function
        vmp::str get(vmp::str field);
        
        //! Virtual function
        void set(vmp::str field,vmp::str data);

        //! Virtual function
        void read(vmp::Buf *buf);

        //! Virtual function
        void write(vmp::Buf *buf);
        
        //! Virtual function
        vmp::str print();

        //! Get data in raw mode
        /*!
            @param field input field
            @return data
        */
        vmp_size get_data(vmp::str field);
        
        //! Set data in raw mode
        /*!
             @param field input field
             @param data data value
             @return void or except by mistake
        */
        void set_data(vmp::str field,vmp_size data);
};

}}

#endif

#include "packet/datatypes/address.h"
#include "packet/datatypes/bitwise.h"
#include "packet/datatypes/buffer.h"
#include "packet/datatypes/size.h"
#include "packet/datatypes/types.h"

