/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 *Date last update: 06/06/2020
 */

#include "net.h"

namespace vampiria { namespace net {

net::Socket socket_datagram(vmp_int domain,vmp_bool blocking)
{
    net::Socket sock;
    if((sock=socket(domain,SOCK_DGRAM,IPPROTO_UDP)) == -1)
        vmp::except_errno();
    if(!blocking)
        vmp::fd_noblock(sock);
    vmp_int sockopt = 1;
    if(setsockopt(sock,SOL_SOCKET,SO_REUSEADDR,(const void *)&sockopt,sizeof(sockopt)) == -1)
    {
        socket_close(&sock);
        vmp::except_errno();
    }
    return sock;
}

net::Socket socket_stream(vmp_int domain,vmp_bool blocking)
{
    net::Socket sock;
    if((sock=socket(domain,SOCK_STREAM,0)) == -1)
        vmp::except_errno();
    if(!blocking)
        vmp::fd_noblock(sock);
    return sock;
}

vmp_bool socket_connect(net::Socket sock,net::Address *remote)
{
    if(sock < 0)
        vmp::except_s("net::socket_connect invalid input socket");
    if(remote == 0 || remote->iptype() == PF_UNSPEC)
        vmp::except_s("net::socket_connect invalid input remote address");
    if(connect(sock,remote->addr(),remote->addrsize()) == -1)
    {    
        if(!((errno == EINTR) || (errno == EINPROGRESS)))
           vmp::except_errno();
        return false;
    }
    return true;
}

vmp_bool socket_connect_check(net::Socket sock)
{
    if(sock < 0)
        vmp::except_s("net::socket_connect_check invalid input socket");
    vmp_int e;
    vmp_uint se=sizeof(e);
    if(getsockopt(sock,SOL_SOCKET,SO_ERROR,(void*)&e,&se) == -1)
        vmp::except_errno();
    if(e) 
    {
        if((e == EINTR) || (e == EINPROGRESS))
           return false;
        else
           vmp::except_errno_set(e);
    }
    return true;
}

void socket_bind(net::Socket sock,net::Address *local)
{
    if(sock < 0)
        vmp::except_s("net::socket_bind invalid input socket");
    if(local == 0 || local->iptype() == PF_UNSPEC)
        vmp::except_s("net::socket_bind invalid input local address");
    if(bind(sock,local->addr(),local->addrsize()) == -1)
        vmp::except_errno();
}

void socket_listen(net::Socket sock,vmp_uint backlog)
{
    if(sock < 0)
        vmp::except_s("net::socket_listen invalid input socket");
    if(backlog == 0)
        vmp::except_s("net::socket_listen invalid backlog input");
    if(listen(sock,backlog) == -1)
        vmp::except_errno();
}

vmp_int socket_accept(net::Socket sock,vmp_bool blocking)
{
    vmp_int ret;
    if(sock < 0)
        vmp::except_s("net::socket_accept invalid input socket");
    ret=accept(sock,0,0);
    if(ret == -1)
    {
        if((errno == EAGAIN) || (errno == EWOULDBLOCK) || (errno == EINTR))
            return -1;
        vmp::except_errno();
    }
    if(!blocking)
        vmp::fd_noblock(ret);
    return ret;
}

void socket_addrlocal(net::Socket sock,net::Address *result)
{
    struct sockaddr_storage addr;
    socklen_t len=sizeof(addr);
    if(sock < 0)
        vmp::except_s("net::socket_addrlocal invalid input socket");   
    else if(result == 0)
         vmp::except_s("net::socket_addrlocal null pointer input address result");
    else if(getsockname(sock,(net::Sockaddr *)&addr,&len) == -1)
         vmp::except_errno();
    result->set((net::Sockaddr *)&addr,len);
}

void socket_addrpeer(net::Socket sock,net::Address *result)
{
    struct sockaddr_storage addr;
    socklen_t len=sizeof(addr);
    if(sock < 0)
        vmp::except_s("net::socket_addrpeer invalid input socket");
    else if(result == 0)
         vmp::except_s("net::socket_addrpeer null pointer input address result");
    else if(getpeername(sock,(net::Sockaddr *)&addr,&len) == -1)
         vmp::except_errno();
    result->set((net::Sockaddr *)&addr,len);
}

void socket_close(net::Socket *socket)
{
    if((*socket) != -1)
    {
         shutdown((*socket),SHUT_RDWR);
         vmp::fd_close(socket);
    }
}

}}

