/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 17/03/2020
 */

#include "vmp.h"
#include "net.h"
#include "xml.h"
#include "process.h"


#include "modules/modparser.h"
#include "modules/modinput.h"

#ifndef VAMPIRIA_MODULES_H

#define VAMPIRIA_MODULES_H 1

namespace vampiria { namespace modules {

const vmp::str dtdname_="module.dtd";/*<! Dtd file name in framework dtd directory*/

//!Utility to run an module
class Module
{
   private:
       vmp::str name_;/*!<Module name*/
   public:
       process::Pyexec *pyexec_;/*!<Python exec for module*/

       //! A Constuctor
       /*!
           @param name module name
           @param id id of the module process
           @param executable if true the module is executable,otherwise throw exceptions
           @param debug debug? of the module process
           @param envp environment of the module process
       */
       Module(vmp::str name,vmp::str id,vmp_bool executable,vmp_bool debug,vmp::args::Envp &envp);
       
       //! A Destructor
       ~Module();
       
       //! Returns Module name
       /*!
           @sa name_
           @return module name
       */
       vmp::str name();

       //! Run module process
       void run();
};

//!Utility to build a module not starting from parser xml
class ModData
{
    public:
       vmp::str name_;/*!Module name*/
       vmp_bool debug_;/*!debug? module process*/

       vmp::vector<modules::ModInput *> input_;/*!<Module input list*/
       //! A Constructor
       /*!
           @param name module name
           @param debug debug? module process
           @sa name_
           @sa debug_
       */ 
       ModData(vmp::str name,vmp_bool debug=false);
       
       //! A Destructor
       ~ModData();

       //!Add an input in module
       /*!
           @param name input name
           @param par parameters list
           @param var var data list
       */
       void add_input(vmp::str name,vmp::vector<vmp::str> par,vmp::vector<vmp::str> var);
};

//!Utility class for module management
class ModUtils
{
    private:
       vmp::Table<vmp::str,modules::ModInfo *> modinfo_;/*!<Stores the forms and information generated by the manifest parser*/
       
       //!Used by local functions to get information about forms
       /*!
          @param name module name to required info
          @sa modinfo_
          @return module info
       */
       modules::ModInfo *get_modinfo(vmp::str name);
    public:
       //! A Constructor
       ModUtils();

       //! A Destructor
       ~ModUtils();

       //! Reset Modutils data
       void reset();

       //!Build a module
       /*!
           @param mdata module data
           @param id id of the module process
           @param envp environment of the module process
           @return module structure
       */
       modules::Module *module_build(modules::ModData &mdata,vmp::str id,vmp::args::Envp &envp);

       //!Build a module from an xml tag called module.Used for reading data from an xml file
       /*!
           @param rmodule root tag <module>
           @param id id of the module process
           @param envp environment of the module process
           @return module structure
       */
       modules::Module *module_from_parser(xml::Tag *rmodule,vmp::str id,vmp::args::Envp &envp);
       
       //!Returns the topic string of the module list
       /*!
           @return topic modules list
       */
       vmp::str list();

       //!Returns the topic string of the module description
       /*!
           @param modname module name
           @return topic module description
       */
       vmp::str topic(vmp::str modname);
};

}}

#endif

