#!/usr/bin/python

#This program is free software; you can redistribute it and/or modify
#it under the terms of the GNU General Public License version 3 as
#published by the Free Software Foundation;

#This program is distributed in the hope that it will be useful,
#but WITHOUT ANY WARRANTY; without even the implied warranty of
#MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#GNU General Public License for more details.

#You should have received a copy of the GNU General Public License
#along with this program; if not, write to the Free Software
#Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#Author: Marco Guastella alias Vasta
#Web page:<www.ragnu.it>
#Email: <vasta@ragnu.it>
#Date last Update : 19/06/2020

import vmp
import vmp_xml  as xml
import vmp_gui0 as gui
import vmp_net as net

import config      as cfg
#import gvmp_utils  as utils
import gvmp_pycode as pycode

class GenericTag(xml.Tag):
  def __init__(self):
    xml.Tag.__init__(self)

class DescTag(xml.Tag):
  def __init__(self):
    xml.Tag.__init__(self)

  def build(self):
    self.desc=self.get_text()
    
  def build_mask(self,wmask):
    wdesc=gui.widget.Label(wmask,text=self.desc,bg=cfg.frame_bg_,maxline=cfg.mask_desc_size_)
    wmask.add_widget_w(wdesc,wmask.x,wmask.y,cspan=cfg.mask_grid_width_)
    wmask.row()
   
class VarTag(xml.Tag):
  def __init__(self):
    xml.Tag.__init__(self)
    self.var=''
    self.label=''
    self.widget=None
    self.data=[]
    
  def update_value(self):
    pass

  def get_data(self):
    return tuple(self.data) 

  def build_widget(self,wmask):
    pass

  def build_mask(self,wmask):
    self.var=self.get_attr("var")
    if self.var == '':
      self.parser_error("VarTag Tag requires attribute var")
    wmask.add_vartag(self)
    self.build_widget(wmask)
    if self.widget is not None:
      wlabel=gui.widget.Label(wmask,text=self.label,bg=cfg.frame_bg_,maxline=cfg.mask_label_size_)
      wmask.add_widget_w(wlabel,wmask.x,wmask.y,cspan=cfg.mask_label_width_)
      wmask.column(advance=cfg.mask_label_width_)
      wmask.add_widget_w(self.widget,wmask.x,wmask.y,cspan=cfg.mask_widget_width_)
      wmask.column(advance=cfg.mask_widget_width_)
      wvar=gui.widget.Label(wmask,text="${0}".format(self.var),bg=cfg.frame_bg_)
      wmask.add_widget_w(wvar,wmask.x,wmask.y,cspan=cfg.mask_var_width_)
      wmask.row()

  def var_raise(self,error):
    vmp.except_s("[${0}] {1}".format(self.var,error))
    
class DataTag(xml.Tag):
  def __init__(self):
    xml.Tag.__init__(self,priority=20)
    
  def build(self):
    parent=self.get_parent()
    parent.data.append(self.get_text())

class LabelTag(xml.Tag):
  def __init__(self):
    xml.Tag.__init__(self,priority=20)
    
  def build(self):
    parent=self.get_parent()
    parent.label=self.get_text()
  
class ChkButtonTag(VarTag):
  def __init__(self):
    VarTag.__init__(self)
    self.dselect=[]

  def update_value(self):
    if self.widget.get_value() == True:
      self.data=self.dselect
    else:
      self.data=[]

  def build_widget(self,wmask):
    active=self.get_attr("active")
    if active == "yes":
      self.widget=gui.widget.CheckButton(wmask,bg=cfg.frame_bg_,active=True)
    else:
      self.widget=gui.widget.CheckButton(wmask,bg=cfg.frame_bg_)

class DSelectTag(xml.Tag):
  def __init__(self):
    xml.Tag.__init__(self,priority=20)
    
  def build(self):
    parent=self.get_parent()
    parent.dselect.append(self.get_text())
    
class OptionsTag(VarTag):
  def __init__(self):
    VarTag.__init__(self)
    self.labels=[]
    self.optdata={}

  def update_value(self):
    self.data=self.optdata[self.widget.get_value()]

  def build_widget(self,wmask):
    self.widget=gui.widget.Options(wmask,tuple(self.labels))

class OptdataTag(xml.Tag):
  def __init__(self):
    xml.Tag.__init__(self,priority=10)
    self.label=""
    self.value=[]
  
  def build(self):
    parent=self.get_parent()
    if self.label in parent.optdata:
      self.parser_error("Duplicate label \"{0}\" in data options".format(self.label))
    parent.labels.append(self.label)
    parent.optdata[self.label]=self.value
    
class OptlabelTag(xml.Tag):
  def __init__(self):
    xml.Tag.__init__(self,priority=20)
    
  def build(self):
    parent=self.get_parent()
    parent.label=self.get_text()
     
class OptvalueTag(xml.Tag):
  def __init__(self,priority=20):
    xml.Tag.__init__(self)
    
  def build(self):
    parent=self.get_parent()
    parent.value.append(self.get_text())
     
class EntryAbsTag(VarTag):
  def __init__(self):
    VarTag.__init__(self)
    self.default=[]
    self.frm=[]
    self.size=None
    self.scroll=False
    self.fselect=False
    self.dselect=False
    self.rst=False
    self.mindata=cfg.widget_entry_mindata_
    self.maxdata=cfg.widget_entry_maxdata_

  def build(self):
    err=None
    try:
      self.size=vmp.unicode.str_todigit_range(self.get_attr("size"),cfg.widget_entry_minsize_,cfg.widget_entry_maxsize_)
    except:
      err="Attribute Size must be an integer between {0} and {1}".format(cfg.widget_entry_minsize_,cfg.widget_entry_maxsize_)  
    if self.get_attr("scroll") == 'yes':
      self.scroll=True
    if self.get_attr("fileselect") == 'yes':
      self.fselect=True
    if self.get_attr("dirselect") == 'yes':
      self.dselect=True
    if self.get_attr("rst") == 'yes':
      self.rst=True
    mindata=self.get_attr("mindata")
    maxdata=self.get_attr("maxdata")
    try:
      if not mindata == "": 
        self.mindata=vmp.unicode.str_todigit_range(mindata,cfg.widget_entry_mindata_,cfg.widget_entry_maxdata_)
      if not maxdata == "":
        self.maxdata=vmp.unicode.str_todigit_range(maxdata,self.mindata,cfg.widget_entry_maxdata_)
    except:
      err="Attributes mindata and maxdata must be two integers with 0 <= mindata({0}) <= maxdata({1}) <= {2}".format(self.mindata,self.maxdata,cfg.widget_entry_maxdata_)
    try:
      self.build_sub()
    except Exception as e:
      err=str(e)
    if err is not None:
      self.parser_error(err)  
  
  def build_sub(self):
    pass

  def check_data(self,data):
    data=data.split()
    ldata=len(data)
    if (ldata < self.mindata) or (ldata > self.maxdata):
      self.var_raise("The length of the data must be between {0}-{1} words".format(self.mindata,self.maxdata))
    i=0
    for f in self.frm:
      try:      
        f.check(data,i)
        i=i+1
      except Exception as e:
        self.var_raise(str(e))

class EntryTag(EntryAbsTag):
  def __init__(self):
    EntryAbsTag.__init__(self)
    self.value=''
    self.show=True

  def update_value(self):
    value=self.widget.get_value()
    self.check_data(value)
    self.data=[]
    if value is not '':
      self.data.append(value)

  def build_sub(self):
    if len(self.default) > 0:
      self.value=self.default[0]
    if self.get_attr("show") == "no":
      self.show=False
    
  def build_widget(self,wmask):
    self.widget=gui.widget.Entry(wmask,size=self.size,bg=cfg.widget_entry_bg_,
                                 fbg=cfg.frame_bg_,value=self.value,scroll=self.scroll,
                                 fselect=self.fselect,dselect=self.dselect,rst=self.rst,show=self.show)

class ListEntryTag(EntryAbsTag):
  def __init__(self):
    EntryAbsTag.__init__(self)
    self.paste=False

  def update_value(self):
    value=self.widget.get_value()
    self.data=[]
    for v in value:
      self.check_data(v)
      if value is not '':
        self.data.append(v)
 
  def build_sub(self):
    if self.get_attr("paste") == 'yes':
      self.paste=True

  def build_widget(self,wmask):
    self.widget=gui.widget.Text(wmask,sizew=self.size,sizeh=cfg.widget_listentry_height_,
                               bg=cfg.widget_entry_bg_,fbg=cfg.frame_bg_,value=self.default,scroll=self.scroll,
                               fselect=self.fselect,dselect=self.dselect,paste=self.paste,rst=self.rst)

class DefaultTag(xml.Tag):
  def __init__(self):
    xml.Tag.__init__(self,priority=20)
    
  def build(self):
    parent=self.get_parent()
    parent.default.append(self.get_text())

class FrmAbsTag(xml.Tag):
  def __init__(self):
    xml.Tag.__init__(self)
    self.index=0

  def build(self):
    err=None
    try:
      self.index=vmp.unicode.str_todigit_range(self.pytag.get_attr("index"),cfg.widget_entry_mindata_,cfg.widget_entry_maxsize_-1)
    except:
      err="Attribute index must be an integer between {0} and {1}".format(cfg.widget_entry_mindata_,cfg.widget_entry_maxsize_-1)
    try:
      self.build_sub()
    except Exception as e:
      err=str(e)
    if err is not None:
      self.parser_error(err)
    parent=self.get_parent()
    parent.frm.append(self)
  
  def check(self,data,line):
    if self.index < len(data):
      self.check_value(data[self.index],line)
        
  def build_sub(self):
    pass

  def check_value(self,sdata,line):
    pass
   
class FrmIntTag(FrmAbsTag):
  def __init__(self):
    FrmAbsTag.__init__(self)
    self.min=vmp.INTMIN
    self.max=vmp.INTMAX

  def build_sub(self):
    smin=self.get_attr("min")
    smax=self.get_attr("max")
    try:
      if not smin == "":
        self.min=vmp.unicode.str_todigit_range(smin,vmp.INTMIN,vmp.INTMAX) 
      if not smax == "":
        self.max=vmp.unicode.str_todigit_range(smax,self.min,vmp.INTMAX) 
    except:
      if not smax == "":
        vmp.except_s("Attributes min and max must be two integers with min({0}) <= max({1})".format(self.min,smax))  
      else:
        vmp.except_s("Attributes min and max must be two integers with min({0}) <= max({1})".format(self.min,self.max))

  def check_value(self,sdata,line):
    try:
      vmp.unicode.str_todigit_range(sdata,self.min,self.max)
    except:
      vmp.except_s("Value in position [{0},{1}] must be an integer between {2} and {3}".format(line,self.index,self.min,self.max))
      
class FrmRealTag(FrmAbsTag):
  def __init__(self):
    FrmAbsTag.__init__(self)
    self.min=float(vmp.INTMIN)
    self.max=float(vmp.INTMAX)

  def build_sub(self):
    smin=self.get_attr("min")
    smax=self.get_attr("max")
    try:
      if not smin == "":
        self.min=vmp.unicode.str_toreal_range(smin,float(vmp.INTMIN),float(vmp.INTMAX)) 
      if not smax == "":
        self.max=vmp.unicode.str_toreal_range(smax,self.min,float(vmp.INTMAX)) 
    except Exception as e:
      if not smax == "":
        vmp.except_s("Attributes min and max must be two reals with min({0}) <= max({1})".format(self.min,smax))  
      else:
        vmp.except_s("Attributes min and max must be two reals with min({0}) <= max({1})".format(self.min,self.max))
      
  def check_value(self,sdata,line):
    try:
      vmp.unicode.str_toreal_range(sdata,self.min,self.max)
    except Exception as e:
      vmp.except_s("Value in position [{0},{1}] must be an real between {2} and {3}".format(line,self.index,self.min,self.max))

class FrmHexTag(FrmAbsTag):
  def __init__(self):
    FrmAbsTag.__init__(self)

  def check_value(self,sdata,line):
    if not vmp.unicode.str_isxdigit(sdata):
       vmp.except_s("Value in position [{0},{1}] must be an hexadecimal".format(line,self.index))
    
class FrmOptTag(FrmAbsTag):
  def __init__(self):
    FrmAbsTag.__init__(self)
    self.data=[]

  def check_value(self,sdata,line):
    for d in self.data:
      if d == sdata:
        return
    vmp.except_s("Value in position [{0},{1}] must be an value in {2}".format(line,self.index,self.data)) 

class FrmMacTag(FrmAbsTag):
  def __init__(self):
    FrmAbsTag.__init__(self)

  def check_value(self,sdata,line):
    mac=net.MacAddress()
    try:
      mac.set(sdata)
    except:
      vmp.except_s("Value in position [{0},{1}] must be an Mac address value".format(line,self.index)) 

class FrmIpTag(FrmAbsTag):
  def __init__(self):
    FrmAbsTag.__init__(self)
    self.type=None

  def build_sub(self):
    self.type=self.get_attr("type")
    if self.type == '':
       self.type='all'

  def check_value(self,sdata,line):
    ip=net.Address()
    try:
      ip.set_ip_raw(sdata)
      if (self.type == "all" or self.type == '4') and ip.is_ipv4():
        return
      elif (self.type == "all" or self.type == '6') and ip.is_ipv6():
        return
    except Exception as e:
      pass
    vmp.except_s("Value in position [{0},{1}] must be an ip[{2}] address value ".format(line,self.index,self.type))

class FrmFileTag(FrmAbsTag):
  def __init__(self):
    FrmAbsTag.__init__(self)
    self.fwrite='no'
    self.fexec='no'

  def build_sub(self):
    tmp=self.get_attr("write")
    if tmp == 'yes':
      self.fwrite='yes'
    tmp=self.get_attr("exec")
    if tmp == 'yes':
      self.fexec='yes'

  def check_value(self,sdata,line):
    if not vmp.fs.isrfile(sdata):
      vmp.except_s("Value in position [{0},{1}] must be an file".format(line,self.index))
    if (self.fwrite == 'yes') and (not vmp.fs.iswfile(sdata)):
      vmp.except_s("Value in position [{0},{1}] must be an file in write mode".format(line,self.index))
    if (self.fexec == 'yes') and (not vmp.fs.isxfile(sdata)):
      vmp.except_s("Value in position [{0},{1}] must be an file in exec mode".format(line,self.index))

class FrmDirTag(FrmAbsTag):
  def __init__(self):
    FrmAbsTag.__init__(self)
    self.dwrite='no'
    self.dexec='no'

  def build_sub(self):
    tmp=self.get_attr("write")
    if tmp == 'yes':
      self.dwrite='yes'
    tmp=self.get_attr("exec")
    if tmp == 'yes':
      self.dexec='yes'

  def check_value(self,sdata,line):
    if not vmp.fs.isrdir(sdata):
      vmp.except_s("Value in position [{0},{1}] must be an dir".format(line,self.index))
    if (self.dwrite == 'yes') and (not vmp.fs.iswdir(sdata)):
      vmp.except_s("Value in position [{0},{1}] must be an dir in write mode".format(line,self.index))
    if (self.dexec == 'yes') and (not vmp.fs.isxdir(sdata)):
      vmp.except_s("Value in position [{0},{1}] must be an dir in exec mode".format(line,self.index))
 
class PyCodeTag(xml.Tag):
  def __init__(self):
    xml.Tag.__init__(self)
    self.var=None
    self.pyobj=None
    self.args=[]
    self.iargs=[]
    self.code=None
   
  def build(self):
    self.code=self.get_attr("code")
    self.pyobj=pycode.PyObjTable[self.code]
    self.var=self.pytag.get_attr("var")
    
  def build_mask(self,wmask):
    i=0
    if len(self.pyobj.args) == len(self.args):
      while i < len(self.pyobj.args):  
        atype=self.pyobj.args[i]
        if atype == 'var':
          if not pycode.data_isvar(self.args[i]):
            break
          self.iargs.append(self.args[i])
        elif atype == 'string': 
          if not pycode.data_isstatic(self.args[i]):
            break
          self.iargs.append(pycode.data_value(self.args[i]))
        elif atype == 'index':
          if not pycode.data_isstatic(self.args[i]):
            break
          try:
            self.iargs.append(vmp.unicode.str_todigit_range(pycode.data_value(self.args[i]),0,vmp.INTMAX-1))
          except:
            break    
        i=i+1
    if i == len(self.pyobj.args):
       wmask.pycode.append(self)
       return
    self.parser_error("Invalid arguments for code \"{0}\" required {1}".format(self.code,self.pyobj.args))
    
  def exec_code(self,var):
    i=0
    args=[]
    while i < len(self.pyobj.args):
      if self.pyobj.args[i] == 'var':
        args.append(pycode.data_dict_value(self.iargs[i],var))
      else:
        args.append(self.iargs[i])
      i=i+1
    return tuple(self.pyobj.exec_code(args))
    
class PyArgTag(xml.Tag):
  def __init__(self):
    xml.Tag.__init__(self)

  def build(self):
    parent=self.get_parent()
    try:
      parent.args.append(pycode.new_data_type(self.get_text()))
    except Execption as e:
      self.parser_error(e)
    
class RootTag(xml.Tag):
  def __init__(self):
    xml.Tag.__init__(self)
    
  def build_mask(self,wmask):
    wmask.root=self

  def parse(self,wparser,var):
    root=wparser.init(self.get_name())
    nc=self.get_child_number()
    for i in range(0,nc):
      c=self.get_child(i)
      c.parse(root,var)
      
class TreeTag(xml.Tag):
  def __init__(self):
    xml.Tag.__init__(self)
    self.if_=None
    self.ifnot_=None
    self.attrs=[]

  def build(self):
    tmp=self.get_attrs_name()
    for t in tmp:
      value=pycode.new_data_type(self.pytag.get_attr(t))
      if t == 'if_':
        if pycode.data_isvar(value):
          self.if_=value
        else:
          self.parser_error("Attribute if_ must be an var data")
      if t == 'ifnot_':
        if pycode.data_isvar(value):
          self.ifnot_=value
        else:
          self.parser_error("Attribute ifnot_ must be an var data")
      else:
        self.attrs.append((t,value))
       
  def if_check(self,var):
    if self.if_ is not None:
      v=pycode.data_dict_value(self.if_,var)
      if len(v) == 0:
        return False
    if self.ifnot_ is not None:
      v=pycode.data_dict_value(self.ifnot_,var)
      if not len(v) == 0:
        return False
    return True
  
  def add_attrs(self,tag,var):
    for attr in self.attrs:
      if pycode.data_isvar(attr[1]):
        value=pycode.data_dict_value(attr[1],var)
        if len(value) > 0 and len(value[0]) > 0:
          tag.add_attr(attr[0],value[0])  
      else:
        tag.add_attr(attr[0],pycode.data_value(attr[1]))
    

  def add_tag_text(self,ptag,text):
    tag=ptag.add_child(self.get_name())
    tag.set_text(text)    
    return tag

  def parse(self,ptag,var):
    if self.if_check(var):
      nc=self.get_child_number()
      if nc == 0:
        text=self.get_text()
        if text == "":
          tag=ptag.add_child(self.get_name())
          self.add_attrs(tag,var)
        else:
          vtext=pycode.new_data_type(text)
          if pycode.data_isstatic(vtext):
            tag=self.add_tag_text(ptag,text)
            self.add_attrs(tag,var)
          else:
            data=pycode.data_dict_value(vtext,var)
            for d in data:
              tag=self.add_tag_text(ptag,d)
              self.add_attrs(tag,var)
      else:
        tag=ptag.add_child(self.get_name())
        self.add_attrs(tag,var)
        for i in range(0,nc):
          c=self.get_child(i)
          c.parse(tag,var)
          
def parse(filepath,mask):
  parser=xml.Parser()
  parser.add_tag("gvampiria",GenericTag)
  parser.add_tag("description",DescTag)
  parser.add_tag("fixed",VarTag)
  parser.add_tag("checkbutton",ChkButtonTag)
  parser.add_tag("label",LabelTag)
  parser.add_tag("dataselect",DSelectTag)
  parser.add_tag("options",OptionsTag)
  parser.add_tag("optdata",OptdataTag)
  parser.add_tag("optlabel",OptlabelTag)
  parser.add_tag("optvalue",OptvalueTag)
  parser.add_tag("entry",EntryTag)
  parser.add_tag("default",DefaultTag)
  parser.add_tag("frm-int",FrmIntTag)
  parser.add_tag("frm-real",FrmRealTag)
  parser.add_tag("frm-hex",FrmHexTag)
  parser.add_tag("frm-opt",FrmOptTag)
  parser.add_tag("data",DataTag)
  parser.add_tag("frm-mac",FrmMacTag)
  parser.add_tag("frm-ip",FrmIpTag)
  parser.add_tag("frm-file",FrmFileTag)
  parser.add_tag("frm-dir",FrmDirTag)
  parser.add_tag("listentry",ListEntryTag)
  parser.add_tag("pycode",PyCodeTag)
  parser.add_tag("pyarg",PyArgTag)
  parser.add_tag("vampiria",RootTag)
  parser.add_tag("process",TreeTag)
  parser.add_tag("putenv",TreeTag)
  parser.add_tag("preload",TreeTag)
  parser.add_tag("import",TreeTag)
  parser.add_tag("module",TreeTag)
  parser.add_tag("input",TreeTag)
  parser.add_tag("par",TreeTag)
  parser.add_tag("var",TreeTag)
  parser.add_tag("stderr",TreeTag)
  parser.add_tag("stdout",TreeTag)
  parser.add_tag("fileout",TreeTag)
  parser.add_tag("processout",TreeTag)
  parser.add_tag("export",TreeTag)
  parser.add_tag("kill",TreeTag)
  try:
    parser.build(filepath,cfg.path_dtd_);
    root=parser.get_root()
    i=0
    tag=root.get_child(i)
    while tag is not None:
      tag.build_mask(mask)
      i=i+1
      tag=root.get_child(i)
  except Exception as e:
    parser.destroy()
    vmp.except_s(str(e))
  return parser

def write_parser(root,filepath,var):
  wparser=xml.WParser()
  root.parse(wparser,var)
  wparser.write(filepath)
  

