/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 * Date last update: 21/09/2020
 */

#include "openssl0/openssl0.h"

namespace vampiria { namespace openssl { namespace pkg {

SslUI::SslUI(event::Manager *manager):event::UI(manager)
{
     set_event_client(0,0,0);
     set_event_server(0,0,0,0);  
}

SslUI::~SslUI()
{
}

vmp::str SslUI::identity(event::Cell *cell)
{
    vmp::str ret="";
    openssl::pkg::EventSsl *ssl=cell->event<openssl::pkg::EventSsl>();
    vmp::str type=ssl->evt_ssl_type();
    net::Address *local=ssl->evt_ssl_localaddr();
    net::Address *peer;
    if(type == "client")
    {     
        peer=ssl->evt_ssl_peeraddr();
        vmp::unicode::str_write(&ret,"Ssl Client event: [%s:%s]<->[%s:%s]",local->host().c_str(),local->service().c_str(),peer->host().c_str(),peer->service().c_str());
    }
    else if(type == "listen")
        vmp::unicode::str_write(&ret,"Ssl Server Listen event: [%s:%s]",local->host().c_str(),local->service().c_str());
    else if(type == "server")
    {     
        peer=ssl->evt_ssl_peeraddr();
        vmp::unicode::str_write(&ret,"Ssl Server event:[%s:%s]<->[%s:%s]",local->host().c_str(),local->service().c_str(),peer->host().c_str(),peer->service().c_str());
    }
    return ret;
}
       
void SslUI::close_event(event::Cell *cell)
{
    openssl::pkg::EventSsl *ssl=cell->event<openssl::pkg::EventSsl>();
    ssl->evt_ssl_close();
}
       
void SslUI::free_ref(event::Cell *cell)
{
    openssl::pkg::EventSsl *ssl=cell->event<openssl::pkg::EventSsl>();
    ssl->evt_ssl_free();
    sref_.free(ssl);
}

event::Event *SslUI::child_event_new(event::Cell *cell)
{
    return (event::Event *) sref_.get();  
}

void SslUI::set_event_client(event::EVTCB connect,net::EVTCBRECV crecv,event::EVTCB cclose)
{
    connect_=connect;
    crecv_=crecv;
    cclose_=cclose;
}
 
void SslUI::set_event_server(net::EVTCBACCEPT acptevent,event::EVTCB svlcevent,net::EVTCBRECV svrevent,event::EVTCB svcevent)
{
    acptevent_=acptevent;
    svlcevent_=svlcevent;
    svrevent_=svrevent;
    svcevent_=svcevent;
}

event::Cell *SslUI::new_client(net::Address *raddress,openssl::pkg::Ctx *ctx)
{
    event::Cell *cell=0;
    manager_->lock();
    openssl::pkg::EventSsl *ssl=sref_.get();
    try
    {
        cell=ssl->evt_ssl_client(this,raddress,ctx,connect_,crecv_,cclose_);
    }
    catch(vmp::exception &x)
    {
        sref_.free(ssl);
        manager_->unlock();
        vmp::except("event::SslUI::new_client() '%s'",x.what());
    }
    manager_->unlock();
    return cell;
}

event::Cell *SslUI::new_server(net::Address *local,vmp_size backlog,openssl::pkg::Ctx *ctx)
{
    event::Cell *cell=0;
    manager_->lock();
    openssl::pkg::EventSsl *ssl=sref_.get();
    try
    {
        cell=ssl->evt_ssl_server(this,local,backlog,ctx,acptevent_,svlcevent_,svrevent_,svcevent_);
    }
    catch(vmp::exception &x)
    {
        sref_.free(ssl);
        manager_->unlock();
        vmp::except("event::SslUI::add_server() '%s'",x.what());
    }
    manager_->unlock();
    return cell;
}

void SslUI::send(event::Cell *cell,vmp::Buf *buf)
{
    openssl::pkg::EventSsl *ssl=cell->event<openssl::pkg::EventSsl>();
    event::Manager *manager=cell->get_manager();
    manager->lock();
    try
    {
        ssl->evt_ssl_send(buf);
    }
    catch(vmp::exception &x)
    {
        manager->unlock();
        vmp::except("event::SslUI::send() '%s'",x.what());
    }
    manager->unlock();
}

void SslUI::shutdown(event::Cell *cell)
{
     openssl::pkg::EventSsl *ssl=cell->event<openssl::pkg::EventSsl>();
     event::Manager *manager=cell->get_manager();
     manager->lock(); 
     ssl->evt_ssl_shutdown();
     manager->unlock();	
}

net::Address *SslUI::localaddr(event::Cell *cell)
{
    openssl::pkg::EventSsl *ssl=cell->event<openssl::pkg::EventSsl>();
    event::Manager *manager=cell->get_manager();
    manager->lock();
    net::Address *ret=ssl->evt_ssl_localaddr();
    manager->unlock();
    return ret;
}

net::Address *SslUI::peeraddr(event::Cell *cell)
{
    openssl::pkg::EventSsl *ssl=cell->event<openssl::pkg::EventSsl>();
    event::Manager *manager=cell->get_manager();
    manager->lock();
    net::Address *ret=ssl->evt_ssl_peeraddr();
    manager->unlock();
    return ret;
}

openssl::pkg::Ssl *SslUI::ssl(event::Cell *cell)
{
    openssl::pkg::EventSsl *ssl=cell->event<openssl::pkg::EventSsl>();
    event::Manager *manager=cell->get_manager();
    manager->lock();
    openssl::pkg::Ssl *ret=ssl->evt_ssl_ref();
    manager->unlock();
    return ret;
}

}}}

