/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 08/12/2020
*/
 
#include "xml.h"

namespace vampiria { namespace xml {

void dom_parser_error(vmp_uint line,vmp::str error)
{
    vmp::str err;
    vmp::unicode::str_write(&err,"xml::DomParser error line %u: %s",line,error.c_str());
    vmp::except_s(err);
}

void attr_integrity(vmp_uint line,vmp::str name,vmp::str value)
{
    vmp::str error;
    if(value.size() == 0)
    {    
        vmp::unicode::str_write(&error,"Attribute %s have empty value",name.c_str());
        dom_parser_error(line,error);
    }
    if(!vmp::unicode::str_isword(value))
    {    
        vmp::unicode::str_write(&error,"Attribute name [%s] have value malformated",name.c_str());
        dom_parser_error(line,error);
    }
}
 
Tag::Tag()
{
    init(); 
}

Tag::~Tag()
{
}

void Tag::init()
{
    filepath_="";
    line_=0;
    name_="";
    text_="";
    parent_=0;
    priority_=0;
    order_=0;
}

void Tag::destroy_impl()
{
}


void Tag::set_filepath(vmp::str filepath)
{
     filepath_=filepath;
}
        
vmp::str Tag::get_filepath()
{
    return filepath_;
}

void Tag::set_line(vmp_uint line)
{
    line_=line;
}
        
vmp_uint Tag::get_line()
{
    return line_;
}

void Tag::set_name(vmp::str name)
{
    name_=name;
}
        
vmp::str Tag::get_name()
{
    return name_;
}

void Tag::set_text(vmp::str text)
{
    text_=text;
}
        
vmp::str Tag::get_text()
{
    return text_;
}

void Tag::add_attr(vmp::str name,vmp::str value)
{
    attr_integrity(get_line(),name,value);
    try
    {
        attrs_.insert(name,value);    
    }
    catch(vmp::exception &x)
    {
	vmp::except("net::Tag::add_attr error name '%s' already",name.c_str());    
    }
}

vmp::str Tag::get_attr(vmp::str name)
{
    vmp::str ret;
    if(attrs_.search(name,&ret))
        return ret;
    return "";
}

vmp::vector<vmp::str> Tag::get_attrs_name()
{
    return attrs_.all_keys();
}

void Tag::set_parent(Tag *parent)
{
    parent_=parent;
}
        
Tag *Tag::get_parent()
{
    return parent_;
}

void Tag::add_child(Tag *child)
{
    childs_.push_back(child);
}
        
Tag *Tag::get_child(vmp_index index)
{
    if(index < childs_.size())
        return childs_[index];
    return 0;
}

vmp_size Tag::get_child_number()
{
    return childs_.size();
}

void Tag::set_priority(vmp_uint priority)
{
    priority_=priority;
}
        
vmp_uint Tag::get_priority()
{
   return priority_;
}

        
void Tag::set_order(vmp_uint order)
{
    order_=order;
}

vmp_uint Tag::get_order()
{
    return order_;
}

void Tag::parser_error(const vmp_char *fmt,...)
{
    va_list ap;
    va_start(ap,fmt);    
    vmp::str sfmt=vmp::va_wrap(fmt,ap);
    va_end(ap);
    parser_error_s(sfmt); 
}

void Tag::parser_error_s(vmp::str error)
{
    vmp::except("%s::%u tag <%s> parser error:\n%s",get_filepath().c_str(),get_line(),get_name().c_str(),error.c_str());
}

void Tag::destroy()
{
    destroy_impl();
    attrs_.clear();
    childs_.clear();
    init();
}

void Tag::build()
{
}

BaseTag::BaseTag():Tag()
{
}
        
BaseTag::~BaseTag()
{
}

PriorityTag::PriorityTag()
{
}

PriorityTag::~PriorityTag()
{
}

vmp_bool PriorityTag::operator()(xml::Tag *&t1,xml::Tag *&t2)
{
    vmp_uint t1p=t1->get_priority();
    vmp_uint t2p=t2->get_priority();
    if((t1p < t2p) || ((t1p == t2p) && (t1->get_order() > t2->get_order())))
        return true;
    return false;
}

DomParser::DomParser()
{
    init();
}

DomParser::~DomParser()
{
}

xml::Tag *DomParser::evaluate(const xmlpp::Node *node,xml::Tag *parent,vmp::str filexml)
{
    xml::Tag *ret=0;
    const xmlpp::ContentNode* nodeContent = dynamic_cast<const xmlpp::ContentNode*>(node);
    const xmlpp::TextNode* nodeText = dynamic_cast<const xmlpp::TextNode*>(node);
    const xmlpp::CommentNode* nodeComment = dynamic_cast<const xmlpp::CommentNode*>(node);

    if(nodeText && !(nodeText->is_white_space()) && parent != 0)
        parent->set_text(vmp::unicode::str_format(nodeText->get_content()));
    Glib::ustring nodename;
    if(!nodeText && !nodeComment && !nodeContent)
        nodename = node->get_name();
    if(!nodeText && !nodeComment && !nodename.empty() && !nodeContent)
    {
        ret=create_tag(nodename);
        ret->set_line(node->get_line());
        ret->set_name(nodename);
        ret->set_order(gorder_++);
        ret->set_filepath(filexml);
        if(parent != 0)
        {
            ret->set_parent(parent);
            parent->add_child(ret);
        }
        const xmlpp::Element* nodeElement = dynamic_cast<const xmlpp::Element*>(node);
        xmlpp::Element::AttributeList attrs=nodeElement->get_attributes();
        for (xmlpp::Element::AttributeList::iterator iter=attrs.begin(); iter !=attrs.end(); ++iter)
        {
             const xmlpp::Attribute* attr=(const xmlpp::Attribute*) (*iter);
             ret->add_attr(attr->get_name(),attr->get_value());
        }
        ptag_.push(ret); //code element
        xmlpp::Node::NodeList list = node->get_children();
        for(xmlpp::Node::NodeList::iterator iter = list.begin(); iter != list.end(); ++iter)
            evaluate(*iter,ret,filexml);
    } 
    return ret;
}

void DomParser::destroy_tag(xml::Tag *tag)
{
    vmp_index i=0;
    xml::Tag *tmp;
    if(tag == 0)
       return;
    while((tmp=tag->get_child(i++)) != 0)
       destroy_tag(tmp);
    tag->destroy();
    delete tag;
}

void DomParser::init()
{
    gorder_=0;
    root_=0;
}

xml::Tag *DomParser::get_root()
{
    return root_;
}

void DomParser::build(vmp::str filexml,vmp::str dtd)
{
    xmlpp::DomParser parser;
    xmlpp::Document *doc;
    xmlpp::DtdValidator validator;
    try
    {
        parser.set_throw_messages();
        validator.parse_file(dtd);
        parser.set_substitute_entities();
        parser.parse_file(filexml.c_str());
        if((doc=parser.get_document()) == 0)
            vmp::except_s("Null parser");
        validator.validate(doc);
    }
    catch(const vmp::exception &x)
    {        
        vmp::except("xml::DomParser::build() %s",x.what());
    }
    const xmlpp::Node *rnode=doc->get_root_node();
    root_=evaluate(rnode,0,filexml);
    xml::Tag *tmp;        
    while(!ptag_.empty())
    {
        tmp=ptag_.top();
        ptag_.pop();
        tmp->build();
    }	
}

void DomParser::destroy()
{
    while(!ptag_.empty())
        ptag_.pop();
    destroy_tag(root_);  
    destroy_impl();
    init();
}

WTag::WTag(vmp::str name,vmp_uint space)
{
    if(!vmp::unicode::str_isword(name))
        vmp::except("xml::WTag() invalid tag name %s",name.c_str());
    name_=name;
    text_="";
    space_=space;
    parent_=0;
}
        
WTag::~WTag()
{
    name_="";
    text_="";
    space_=0;
    attrs_.clear();
    for(vmp_index i=0;i<childs_.size();i++)
        delete childs_[i];
    childs_.clear(); 
}

void WTag::set_text(vmp::str text)
{
    text_=vmp::unicode::str_format(text);
}

void WTag::add_attr(vmp::str name,vmp::str value)
{
    if(!vmp::unicode::str_isword(name))
        vmp::except("xml::WTag::add_attr() invalid attr name %s",name.c_str());
    if(!vmp::unicode::str_isword(value))
        vmp::except("xml::WTag::add_attr() invalid attr value %s",value.c_str());
    try
    {
        attrs_.insert(name,value);
    }
    catch(vmp::exception &x)
    {
        vmp::except("xml::WTag::add_attr duplicate attribute name %s",name.c_str());
    }
}

WTag *WTag::add_child(vmp::str name)
{
    WTag *child=new WTag(name,space_+2);
    child->set_parent(this);
    childs_.push_back(child);
    return child;
}

void WTag::set_parent(WTag *parent)
{
    parent_=parent;
}

WTag *WTag::get_parent()
{
    return parent_;
}

void WTag::write(vmp::str *data)
{
    vmp::str attrvalue="";
    vmp::vector<vmp::pair<vmp::str,vmp::str>> values=attrs_.all_pair_values();
    for(vmp_index i=0;i<values.size();i++)
        vmp::unicode::str_cwrite(&attrvalue," %s=\"%s\"",values[i].first.c_str(),values[i].second.c_str());
    vmp::str spacedata="";
    for(vmp_index i=0;i<space_;i++)
        vmp::unicode::str_cwrite(&spacedata," ");
    if(childs_.size() == 0)
    {
        if(text_=="")
            vmp::unicode::str_cwrite(data,"%s<%s%s/>\n",spacedata.c_str(),name_.c_str(),attrvalue.c_str());
        else
            vmp::unicode::str_cwrite(data,"%s<%s%s>%s</%s>\n",spacedata.c_str(),name_.c_str(),attrvalue.c_str(),text_.c_str(),name_.c_str());    
    }
    else
    {
        vmp::str datachild="";
        for(vmp_index i=0;i<childs_.size();i++)
            childs_[i]->write(&datachild);
        vmp::unicode::str_cwrite(data,"%s<%s%s>\n%s%s</%s>\n",spacedata.c_str(),name_.c_str(),attrvalue.c_str(),datachild.c_str(),spacedata.c_str(),name_.c_str());
    }
}

WParser::WParser()
{
    root_=0;
}
        
WParser::~WParser()
{
    if(root_ != 0) 
       delete root_;
}

xml::WTag *WParser::init(vmp::str rootname)
{
    if(root_ != 0)
        vmp::except("xml::WParser duplex init");
    rootname_=rootname;
    root_ = new WTag(rootname_,0);
    return root_;
}

void WParser::write(vmp::str filepath)
{
    if(root_ == 0)
        vmp::except_s("xml::WParser no init");
    vmp::str data;
    vmp::unicode::str_write(&data,"%s\n",vmp::env::xml_hdr().c_str());
    root_->write(&data);
    vmp::output_file(filepath,true,"%s",data.c_str());
}

}}
 
