/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 01/01/2020
 */

#include "vmp.h"

namespace vampiria { namespace vmp { namespace unicode {

vmp_size str_write(vmp::str *pstr,const vmp_char *fmt,...)
{
    va_list ap;
    va_start(ap,fmt);    
    vmp::str sfmt=va_wrap(fmt,ap);
    va_end(ap);
    (*pstr)=sfmt;
    return (*pstr).size();
}

vmp_size str_cwrite(vmp::str *pstr,const vmp_char *fmt,...)
{
    va_list ap;
    va_start(ap,fmt);    
    vmp::str sfmt=va_wrap(fmt,ap);
    va_end(ap);
    (*pstr)+=sfmt;
    return (*pstr).size();
}

vmp::vector<vmp::str> str_split(vmp::str str,vmp::str delimiter)
{
    vmp::vector<vmp::str> ret;
    vmp::str tmp="";         
    vmp_index i=0;
    if(delimiter.size() == 0)
    {
        for(i=0;i<str.size();i++)
        {
            tmp=str[i];
            ret.push_back(tmp);
            tmp="";
        }
    }
    else
    {        
        while(i<str.size())
        {
            if(str.substr(i,delimiter.size()) == delimiter)
            {
                if(tmp.size() != 0)
                    ret.push_back(tmp);
                tmp="";                        
                i += delimiter.size();
            }    
            else
                tmp += str[i++];
        }
    }
    if(tmp.size() != 0)
        ret.push_back(tmp);
    return ret;
}
vmp::str str_join(vmp::vector<vmp::str> list,vmp::str token)
{
    vmp::str ret="";
    vmp_size s=list.size()-1;
    for(vmp_index i=0;i<s+1;i++)
    {
        ret += list[i];
        if(i != s)                
           ret += token;
    }
    return ret;
}

vmp_bool byte_isdigit(vmp_byte byte)
{
    if(byte >= '0' && byte <= '9')
        return true;    
    return false;
}

vmp_bool byte_isxdigit(vmp_byte byte)
{
    if(((byte >= '0') && (byte <= '9')) || ((byte >= 'a') && (byte <= 'f')) || ((byte >= 'A') && (byte <= 'F')))
        return true;
    return false;  
}

vmp_bool str_isdigit(vmp::str str)
{
    vmp_index ivalue=0;
    if(str.size() == 0)
        return false;
    if(str[0] == '-')
    {
        if(str.size() == 1)
            return false;
        else
            ivalue=1;
    }    
    for(vmp_index i=ivalue;i<str.size();i++)
        if(!vmp::unicode::byte_isdigit((vmp_byte)str[i]))
            return false;
    return true;
}

vmp_bool str_isxdigit(vmp::str str)
{
    if(str.size() == 0)
        return false;    
    for(vmp_index i=0;i<str.size();i++)
        if(!vmp::unicode::byte_isxdigit((vmp_byte)str[i]))
            return false;    
    return true;
}

vmp_bool str_isreal(vmp::str str)
{
    vmp::vector<vmp::str> numbers=vmp::unicode::str_split(str,".");
    if(numbers.size() == 1)
        return vmp::unicode::str_isdigit(numbers[0]);
    else if (numbers.size() == 2)
    {
        if(vmp::unicode::str_isdigit(numbers[0]))
            if(!((numbers[1].size() == 0) || (numbers[1][0] == '-')))
                return vmp::unicode::str_isdigit(numbers[1]);
    }  
    return false;
}

vmp_int str_todigit(vmp::str istr)
{
    if(!vmp::unicode::str_isdigit(istr))
        vmp::except("Invalid Conversion from string to int because '%s' not integer",istr.c_str());   
    return vmp::atoi_wrap(istr);
}

vmp_int str_todigit_range(vmp::str istr,vmp_int min,vmp_int max)
{
    if(min > max)
        vmp::except("Min value %d not less max value %d in input",min,max);
    vmp_int ret=vmp::unicode::str_todigit(istr);
    if((ret < min) || (ret > max))
        vmp::except("%s not integer in range [%d,%d]",istr.c_str(),min,max);
    return ret; 
}

vmp_real str_toreal(vmp::str dstr)
{
    vmp_real ret;
    if(!vmp::unicode::str_isreal(dstr))
        vmp::except("Invalid Conversion from string to int because '%s' not real",dstr.c_str());   
    ret=vmp::atof_wrap(dstr.c_str());
    return ret;   
}

vmp_real str_toreal_range(vmp::str dstr,vmp_real min,vmp_real max)
{
    if(min > max)
        vmp::except("Min value %f not less max value %f in input",min,max);
    vmp_real ret=vmp::unicode::str_toreal(dstr);
    if((ret < min) || (ret > max))
        vmp::except("%s not double in range [%f,%f]",dstr.c_str(),min,max);
    return ret; 
}

vmp::vector<vmp_byte> xstr_tobytes(vmp::str xstr)
{
    vmp::vector<vmp_byte> ret;
    vmp_byte b=0x00;
    vmp::str tmp;
    if((xstr.size() % 2) == 1)
        vmp::unicode::str_write(&tmp,"0%s",xstr.c_str());
    else
        tmp=xstr;
    for(vmp_index i=0;i<tmp.size();i++)
    {
        switch(tmp[i])
        {
            case '0': b|=0x00;break;
            case '1': b|=0x01;break;
            case '2': b|=0x02;break;
            case '3': b|=0x03;break;
            case '4': b|=0x04;break;
            case '5': b|=0x05;break;
            case '6': b|=0x06;break;
            case '7': b|=0x07;break;
            case '8': b|=0x08;break;
            case '9': b|=0x09;break;
            case 'a': b|=0x0A;break;
            case 'A': b|=0x0A;break;
            case 'b': b|=0x0B;break;
            case 'B': b|=0x0B;break;
            case 'c': b|=0x0C;break;
            case 'C': b|=0x0C;break;
            case 'd': b|=0x0D;break;
            case 'D': b|=0x0D;break;
            case 'e': b|=0x0E;break;
            case 'E': b|=0x0E;break;
            case 'f': b|=0x0F;break;
            case 'F': b|=0x0F;break;
            default:
                vmp::except("String '%s' is not hexadecimal",xstr.c_str());
       }
       if((i%2) == 0)
           b = b << 4;
       else
       {
           ret.push_back(b);
           b=0x00;
       }    
    }
    return ret;
}

vmp::str bytes_toxstr(vmp::vector<vmp_byte> bytes)
{
    vmp::str ret="";
    for(vmp_index i=0;i<bytes.size();i++)
       vmp::unicode::str_cwrite(&ret,"%02X",bytes[i]);
    return ret;
}

vmp::str bytes_toxstr_hm(vmp::vector<vmp_byte> bytes,vmp::str delimiter)
{
    vmp::str ret="";
    for(vmp_index i=0;i<bytes.size();i++)
    {
       if(i == 0)
           vmp::unicode::str_cwrite(&ret,"%02X",bytes[i]);
       else
           vmp::unicode::str_cwrite(&ret,"%s%02X",delimiter.c_str(),bytes[i]);
    }
    return ret;
}

vmp::str str_toxstr(vmp::str str,vmp::str delimiter)
{
    vmp::Buf buf;
    buf.write_str(str);
    buf.index();
    return buf.read_xstr_hm(buf.size(),delimiter);
}

vmp::str xstr_tostr(vmp::str xstr,vmp::str delimiter)
{
    vmp::str cstr;
    if(delimiter != "")
    {
        vmp::vector<vmp::str> split=vmp::unicode::str_split(xstr,delimiter);
        for(vmp_index i=0;i<split.size();i++)
        {
            if(split[i].size() == 1)
                vmp::unicode::str_write(&split[i],"0%s",split[i].c_str());
            else if(split[i].size() == 0)
                vmp::unicode::str_write(&split[i],"00");
            else if (split[i].size() > 2)
                vmp::except("String '%s' is not hexadecimal(delimiter = \'%s\')",xstr.c_str(),delimiter.c_str());
    
        }
        cstr=vmp::unicode::str_join(split,"");
    }
    else
        cstr=xstr;
    vmp::Buf buf;
    try
    {
        buf.write_xstr(cstr);
    }
    catch(vmp::exception &x)
    {
        vmp::except("String '%s' is not hexadecimal(delimiter = \'%s\')",xstr.c_str(),delimiter.c_str());
    }
    buf.index();
    return buf.read_str(buf.size());
}

vmp_bool str_findsub(vmp::str str,vmp::str sub)
{
    vmp_size found=str.find(sub);
    if(found == std::string::npos)
        return false;
    return true;
}

vmp_bool str_isword(vmp::str str)
{
    if(str.size() == 0)
        return false; 
    for(vmp_index i=0;i<str.size();i++)
        if((str[i] <= 0x20)||(str[i] >= 0x7E))
	   return false;
    return true;
}

vmp::str str_format(vmp::str content)
{
    vmp::str ret="";
    for(vmp_index i=0;i<content.size();i++)
    {
        if((content[i] <= 0x20) || (content[i] >= 0x7E))
        {    
             if((ret.size() != 0) && (ret[ret.size()-1] != 0x20))
                ret += " ";
        }
        else
             ret +=content[i];        
    }
    return ret;
}

vmp::str str_format_maxline(vmp::str content,vmp_size maxline)
{
    vmp::str ret="";
    content=vmp::unicode::str_format(content);
    vmp::vector<vmp::str> split=vmp::unicode::str_split(content," ");
    vmp_index line=0;        
    for(vmp_index i=0;i<split.size();i++)
    {
        if(line >= maxline)
        {
            ret += "\n";
            line=0;
        }
        else if(i != 0)
        {
            ret += " ";
            line++;                
        }                
        ret += split[i];
        line += split[i].size();        
    }
    return ret;
}

vmp::str str_format_end(vmp::str content)
{
    vmp_index i;
    for(i=content.size()-1;i>=0;i--)
        if((content[i] > 0x20) && (content[i] <= 0x7E))
            break;
    return content.substr(0,i+1);
}

vmp::str str_extract_char(vmp::str str,vmp::str clist)
{
    vmp::str ret="";
    for(vmp_index i=0;i<str.size();i++)
    {
        for(vmp_index j=0;j<clist.size();j++)
        {
            if(str[i] == clist[j])
            {
                ret+=clist[j];
                break;
            }
        }
    }
    return ret;
}

vmp::vector<vmp::str> str_regex_iterator(vmp::str str,std::regex regex)
{
    vmp::vector<vmp::str> ret;
    auto regex_begin=std::sregex_iterator(str.begin(),str.end(),regex);
    auto regex_end = std::sregex_iterator();
    for (std::sregex_iterator i = regex_begin; i != regex_end; ++i)
    {
        std::smatch match = *i;
        ret.push_back(match.str());
    }
    return ret;
}

vmp::vector<vmp::str> str_regex_posix_icase(vmp::str str,vmp::str str_regex)
{
    std::regex regex(str_regex,std::regex::basic|std::regex::icase);
    return str_regex_iterator(str,regex);
}

}}}

