/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 13/02/2020
 */

#ifndef VAMPIRIA_VMP_ARGS_H

#define VAMPIRIA_VMP_ARGS_H 1

namespace vampiria { namespace vmp { namespace args {

//!Callback performed when the argument parser is done in getopt
/*!
    @param argc   Total number of process input arguments
    @param argv   Arguments list
    @param optarg option arguments data(ex -c optarg).Null pointer if not optarg 
*/
typedef void (*GETOPTCB)(vmp_int argc,vmp_char *const argv[],vmp_char *optarg);

//!Parses the command-line arguments
class GetOpt
{
    private:
        vmp::Table<vmp_char,vmp::pair<GETOPTCB,vmp_bool>> opt_;/*!<Table (OPTCB,have opt argument).It is used to insert the values accepted by the parser*/
    public:
        //!A constructor
        GetOpt();

        //!A destructor
        ~GetOpt();
        
        //!Set an option arguments
        /*!
             @param opt argument(c for -c)
             @param cb Parser callback when argument is found in input(see vmp::args::GETOPTCB) 
        */
        void set_opt(vmp_char opt,vmp::args::GETOPTCB cb,vmp_bool optarg);
        
        //!Arguments parsing. Call after the sets
        /*!
             @param argc Total number of process input arguments
             @param argv Arguments list
             @return void(except to error)
        */
        void parse(vmp_int argc,vmp_char *const argv[]); 
};

//!Utility for managing process environment variables
class Envp
{
    private:
        vmp::Table<vmp::str,vmp::str> data_;/*!<Table (name,value) of environment variables ex.(LD_PRELOD,/usr/lib/lib.so)*/
        
        //!Integrity check of the complete string of an environment variable (ex LD_PRELOAD =/usr/lib/lib.so;)
        /*!
            @param value input value
            @return enviroment variable name(ex LD_PRELOAD).(Except to error)
        */
        vmp::str integrity(vmp::str value);
    public:
        //!A constructor
        Envp();

        //! A constructor, setting arguments list in envp
        /*!
            @param enviroments list(main envp arguments)
        */ 
        Envp(vmp_char *envp[]);

        //! A destructor        
        ~Envp();

        //! Operator =
        Envp& operator=(Envp env);

        //!Insert enviroment variable in data_
        /*!
            @param env enviroment string value(ex LD_PRELOAD =/usr/lib/lib.so;)
            @param replace if true substitutes the value of the variable in env(if exist), otherwise if it already finds the value it returns an exception
            @return enviroment variable name(except error)
            @sa data_ 
       */
        vmp::str insert_data(vmp::str env,vmp_bool replace);
        
        //!Returns the environment variable associated with the name
        /*!
            @param name enviroment variable name
            @return enviroment variable(except not found)
            @sa data_
        */
        vmp::str get_data(vmp::str name);
        
        //! Returns all enviroment variables
        /*!
            @return list enviroment variables
            @sa data_
        */
        vmp::vector<vmp::str> get_data();

        //!Add a local library to the environment variable LD_PRELOAD. (This library must be installed as a software or a link to an external library in VLIB path)
        /*!
            @param libname library name(no extensione .so)
            @return void(except error)
        */
        void preload_local_lib(vmp::str libname);
};

}}}

#endif

