/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 12/02/2020
 */

#ifndef VAMPIRIA_PROCESS_STDW_H

#define VAMPIRIA_PROCESS_STDW_H 1

namespace vampiria { namespace process {

//!Interface for creating factory objects (see StdWFactory)
class StdW
{
    private:
       vmp_index refcount_;/*!<References count std object*/
       vmp::str name_;/*!<Factory assigned name*/

       //!Virtual function to open a StdW resource
       virtual void open_impl();

       //!Virtual function to write data  to StdW resource
       /*!
           @param data written data
       */
       virtual void write_impl(vmp::str data);

       //!Virtual function to free a StdW resource
       virtual void close_impl();

    public:
       //!A constructor
       StdW();

       //! A virtual destructor
       virtual ~StdW();

       //!Returns std write
       /*!
            @return std object name
            @sa name_
       */
       vmp::str name();
       
       //!Returns references count
       /*!
           @return references count
           @sa refcount_
       */
       vmp_index refcount();

       //!Call in process::StdWFactory when creating object. If std write object has just been created and refcount_ is equal to zero, call the virtual function open_impl() and increase the refcount_, otherwise only increase the refcount_
       /*!
           @name std write object name
       */
       void open(vmp::str name);

       //!Writes the output data(Call virtual function write_impl())
       /*!
           @param data written data
       */
       void write(vmp::str data);

       //!Call in process::StdWFactory when creating object.Increment refcount_
       /*!
           @return refcount_
           @sa refcount_
       */      
       vmp_index incref();

       //!Call in process::StdWFactory::close.Decrement refcount_.
       /*!
           @return refcount_
           @sa refcount_
       */
       vmp_index decref();

       //!Call in process::StdWFactory::close.Free StdW resource(call virtual function free_impl())if refcount_ is equal to zero call 
       void close();
};

//!Implementation of a StdW for writing data to standard output
class StdWOut:public StdW
{
    private:
       //!Implementation virtual write_impl()
       void write_impl(vmp::str data);
    
    public:
       //!A constructor
       StdWOut();

       //!A destructor
       ~StdWOut();
};

//!Implementation of a StdW for writing data to standard output
class StdWErr:public StdW
{
    private:
       //!Implementation virtual write_impl()
       void write_impl(vmp::str data);
    public:
      //!A constructor 
      StdWErr();
     
      //!A destructor 
      ~StdWErr();
};

//!Implementation of a StdW for writing data to file
class StdWFile:public StdW
{
    private:
        //!Implementation virtual open_impl()
        void open_impl();
        
        //!Implementation virtual write_impl()
        void write_impl(vmp::str data);
    public:
        //!A constructor 
        StdWFile();
         
        //!A destructor
        ~StdWFile();

        vmp::str filepath_;/*!<File path to write data*/
};

//!Implementation of a StdW for writing data to process box(see process::Box)
class StdWBox:public StdW
{
    private:
        //!Implementation virtual write_impl()
        void write_impl(vmp::str data);
    public:
        //!A constructor 
        StdWBox();
        
        //!A destructor
        ~StdWBox();

        Box *box_;/*!Pointer to output process box*/
};

//! Factory method pattern for creating objects used to write the output and the error in the std, in the files and in the process boxes (see process :: Box)
class StdWFactory
{
    private:
       vmp::Table<vmp::str,StdW *> stdwref_;/*!<Table (name,Stdw object).It is used to index objects*/
    public:
       //!A constructor
       StdWFactory();

       //!A destructor
       ~StdWFactory();

       //!Return a reference to standard output.
       /*!
           @return standard output StdW
       */
       process::StdW *out();

       //!Return a reference to standard error.
       /*!
           @return standard error StdW
       */
       process::StdW *err();

       //!Return a reference to a file for writing.
       /*!
           @param filepath file path
           @return file StdW
       */
       process::StdW *file(vmp::str filepath);

       //!Return a reference to a process box.Use the box id as a key, if several boxes with the same id are inserted, consider the first one. The process box must be created before insertion and the memory must not be freed before the destruction of the StdWBox object(see process::Box)
       /*!
            @param box pointer to process box
            @return process box StdW
       */
       process::StdW *box(process::Box *box);

       //!Close a reference StdW.Frees memory if refcount equals zero and freeref equals true
       /*!
           @param stdw refernces to close
           @param freeref Frees the resource if refcount_ is zero?.(default true)
           @return refrences count
       */
       vmp_index close(process::StdW *stdw,vmp_bool freeref=true);       
};

}}

#endif

