/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 13/11/2020
 */

#include "packet.h"

namespace vampiria { namespace packet {

Packet::Packet(vmp::str type)
{
    type_=type;
    next_=0;
    prev_=0;
    subtype_=false;
    master_="";
}

Packet::~Packet()
{
    type_="";
    master_="";
    if(next_ != 0)
       delete next_;
    next_=0;
    prev_=0;
    for(vmp_index i=0;i<data_.size();i++)
        delete data_[i];
    data_.clear();
    search_.clear();
}

void Packet::set_subtype(vmp::str master)
{
    subtype_=true;
    master_=master;
}

vmp_bool Packet::subtype(vmp::str master)
{
    if(subtype_ && (master_ == master))
        return true;
    return false;
}

void Packet::read(vmp::Buf *buf)
{
    for(vmp_index i=0;i<data_.size();i++)
        data_[i]->read(buf);
}

void Packet::write(vmp::Buf *buf)
{
    for(vmp_index i=0;i<data_.size();i++)
        data_[i]->write(buf);
}

void Packet::pseudo_header(vmp::Buf *buf)
{
    vmp::except("packet::Packet() type='%s' it does not implement a pseudo header",type_.c_str());
}

void Packet::insert_data(packet::DataTypes *bytes)
{
    vmp::vector<vmp::str> fields=bytes->fields();
    data_.push_back(bytes);
    vmp_index index=data_.size()-1;
    for(vmp_index i=0;i<fields.size();i++)
    {
        try
        {
            search_.insert(fields[i],index);
        }
        catch(vmp::exception &x)
        {
            vmp::except("Fatal error insert data with duplicate labels '%s'",fields[i].c_str());
        }
    }
}

vmp::str Packet::get(vmp::str field)
{
    vmp_index index;
    if(!search_.search(field,&index))
    {
        if((next_ != 0) && next_->subtype(type_))
        {
            try
            {
                return next_->get(field);
            }
            catch(vmp::exception &x)
            {
            }
        } 
        vmp::except("packet::Packet::get() the '%s' package does not have a field called '%s'",type_.c_str(),field.c_str());
    }
    return data_[index]->get(field);
}

void Packet::set(vmp::str field,vmp::str data)
{
    vmp_index index;
    vmp::str type;
    if(subtype_)
        type=master_;
    else
        type=type_; 
    if(!search_.search(field,&index))
    {
        if((next_ != 0) && next_->subtype(type_))
        {
            try
            {
                 next_->set(field,data);
                 return;
            }
            catch(vmp::exception &x)
            {
                 vmp::except_s(x.what());
            }
        }
        vmp::except("packet::Packet::set() the '%s' package does not have a field called '%s'",type.c_str(),field.c_str());
    }
    try
    {
        data_[index]->set(field,data);
    }
    catch(vmp::exception &x)
    {
        vmp::except("packet::Packet::set() in packet '%s' check failed to insert data '%s' for field '%s'[%s]",
                    type.c_str(),data.c_str(),field.c_str(),x.what());
    }
}

vmp::str Packet::print()
{
    vmp::str ret;
    vmp::str next_t=next_type();
    if(!subtype_)
    {
        if(next_t != "")
            vmp::unicode::str_write(&ret,"----[%s (next_type = %s)]----",type_.c_str(),next_t.c_str());
        else
            vmp::unicode::str_write(&ret,"----[%s]----",type_.c_str());
    }
    else
    {
        if(next_t != "")
            vmp::unicode::str_write(&ret,"----[Subtype %s (next_type = %s)]----",master_.c_str(),next_t.c_str());
        else
            vmp::unicode::str_write(&ret,"----[Subtype %s]----",type_.c_str());
    }
    for(vmp_index i=0;i<data_.size();i++)
        vmp::unicode::str_cwrite(&ret,"\n%s",data_[i]->print().c_str());   
    return ret;
}

vmp::str Packet::next_type()
{
    return "";
}

Packet *Packet::next()
{
    return next_;
}

Packet *Packet::prev()
{
    return prev_;
}

Packet *Packet::operator/(Packet *next)
{
    packet::Packet *p=this;
    while(p->next() != 0)
        p=p->next();
    p->next_=next;
    next->prev_=p;
    return (Packet *)this;
}
       
Packet *Packet::operator[](vmp::str type)
{
    if(type_ == type)
        return (Packet *)this;
    else if(next_ != 0)
        return (*next_)[type];
    vmp::except("packet::Packet::operator[\"%s\"]:packet type not found",type.c_str());
    return 0;
}

packet::DataTypes *Packet::data(vmp_index i)
{
    vmp_size s=data_.size();
    if(i < s)
        return data_[i];
    if((next_ != 0) && next_->subtype(type_))
        return next_->data(i-s);
    return 0;
}

void Packet::payload(vmp::Buf *buf)
{
    packet::Packet *p=next();
    if((p != 0) && p->subtype(type_))
        p=p->next();
    while(p != 0)
    {
        p->write(buf);   
        p=p->next();
    }    
}

void packet_free(packet::Packet *packet)
{
    if (packet != 0)
        delete packet;
}

PacketHelper::PacketHelper()
{
    packdefault_=packet::Raw_D;
    
    //arp
    insert_packet(packet::P_ARP,packet::Arp_D);
    insert_packet(packet::arpsub_str_generic(1,0x0800),packet::Arp_Ethernet_Ipv4_S);
    
    //ethernet
    insert_packet(packet::P_ETHERNET,packet::Ethernet_D);

    // ipv4
    insert_packet(packet::P_IPV4,packet::Ipv4_D);
	
    // tcp
    insert_packet(packet::P_TCP,packet::Tcp_D);
	
    // udp
    insert_packet(packet::P_UDP,packet::Udp_D);
}
     
PacketHelper::~PacketHelper()
{
    packlist_.clear();
    packdefault_=0;
}

void PacketHelper::insert_packet(vmp::str type,packet::PACKBUILD builder)
{
    packlist_.update(type,&builder);
}

void PacketHelper::clone_type(vmp::str type,vmp::str clone)
{
    packet::PACKBUILD builder;
    if(!packlist_.search(type,&builder))
        vmp::except("packet::PacketHelper::clone_type() type '%s' not found",type.c_str());
    insert_packet(clone,builder);  
}

packet::Packet *PacketHelper::read(vmp::str basetype,vmp::Buf *buf)
{
    vmp::except_check_pointer((void *)buf,"packet::PacketHelper::read() null pointer input buf"); 
    buf->index();
    packet::PACKBUILD builder;
    vmp::str type=basetype;
    packet::Packet *ret=0,*prev,*current=0;
    while(!buf->eob())
    {
        if(!packlist_.search(type,&builder))
            builder=packdefault_;
        prev=current;
        current=builder();
        try
        {
            current->read(buf);
        }
        catch(vmp::exception &x)
        {
            if(ret != 0)
                packet_free(ret);
            packet_free(current);
            vmp::except("packet::PacketHelper::read() internal bugs");
        }
        if(ret == 0)
            ret=current;
        else
            (*prev)/current;
        type=current->next_type();
    }
    return ret;
}

void PacketHelper::write(packet::Packet *packet,vmp::Buf *buf)
{
    vmp::except_check_pointer((void *)buf,"packet::PacketHelper::write() null pointer input buf");
    buf->reset();
    packet::Packet *tmp=packet;
    while(tmp != 0)
    {
        tmp->write(buf);
        tmp=tmp->next();
    }
}

packet::Packet *PacketHelper::clone_hdr(packet::Packet *packet)
{
    vmp::except_check_pointer((void *)packet,"packet::PacketHelper::clone_hdr() packet input null");
    packet::PACKBUILD builder;
    if(!packlist_.search(packet->type_,&builder))
        builder=packdefault_;
    packet::Packet *ret=builder();
    vmp::Buf buf;
    packet->write(&buf);
    buf.index();
    ret->read(&buf);
    buf.reset();
    packet::Packet *pnext=packet->next();
    packet::Packet *next;
    if((pnext != 0) && pnext->subtype(packet->type_))
    {
       if(!packlist_.search(pnext->type_,&builder))
           builder=packdefault_;
       next=builder();
       pnext->write(&buf);
       buf.index();
       next->read(&buf);
       (*ret)/next;
       buf.reset();  
    }
    return ret;
}

vmp::str PacketHelper::print_all(packet::Packet *packet)
{
    vmp::except_check_pointer((void *)packet,"packet::PacketHelper::print_all() packet input null");
    vmp::str ret=packet->print();
    packet::Packet *p=packet->next();
    while(p != 0)
    {
        vmp::unicode::str_cwrite(&ret,"\n%s",p->print().c_str());
        p=p->next();
    }
    return ret;
}

packet::Packet *Arp_D()
{
    return (packet::Packet *) new packet::Arp_P();
}

vmp::str arpsub_str_generic(vmp_uint16 htype,vmp_uint16 ptype)
{
    vmp::str ret;
    vmp::unicode::str_write(&ret,"%s_%04X_%04X",packet::P_ARP.c_str(),htype,ptype);
    return ret;
}

packet::Packet *Arp_Ethernet_Ipv4_S()
{
    return (packet::Packet *) new packet::Arp_Ethernet_Ipv4_P();    
}

packet::Packet *Arp_Ethernet_Ipv4_D()
{
    packet::Packet *p=packet::Arp_D();
    packet::Packet *s=packet::Arp_Ethernet_Ipv4_S();
    try
    {
        p->set("htype",packet::P_ETHERNET);
        p->set("ptype",packet::P_IPV4);
        p->set("hlen","6");
        p->set("plen","4");
        (*p)/s;
    }
    catch(vmp::exception &x)
    {
        packet::packet_free(p);
        packet::packet_free(s);
        vmp::except_s(x.what());
    }
    return p;
}

packet::Packet *Arp_Ethernet_Ipv4(vmp::str oper,vmp::str sha,vmp::str spa,vmp::str tha,vmp::str tpa)
{
    packet::Packet *p=packet::Arp_Ethernet_Ipv4_D();
    try
    {
        p->set("oper",oper);
        p->set("sha",sha);
        p->set("spa",spa);
        p->set("tha",tha);
        p->set("tpa",tpa);
    }
    catch(vmp::exception &x)
    {
        packet::packet_free(p);
        vmp::except_s(x.what());
    }
    return p;
}

packet::Packet *Ethernet_D()
{
    return (packet::Packet *) new packet::Ethernet_P();
}

packet::Packet *Ethernet(vmp::str dstmac,vmp::str srcmac,vmp::str ethertype)
{
    packet::Packet *p=packet::Ethernet_D();
    try
    {
        p->set("dstmac",dstmac);
        p->set("srcmac",srcmac);
        p->set("ethertype",ethertype);
    }
    catch(vmp::exception &x)
    {
        packet::packet_free(p);
        vmp::except_s(x.what());
    }
    return p;
}

packet::Packet *Ipv4_D()
{
    return (packet::Packet *) new packet::Ipv4_P();
}

packet::Packet *Ipv4(vmp::str total_length,vmp::str identification,vmp::str ttl,vmp::str protocol,vmp::str source,vmp::str destination,vmp::str options)
{
    packet::Packet *p=packet::Ipv4_D();
    try
    {
        p->set("total_length",total_length);
        p->set("identification",identification);
        p->set("ttl",ttl);
        p->set("protocol",protocol);
        p->set("source",source);
        p->set("destination",destination);
        p->set("options",options);
    }
    catch(vmp::exception &x)
    {
        packet::packet_free(p);
        vmp::except_s(x.what());
    }
    return p;
}

vmp_bool Ipv4_integrity_check(packet::Packet *packet)
{
    vmp::except_check_pointer((void *)packet,"Ipv4_integrity_check() null pointer input packet");
    packet::Ipv4_P *ipv4=(packet::Ipv4_P *)(*packet)[packet::P_IPV4];
    return ipv4->integrity_check();
}

void Ipv4_adjust_check(packet::Packet *packet)
{
    vmp::except_check_pointer((void *)packet,"Ipv4_adjust_check() null pointer input packet");
    packet::Ipv4_P *ipv4=(packet::Ipv4_P *)(*packet)[packet::P_IPV4];
    packet::DataBuffer *options=(packet::DataBuffer *)ipv4->data(10);
    vmp::Buf *obuf=options->get_data();
    vmp::Buf pload;
    ipv4->payload(&pload);
    vmp_size len=20+obuf->size()+pload.size();
    vmp::str slen;
    vmp::unicode::str_write(&slen,"%u",len);
    ipv4->set("total_length",slen);
}

packet::Packet *Raw_D()
{
    return (packet::Packet *) new packet::Raw_P();
}

packet::Packet *Raw(vmp::str data)
{
    packet::Packet *p=Raw_D();
    try
    {
        p->set("data",data);
    }
    catch(vmp::exception &x)
    {
        packet::packet_free(p);
        vmp::except_s(x.what());
    }
    return p;
}

packet::Packet *Tcp_D()
{
    return (packet::Packet *) new packet::Tcp_P();
}

packet::Packet *Tcp(vmp::str srcport,vmp::str dstport,vmp::str sequence_n,vmp::str ack_n,vmp::str cwr,
                    vmp::str ece,vmp::str urg,vmp::str ack,vmp::str psh,vmp::str rst,vmp::str syn,vmp::str fin,
                    vmp::str window,vmp::str urgent_p,vmp::str options)
{
    packet::Packet *p=packet::Tcp_D();
    try
    {
        p->set("srcport",srcport);
        p->set("dstport",dstport);
        p->set("sequence_n",sequence_n);
        p->set("ack_n",ack_n);
        
        p->set("cwr",cwr);
        p->set("ece",ece);
        p->set("urg",urg);
        p->set("ack",ack);
        p->set("psh",psh);
        p->set("rst",rst);
        p->set("syn",syn);
        p->set("fin",fin);

        p->set("window",window);
        p->set("urgent_p",urgent_p);
        p->set("options",options);
    }
    catch(vmp::exception &x)
    {
        packet::packet_free(p);
        vmp::except_s(x.what());
    }
    return p;  
}

vmp_bool Tcp_integrity_check(packet::Packet *packet)
{
    vmp::except_check_pointer((void *)packet,"packet::Tcp_integrity_check() null pointer input packet");
    packet::Tcp_P *tcp=(packet::Tcp_P *)((*packet)[packet::P_TCP]);
    vmp::Buf pseudo;
    packet::Packet *prev=tcp->prev();
    if(prev != 0)
    {
        try
        {
            prev->pseudo_header(&pseudo);
            tcp->pseudo_header(&pseudo);
            tcp->payload(&pseudo);
            packet::DataHexSize *checksum=(packet::DataHexSize *) tcp->data(7);
            return (vmp::math::checksum16(&pseudo) == checksum->get_data());
        }
        catch(vmp::exception &x)
        {
        }
    }
    return false;
}

void Tcp_adjust_check(packet::Packet *packet)
{
    vmp::except_check_pointer((void *)packet,"packet::Tcp_adjust_check() null pointer input packet");
    packet::Tcp_P *tcp=(packet::Tcp_P *)((*packet)[packet::P_TCP]);
    vmp::Buf pseudo;
    packet::Packet *prev=tcp->prev();
    vmp::except_check_pointer((void *)prev,"packet::Tcp_adjust_check() previous tcp packet not found");
    try
    {
        prev->pseudo_header(&pseudo);
        tcp->pseudo_header(&pseudo);
	tcp->payload(&pseudo);
        packet::DataHexSize *checksum=(packet::DataHexSize *) tcp->data(7);
        checksum->set_data(vmp::math::checksum16(&pseudo));
    }
    catch(vmp::exception &x)
    {
        vmp::except_s(x.what());
    }
}

packet::Packet *Udp_D()
{
     return (packet::Packet *) new packet::Udp_P();
}

packet::Packet *Udp(vmp::str srcport,vmp::str dstport)
{
    packet::Packet *p=packet::Udp_D();
    try
    {
        p->set("srcport",srcport);
        p->set("dstport",dstport);
    }
    catch(vmp::exception &x)
    {
        packet::packet_free(p);
        vmp::except_s(x.what());
    }
    return p;  
}

vmp_bool Udp_integrity_check(packet::Packet *packet)
{
    vmp::except_check_pointer((void *)packet,"packet::Udp_integrity_check() null pointer input packet");
    packet::Udp_P *udp=(packet::Udp_P *)((*packet)[packet::P_UDP]);
    vmp::Buf pseudo;
    vmp::Buf payload;
    udp->payload(&payload);
    packet::DataSize *l=(packet::DataSize *) udp->data(2);
    packet::Packet *prev=udp->prev();
    if((prev != 0) && (l->get_data()) == (payload.size()+8))
    {
        try
        {
            prev->pseudo_header(&pseudo);
            udp->pseudo_header(&pseudo);
            pseudo.write_buf(&payload);
            packet::DataHexSize *checksum=(packet::DataHexSize *) udp->data(3);
            return (vmp::math::checksum16(&pseudo) == checksum->get_data());
        }
        catch(vmp::exception &x)
        {
        }
    }
    return false;
}

void Udp_adjust_check(packet::Packet *packet)
{
    vmp::except_check_pointer((void *)packet,"packet::Udp_adjust_check() null pointer input packet");
    packet::Udp_P *udp=(packet::Udp_P *)((*packet)[packet::P_UDP]);
    vmp::Buf payload;
    vmp::Buf pseudo;
    udp->payload(&payload);
    packet::Packet *prev=udp->prev();
    vmp::except_check_pointer((void *)prev,"packet::Udp_adjust_check() previous udp packet not found");
    packet::DataSize *l=(packet::DataSize *) udp->data(2);
    l->set_data(payload.size()+8);
    try
    {
        prev->pseudo_header(&pseudo);
        udp->pseudo_header(&pseudo);
        pseudo.write_buf(&payload);
        packet::DataHexSize *checksum=(packet::DataHexSize *) udp->data(3);
        checksum->set_data(vmp::math::checksum16(&pseudo));
    }
    catch(vmp::exception &x)
    {
        vmp::except_s(x.what());
    }
}

}}

