/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 17/03/2020
 */

#include "modules.h"

namespace vampiria { namespace modules {

Module::Module(vmp::str name,vmp::str id,vmp_bool executable,vmp_bool debug,vmp::args::Envp &envp)
{
    name_=name;
    if(executable)
    {
        vmp::str path=vmp::env::module_main_path(name);
        pyexec_=new process::Pyexec(path,id,debug,envp);
    }
    else
        vmp::except("Module \'%s\' is not executable",name.c_str());
}

Module::~Module()
{
    name_="";
    delete pyexec_;
}

void Module::run()
{
    pyexec_->run();
    
}

ModData::ModData(vmp::str name,vmp_bool debug)
{
    name_=name;
    debug_=debug;
}

ModData::~ModData()
{
    name_="";
    debug_="";
    vmp::vector_delete_alldata<modules::ModInput *>(input_);
}

void ModData::add_input(vmp::str name,vmp::vector<vmp::str> par,vmp::vector<vmp::str> var)
{
    modules::ModInput *modinput=new modules::ModInput(input_.size(),name,par,var);
    input_.push_back(modinput); 
}

ModUtils::ModUtils()
{
}

ModUtils::~ModUtils()
{
    reset();
}

void ModUtils::reset()
{
    vmp::table_delete_alldata<vmp::str,modules::ModInfo *>(modinfo_);
}

modules::ModInfo *ModUtils::get_modinfo(vmp::str name)
{
    modules::ModInfo *ret;
    if(!modinfo_.search(name,&ret))
    {
        try
        {
            if(!vmp::env::module_isinstalled(name))
                vmp::except("Modules name '%s' Not Installed",name.c_str());
            ret=new ModInfo(vmp::env::module_manifest(name));
        }
        catch(vmp::exception &x)
        {
            vmp::except_s(x.what());
        }
        modinfo_.insert(name,ret);
    }
    return ret;
}

void input_size_check_notnull(vmp::str modname,vmp_size isize,modules::ModInputTag *itag)
{
    if((isize == 0) && ((itag->required_ == "o") || itag->required_ == "op" ))
        vmp::except("Module '%s' required %s input with name [%s]",modname.c_str(),itag->required_format().c_str(),itag->name_.c_str());   
}

void input_size_check_notmulti(vmp::str modname,vmp_size isize,modules::ModInputTag *itag)
{
    if((isize > 1) && ((itag->required_ == "zo") || itag->required_ == "o" ))
        vmp::except("Module '%s' multiple input with name '%s'",modname.c_str(),itag->name_.c_str());    
}

modules::Module *ModUtils::module_build(modules::ModData &mdata,vmp::str id,vmp::args::Envp &envp)
{
    vmp::MTable<vmp::str,modules::ModInput *> input;
    for(vmp_index i=0;i<mdata.input_.size();i++)
        input.insert(mdata.input_[i]->name_,mdata.input_[i]);  
    modules::ModInfo *modinfo=get_modinfo(mdata.name_);
    modules::Module *ret=new Module(mdata.name_,id,modinfo->executable(),mdata.debug_,envp);
    modules::ModRootTag *root=modinfo->get_root();
    vmp::vector<modules::ModInput *> vinput;
    for(vmp_index i=0;i<root->input_.size();i++)
    {
        modules::ModInputTag *itag=root->input_[i];
        vinput=input.cancel(itag->name_);
        vmp_size isize=vinput.size();
        try
        {
            input_size_check_notnull(mdata.name_,isize,itag);
        }
        catch(vmp::exception &x)
        {
            vmp::except_s(x.what());
        }
        try
        {
            input_size_check_notmulti(mdata.name_,isize,itag);
        }
        catch(vmp::exception &x)
        {
            vmp::str lines="";
            for(vmp_index j=0;j<vinput.size();j++)
            {
                 if(j == (vinput.size()-1))
                     vmp::unicode::str_cwrite(&lines,"%u",j);
                 else
                     vmp::unicode::str_cwrite(&lines,"%u:",j);
            }
            vmp::except("%s in index '%s' (required %s)",x.what(),lines.c_str(),itag->required_format().c_str());
        }
        if(vinput[i]->par_.size() < itag->minpar_)
        {
            if(itag->minpar_ == itag->par_.size())
                vmp::except("Input '%s' required %u par",itag->name_.c_str(),itag->minpar_);
            else
                vmp::except("Input '%s' required between %u and %u par",itag->name_.c_str(),itag->minpar_,itag->par_.size());
        }
        for(vmp_index j=0;j<vinput[i]->par_.size();j++)
        {
            if(vinput[i]->par_[j] == "")
               vmp::except("Par '%u' in Input '%u' required text not empty",j,i);
            try
            {
               itag->par_[j]->check(vinput[i]->par_[j]);
            }
            catch(vmp::exception &x)
            {
               vmp::except("Par '%u' in Input '%u' text '%s' error [%s]",j,i,vinput[i]->par_[j].c_str(),x.what());
            }
            
        }
        if((vinput[i]->var_.size() > 0) && (!itag->var_))
           vmp::except("Input '%u' not accepted var data",i);
        for(vmp_index j=0;j<vinput[i]->var_.size();j++)
        {
            if(vinput[i]->var_[j] == "")
               vmp::except("Var '%u' in Input '%u' required text not empty",j,i);
        }
        ret->pyexec_->add_input(itag->name_,vinput[i]->par_,vinput[i]->var_);
    }
    vmp::vector<vmp::str> remain=input.all_keys();
    if(remain.size() != 0)
    {
        vinput=input.cancel(remain[0]);
        input.clear();
        vmp::except("Invalid Input '%u' name=\"%s\"' ",vinput[0]->index_,vinput[0]->name_.c_str());
    }
    vinput.clear();
    return ret;
}

modules::Module *ModUtils::module_from_parser(xml::Tag *rmodule,vmp::str id,vmp::args::Envp &envp)
{
    if(rmodule->get_name() != "module")
        rmodule->parser_error_s("Root Tag must have the '<module>' name");
    vmp::str name=rmodule->get_attr("name");
    if(name == "")
        rmodule->parser_error_s("Root Tag required 'name' attribute");
    vmp::str sdebug=rmodule->get_attr("debug");
    vmp_bool debug =false;
    if(sdebug != "")
    {
        if(sdebug == "no")
            debug=false;
        else if(sdebug == "yes")
            debug=true;
        else
            rmodule->parser_error_s("Root Tag 'debug' attribute must be 'yes' or 'no'");
    }
    vmp::MTable<vmp::str,xml::Tag *> input;
    vmp_index i=0;
    while(1)
    {
        xml::Tag *tag=rmodule->get_child(i);
        if(tag == 0)
           break;
        if(tag->get_name() != "input")
            tag->parser_error_s("Tag must have the 'input' name");
        vmp::str iname=tag->get_attr("name");
        if(iname == "")
            tag->parser_error_s("Tag required 'name' attribute");
        input.insert(iname,tag);
        i++;
    }
    modules::ModInfo *modinfo=get_modinfo(name);
    modules::Module *ret=new Module(name,id,modinfo->executable(),debug,envp);
    modules::ModRootTag *root=modinfo->get_root();
    vmp::vector<xml::Tag *> vinput;
    for(i=0;i<root->input_.size();i++)
    {
        modules::ModInputTag *itag=root->input_[i];
        vinput=input.cancel(itag->name_);
        vmp_size isize=vinput.size();
        try
        {
            input_size_check_notnull(name,isize,itag);
        }
        catch(vmp::exception &x)
        {
            rmodule->parser_error_s(x.what());
        }    
        try
        {
            input_size_check_notmulti(name,isize,itag);
        }
        catch(vmp::exception &x)
        {
            vmp::str lines="";
            for(vmp_index j=0;j<vinput.size();j++)
            {
                 if(j == (vinput.size()-1))
                     vmp::unicode::str_cwrite(&lines,"%u",vinput[j]->get_line());
                 else
                     vmp::unicode::str_cwrite(&lines,"%u:",vinput[j]->get_line());
            }
            rmodule->parser_error("%s in lines '%s' (required %s)",x.what(),lines.c_str(),itag->required_format().c_str());
        }
        for(vmp_index j=0;j<vinput.size();j++)
        {
            vmp::vector<vmp::str> par;
            vmp::vector<vmp::str> var;
            vmp_index z=0;
            while(1)
            {
                xml::Tag *arg=vinput[j]->get_child(z);
                if(arg == 0)
                   break;
                vmp::str text=arg->get_text();
                if(text == "")
                    arg->parser_error("Tag required text not empty");
                if(arg->get_name() == "par")
                {
                    if(par.size()+1 > itag->par_.size())
                    {
                        if(itag->minpar_ == itag->par_.size())
                            vinput[j]->parser_error("Input '%s' required %u par",itag->name_.c_str(),itag->minpar_);
                        else
                            vinput[j]->parser_error("Input '%s' required between %u and %u par",itag->name_.c_str(),itag->minpar_,itag->par_.size());
                    }
                    try
                    {
                        itag->par_[z]->check(text);
                    }
                    catch(vmp::exception &x)
                    {
                        arg->parser_error("Par text '%s' error [%s]",text.c_str(),x.what());
                    }
                    par.push_back(text);
                }
                else if(arg->get_name() == "var")
                {
                    if(!itag->var_)
                        vinput[j]->parser_error("Input '%s' not accepted var data in line '%u'",itag->name_.c_str(),arg->get_line());
                    else
                        var.push_back(text);
                }
                z++;
            }
            if(par.size() < itag->minpar_)
            {
                if(itag->minpar_ == itag->par_.size())
                    vinput[j]->parser_error("Input '%s' required %u par",itag->name_.c_str(),itag->minpar_);
                else
                    vinput[j]->parser_error("Input '%s' required between %u and %u par",itag->name_.c_str(),itag->minpar_,itag->par_.size());
            }
            ret->pyexec_->add_input(itag->name_,par,var);
        }    
    }
    vmp::vector<vmp::str> remain=input.all_keys();
    if(remain.size() != 0)
    {
        vinput=input.cancel(remain[0]);
        input.clear();
        vinput[0]->parser_error("Invalid Input 'name=\"%s\"' ",vinput[0]->get_attr("name").c_str());
    }
    vinput.clear();
    input.clear();
    return ret;
}

vmp::str ModUtils::list()
{
    vmp::str ret;
    vmp::vector<vmp::str> modules=vmp::fs::listendir(vmp::env::modules_root());
    vmp::unicode::str_write(&ret,"Modules list:");
    for(vmp_index i=0;i<modules.size();i++)
    {    
        if(!vmp::env::python_files_check(modules[i]))
        {
            try
            {
                if(vmp::env::module_isinstalled(modules[i]))
                    vmp::unicode::str_cwrite(&ret,"\n%s [Installed]",modules[i].c_str());
                else
                    vmp::unicode::str_cwrite(&ret,"\n%s [Not Installed]",modules[i].c_str());
            }
            catch(vmp::exception &x)
            {
                vmp::unicode::str_cwrite(&ret,"\n%s [Malformed]",modules[i].c_str());   
            }
        }
    }
    return ret;
}

vmp::str ModUtils::topic(vmp::str modname)
{
    vmp::str ret;
    modules::ModInfo *modinfo=get_modinfo(modname);
    modules::ModRootTag *root=modinfo->get_root();
    vmp::unicode::str_cwrite(&ret,"Module '%s' topic:\n\n%s\n\n",modname.c_str(),root->description_.c_str());
    if(modinfo->executable())
    {
        vmp::unicode::str_cwrite(&ret,"Xml syntax:\n\n<module name=\"%s\" debug=\"([no|yes] *implied default no*)\">\n",modname.c_str());
        for(vmp_index i=0;i<root->input_.size();i++)
        {
            modules::ModInputTag *itag=(ModInputTag *)root->input_[i];
            vmp::unicode::str_cwrite(&ret,"<!–\n%s\nRequired:%s\nminpar:%u\n–>\n<input name=\"%s\">\n",itag->description_.c_str(),itag->required_format().c_str(),itag->minpar_,itag->name_.c_str());
            for(vmp_index j=0;j<itag->par_.size();j++)
                vmp::unicode::str_cwrite(&ret,"<par>%s</par>\n",itag->par_[j]->description_.c_str());
            if(itag->var_)
                vmp::unicode::str_cwrite(&ret,"<var>%s(0...n)</var>\n",itag->vardesc_.c_str());
            vmp::unicode::str_cwrite(&ret,"</input>\n");
        }
        vmp::unicode::str_cwrite(&ret,"</module>");
    }
    else
        vmp::unicode::str_cwrite(&ret,"<Module not executable, only plugins are usable>");
    if(root->plugin_.size() > 0)
    {
        vmp::unicode::str_cwrite(&ret,"\n\nPlugin list:");
        for(vmp_index i=0;i<root->plugin_.size();i++)
        {
            ModPluginTag *tmp=root->plugin_[i];
            vmp::unicode::str_cwrite(&ret,"\n\nname='%s'\ntype='%s'\n%s",tmp->name_.c_str(),tmp->type_.c_str(),tmp->description_.c_str());
        }
    }
    vmp::unicode::str_cwrite(&ret,"\n");
    return ret;
}

}}

