/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 15/05/2020
 */

#include "vmp.h"

#include "event/lib/lib.h"

#ifndef VAMPIRIA_EVENT_H

#define VAMPIRIA_EVENT_H 1

namespace vampiria { namespace event {

//!Base class for creating User Interface
class UI
{
    protected:
        friend event::Cell;
        event::Manager *manager_;/*!<Manager associated with the ui interface*/
    public:
        //! A Constructor
        /*!
            @param manager Manager to associate with the interface
        */
        UI(event::Manager *manager);
        
        //! A Destructor
        virtual ~UI();

        //!Returns manager associated with ui interface
        /*!
            @sa manager_
            @return ui Manager associated with the ui interface
        */
        event::Manager *manager();

        //!Used by some user interfaces to create child events (ex tcp server where the child is the connection)
        /*!
            @param cell Event Cell to generate new event
            @return Cell child(default 0)
        */
        virtual event::Event *child_event_new(event::Cell *cell);

        //!Cell identity
        /*!
            @param cell input cell
            @return A string that identifies the cell 
        */
        virtual vmp::str identity(event::Cell *cell)=0;
        
        //!Called when an event is closed
        /*!
            @param cell input cell
        */
        virtual void close_event(event::Cell *cell)=0;
        
        //!Called when the memory of an event is freed
        /*!
            @param cell input cell
        */
        virtual void free_ref(event::Cell *cell)=0;
};

}}

#include "event/ui/breakloopui.h"

namespace vampiria { namespace event {

typedef enum cellstate {ACTIVE,CLOSEWAIT,CLOSE} CELLSTATE;
typedef enum cellret {NOCLOSE,SUCCESS,ERROR,TIMEOUT} CELLRET;

const vmp_int MAX_CELL_PRIORITY=999;/*!<Max cell priority*/
const vmp_int BREAKLOOP_PRIORITY=1000;/*!<Special event breakloop priority(It must be the second event executed)*/

//! Cell event reference.An event is associated with each cell
class Cell:public vmp::utils::Container<vmp::str>
{
    private:
       friend event::Manager;
       event::CELLSTATE state_;         /*!< Cell state*/
       vmp::time::Time timeout_;        /*!< Cell timeout*/
       vmp::time::Time timewait_;       /*!< Maximum time of inactivity allowed to the Cell*/
       vmp_index   nwait_;              /*!< Number of events executable in the state CLOSEWAIT*/
       event::CELLRET ret_;             /*!<Exit code*/
       vmp::str    err_;                /*!<Error info. With Exit code ERROR*/
       vmp_index   refalloc_;           /*!<References Cell allocated*/
      
       event::UI *ui_;                  /*!<User interface associated with event*/
       event::Event *event_;            /*!<Event handle*/
       event::Manager *manager_;        /*!<Associated manager*/
       vmp_int priority_;               /*!<Priority of the event execution*/
       
       //! Reset cell
       void reset();

     public:
       //! A Constructor
       Cell();

       //! A Destructor
       ~Cell();

       event::EVTCB read_;              /*!<Read callback*/
       event::EVTCB close_;             /*!<close callback*/
       
       vmp::str evtype_;                /*!<Used in UI for managing multiple event types(default generic)*/
       vmp_bool writing_;               /*!<True writing associated select fd,otherwise false(default false)*/

       //!Return a string that identifies the event(monitor)
       /*!
           @return string identity
           @sa ui_
       */
       vmp::str identity();
       
       //!Close event with state SUCCESS(monitor)
       void close();

       //!Assigns a number of events to the event to be executed before closing
       /*!
            @param n number of events executed before closing
       */
       void close_wait(vmp_uint n);
       
       //!Close event with state ERROR(monitor)
       /*!
           @param err error return string
       */
       void close_err(vmp::str err);
       
       //!Alloc cell references(monitor)
       /*!
           @sa refalloc_
       */
       void alloc();

       //!Release cell.(monitor)
       /*!
           @sa refalloc_
       */      
       void release();

       //!Return the closing value(monitor)
       /*!
           @return SUCCESS(Cell closed successfully),ERROR(Cell closed with error,call str_error for data err),TIMEOUT(Cell closed with timeout),NOCLOSE(cell not closed)
           @sa ret_
   
       */
       event::CELLRET ret();

       //!Returns error string(monitor)
       /*!
           @return error string
           @sa err_
       */
       vmp::str str_error();

       //! Set timewait cell,value 0.0 inactive timewait(monitor)
       /*!
           @param timewait input time wait
           \sa timewait_
       */
       void set_timewait(vmp::time::Time timewait=0.0);

       //! Get timewait cell(monitor)
       /*!
           @return timewait cell
       */
       vmp::time::Time get_timewait();

       //!Return manager associated with cell
       /*!
           @return manager
           @sa manager_
       */
       event::Manager *get_manager();

       //!Get user interface associated with cell
       /*!
           @return ui
           @sa ui_
       */
       template<class UI>
       UI *ui()
       {
           return (UI *) ui_;
       }

       //!Get Event associated with cell
       /*!
           @return event
           @sa event_
       */
       template<class EVENT>
       EVENT *event()
       {
           return (EVENT *) event_;
       }

       //! Operator to compare tag which has more or less of the other priorities                
       /*!
            @param c1 cell 1
            @param c2 cell 2
            @return true if c2 has higher priority than c1 otherwise false(hight numbers have higher priority)                
       */                
       vmp_bool operator()(Cell *&c1,Cell *&c2);
};

//! Event manager 
class Manager
{
    private:
       friend event::Cell;
       vmp::time::Timer timer_;/*!<Used to calculate the time elapsed since the manager was created*/
       vmp::time::Time timeout_;/*!<Timeout setting*/
       vmp::time::Time now_;/*!<Time now*/
       vmp::utils::Storage<Cell> cellref_;/*!<Storage utility for cell create*/
       vmp_bool loop_;/*!<variable used internally to verify that the loop is active*/
       vmp::priority_queue<event::Cell *,vmp::vector<event::Cell *>,event::Cell > pcell_;/*!<Priority cell build queue*/ 
       event::BreakloopUI *breakui_;/*!<Breakloop user interface*/
       vmp::thread::Mutex mutex_;/*!<variable mutual exclusion manager*/
       vmp::thread::Pthread self_;/*!<Thread id self*/
    public:
       //! A constructor
       Manager();

       //! A destructor
       ~Manager();

       //!Performs a break loop event to be able to reinitialize the event environment
       void breakloop();

       //!Lock manager
       void lock();

       //!unlock manager
       void unlock();

       //!Create new Cell
       /*!
           @param ui user interface associated
           @param event event associated
           @param priority execution priority of the associated event
           @param read read event callback
           @param close close event callback
           @return new cell
       */
       event::Cell *cell_new(event::UI *ui,event::Event *event,vmp_int priority,event::EVTCB read,event::EVTCB close);
       
       //!Update cell status.(call in read and write function)
       /*!
            @param cell cell update
            @return true if cell is active,otherwise false
       */
       vmp_bool cell_update(event::Cell *cell);

       //! Set timewait cell,value 0.0 inactive timewait
       /*!
           @param timewait input time wait
           @param cell cell to set timewait
       */
       void cell_timewait(event::Cell *cell,vmp::time::Time timewait);

       //!Internal use. Used for close event(no monitor)
       /*!
           @param cell cell to close
           @param retcode (see CELLRET)
           @param errinfo error info(id retcode==ERROR)
       */
       void cell_close(event::Cell *cell,event::CELLRET retcode,vmp::str errinfo="");
       
       //!Assigns a number of events to the event to be executed before closing
       /*!
           @param cell cell to close wait
           @param n number of events executed before closing
       */
       void cell_closewait(event::Cell *cell,vmp_uint n);
       
       //!Returns the current time (since the creation of the manager)
       /*!
           @return the current time
           @sa now_
       */
       vmp::time::Time time_now();

       //!Set timeout variable(Used in manager->loop_ and event timer)
       /*!
           @param timeout if timeout less than or equal to zero does nothing.if timeout_ is different from zero and timeout is less than timeout_ set the new timeout
       */
       void set_timeout(vmp::time::Time timeout);
       
       //!Alloc cell references
       /*!
           @param cell cell alloc     
       */
       void cell_alloc(event::Cell *cell);
       
       //!Release cell
       /*!
           @param cell to release
       */
       void cell_release(event::Cell *cell);

       //!Stop loop event manager(monitor)
       void stop();

       //!Loops for the execution of events(monitor used in loop)
       void loop();

};

}}

#endif

#include "event/lib/eventsimplex.h"
#include "event/lib/eventinput.h"
#include "event/lib/eventtimer.h"
#include "event/lib/eventthread.h"

#include "event/ui/simplexui.h"
#include "event/ui/inputui.h"
#include "event/ui/timerui.h"
#include "event/ui/threadui.h"



