#!/usr/bin/python

#This program is free software; you can redistribute it and/or modify
#it under the terms of the GNU General Public License version 3 as
#published by the Free Software Foundation;

#This program is distributed in the hope that it will be useful,
#but WITHOUT ANY WARRANTY; without even the implied warranty of
#MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#GNU General Public License for more details.

#You should have received a copy of the GNU General Public License
#along with this program; if not, write to the Free Software
#Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#Author: Marco Guastella alias Vasta
#Web page:<www.ragnu.it>
#Email: <vasta@ragnu.it>
#Date last Update : 18/12/2020

import vmp

maxdesc_=50

##Plugin topic base type
class PluginTopic:
  ## A Constructor<BR>
  #
  #@param ptype plugin type<BR>
  #@param modname module name<BR>
  #@param plugname plugin name
  def __init__(self,ptype,modname,plugname):
    self.ptype=ptype
    self.modname=modname
    self.plugname=plugname
    self.desc="no description"
    self.data=[]
    self.function=[]
    self.options=[]

  ##Set plugin description<BR>
  #
  #@param desc description plugin
  def set_desc(self,desc):
    self.desc=desc
    
  ##Adds the description of a plugin data.The data is the fixed parameters used by all plugins.They are inserted in the description in order of insertion<BR>
  #
  #@param datadesc decription data
  def add_data(self,datadesc):
    self.data.append(datadesc)

  ##Set index description of a plugin data.Only sets existing data does not add any data<BR>
  #
  #@param index index data<BR>
  #@param datadesc decription data
  def set_data(self,index,datadesc):
    if index < len(self.data):
      self.data[index]=datadesc

  ##Adds the description of a plugin function<BR>
  #
  #@param fname function name<BR>
  #@param fdesc function description
  def add_function(self,fname,fdesc):
    self.function.append([fname,fdesc])

  ##Sets the description of a plugin function.If the name has not already been entered, no new function is inserted
  #
  #@param fname function name<BR>
  #@param fdesc function description
  def set_function(self,fname,fdesc):
    for i in range(0,len(self.function)):
      if self.function[i][0] == fname:
        self.function[i][1]=fdesc

  ##Adds the description of optional data. Optional data can be different between plugins of the same type and have the form "data_name value"
  #
  #@param optname option name<BR>
  #@param optdef option default data<BR>
  #@param optocc number of occurrences of the option used (1, all ..)<BR>
  #@param optdesc option description
  def add_option(self,optname,optdef,optocc,optdesc):
    self.options.append((optname,optdef,optocc,optdesc,))    

  ##Returns the topic of the plugin in print format
  def print_topic(self):
    ret="<Plugin module=\'{0}\' name=\'{1}\' type=\'{2}\'>\n{3}".format(self.modname,self.plugname,self.ptype,vmp.unicode.str_format_maxline(self.desc,maxdesc_))
    for i in range(0,len(self.data)):
      ret += "\n<Data \'{0}\'>\n{1}".format(i,vmp.unicode.str_format_maxline(self.data[i],maxdesc_))
    for f in self.function:
      ret += "\n<Function \'{0}\'>\n{1}".format(f[0],vmp.unicode.str_format_maxline(f[1],maxdesc_))
    for o in self.options:
      ret += "\n<Option format=\'{0} $VALUE\' default_value=\'{1}\' occurrences_used=\'{2}\'>\n{3}".format(o[0],o[1],o[2],o[3])
    return ret

##Plugin Base type
class Plugin:
  ## A Constructor<BR>
  #
  #@param ptype plugin type<BR>
  #@param data input data<BR>
  #@param datalen number of fixed plugin data. It is used to check the data inputs<BR>
  #@param options input options data
  def __init__(self,ptype,data,datalen,options):
    self.ptype=ptype
    self.data=data
    self.options={}
    if not len(self.data) == datalen:
      vmp.except_s("Invalid number of input parameters [datalen={0}]".format(datalen))
    for o in options:
      opt=vmp.unicode.str_split(o,' ')
      l=len(opt)
      if l > 0:
        name=opt[0]
        if(l > 1):
          code=vmp.unicode.str_join(opt[1:],' ')
        else:
          code=''
        if name in self.options:
          self.options[name].append(code)
        else: 
          self.options[name]=[code]
        
  ##Throw exception of the error function not implemented
  def not_implemented(self,funcname):
    vmp.except_s("In Plugin \'{0}\' virtual function \'{1}\' not implemented".format(self.ptype,funcname))

  ##Throw exception of the error plugin data format
  def opt_err(self,data):
    vmp.except_s("Data {0} bad value".format(opt))
  
  ##Search for an option with input name and return the list of values if it finds it otherwise an empty list<BR>
  #
  #@param name option name
  #@return list value 
  def opt_list(self,name):
    if name in self.options:
      return self.options[name]
    return []

  ##Search for an option using name and index, if it finds it it returns the value otherwise it returns the default value in input<BR>
  #
  #@param name option name<BR>
  #@param index index value<BR>
  #@param defvalue default value<BR>
  #@return option value or None
  def opt_get(self,name,index,defvalue):
    v=self.opt_list(name)
    if index < len(v):
      return v[index]
    return defvalue
    
  
  ##Throw exception of the error option data format
  def opt_err(self,opt):
    vmp.except_s("Option {0} bad value".format(opt))

##Class for a module, contains the structures to manage a series of plugins
class Module:
  ## A Constructor<BR>
  #
  #@param module name<BR>
  #@param plugins A list of plugin handles. A plugin consists of a file with a PluginName string field, a Plugin class (derived from a plugin type) and a PluginTopic class (derived from a plugin topic).
  def __init__(self,modname,plugins):
    try:
      if not vmp.env.module_isinstalled(modname):
        vmp.except_s("Not Installed")
    except Exception as e:
      vmp.except_s(str(e))
    self.plugins={}
    self.types={};
    self.modname=modname
    for p in plugins:
      topic=p.PluginTopic(self.modname,p)
      self.plugins[p.PluginName]=(p,topic.ptype)
      self.types[topic.ptype]=""

  ##Returns the list of plugin types inserted in the module<BR>
  #
  #@return string of types list
  def types_list(self):
    ret=[]
    for t in self.types:
      ret.append(t)
    return ret  

  ##Returns the list of ptype type plugins inserted in the module<BR>
  #
  #@param ptype plugin type<BR>
  #@return string list of plugins
  def plugin_list(self,ptype):
    ret=[]
    for p in self.plugins:
      tmp=self.plugins[p]
      if tmp[1] == ptype:
        ret.append(p)
    return ret

  def plugin(self,name):
    if name in self.plugins:
      return self.plugins[name][0]
    vmp.except_s("not found")

  def print_topic(self):
    ret=""
    types=self.types_list()
    for t in types:
      l=self.plugin_list(t)
      ret = "{0}\n<PluginList type='{1}'>{2}</PluginList>".format(ret,t,l)
    return "<Module name=\'{0}\'>{1}\n</Module>".format(self.modname,ret);


