/* -*- Mode:C; c-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella <vasta@ragnu.it>
 */
 
#include "../core.h"

/*internal*/

const l_str l_str_error(l_int32 err)
{
	return (const l_str) strerror(err);	
}

const  l_str l_str_gaierror(l_int32 err)
{
	return (const l_str) gai_strerror(err);
}

void l_str_alloc(l_str *pstr,l_size len)
{
	l_size l=len+1;
	if((*pstr) != 0)
		l_str_free(pstr);
	(*pstr)=l_mem_malloc0(l);
}

l_bool l_str_isdigit(const l_str str)
{
	l_size len=l_str_len(str),i;
	if(len == 0)
		return FALSE;
	for(i=0;i<len;i++)
		if(! l_digit_is(str[i]))
			return FALSE;
	return TRUE;
}

l_bool l_str_isxdigit(const l_str str)
{
	l_size len=l_str_len(str),i;
	if(len == 0)
		return FALSE;
	for(i=0;i<len;i++)
	{
		for(i=0;i<len;i++)
			if(! l_digit_isx(str[i]))
				return FALSE;
	}
	return TRUE;
}

l_bool l_str_toirange(const l_str str,l_int32 low,l_int32 up,l_int32 *ret)
{
	if(l_str_toi(str,ret) && (*ret) >= low && (*ret) <=up)
		return TRUE;
	(*ret)=0;
	return FALSE;		
}

l_bool l_str_tox(const l_str str,l_int32 *ret)
{
	l_size len,i;
	if(! l_str_isxdigit(str))
		return FALSE;
	len=l_str_len(str);
	(*ret)=0;
	for(i=0;i<len;i++)
	{
		(*ret) = (*ret) << 4;
		(*ret) += l_digit_asciix(str[i]);
	}
	return TRUE;	
}

/*end internal*/

/*extern*/

l_str l_str_new()
{
	return (l_str) l_mem_null();	
}

void l_str_free(l_str *pstr)
{
	l_mem_freezero((void **)pstr,l_str_len((*pstr)));
}

l_size l_str_len(const l_str str)
{
	if(str==0)
		return 0;
	return strlen(str);	
}

l_byte l_str_lentobyte(l_str str)
{
	l_size l=l_str_len(str);
	return (l_byte) (l & 0xFF);	
}

l_int32 l_str_cmp(const l_str s1,const l_str s2)
{
	return g_strcmp0(s1,s2);	
}

l_size l_str_cpy(l_str *pdst,const l_str src)
{
	l_str_free(pdst);
	(*pdst) = g_strdup(src);
	return  l_str_len((*pdst));	
}

l_size l_str_cpy2(l_str dst,const l_str src,l_size len)
{
	return g_strlcpy(dst,src,len);
}

l_size l_str_cat(l_str *pdst,const l_str src)
{
	l_size l1=l_str_len((*pdst)),l2=l_str_len(src),l=l1+l2;
	l_str str=l_str_new();
	l_str_alloc(&str,l);
	if(l1 != 0)
		g_strlcpy(str,(*pdst),l+1);
	g_strlcat(str,src,l+1);
	l_str_free(pdst);
	(*pdst)=str;
	return l;
}

l_bool l_str_isnull(const l_str str)
{
	if(str == 0)
		return TRUE;
	return FALSE;	
}

l_bool l_str_isprefix(const l_str str,const l_str prefix)
{
	if(g_str_has_prefix(str,prefix)) 
		return TRUE;
	else
		return FALSE;
}

l_str l_str_tail(const l_str str,l_num pos)
{
	l_str ret=l_str_new();
	if(pos < l_str_len(str))
		l_str_write(&ret,"%s",&str[pos]);	
	return ret;	
}

l_size l_str_write(l_str *pstr,const l_str fmt,...)
{
	l_size len;
	l_str tmp=l_str_new();
	l_str_alloc(&tmp,L_MAXSTRWRITE);
	va_list ap;
	va_start(ap,fmt);
	g_vsprintf(tmp,fmt,ap);
	va_end(ap);
	len=l_str_cpy(pstr,tmp);
	l_str_free(&tmp);
	return len;
}

l_size l_str_cwrite(l_str *pstr,const l_str fmt,...)
{
	l_size len;
	l_str tmp=l_str_new();
	l_str_alloc(&tmp,L_MAXSTRWRITE);
	va_list ap;
	va_start(ap,fmt);
	g_vsprintf(tmp,fmt,ap);
	va_end(ap);
	len=l_str_cat(pstr,tmp);
	l_str_free(&tmp);
	return len;
}

l_size l_str_reduce(l_str str,l_size len)
{
	l_size s=l_str_len(str);
	if(s > len)
	{	
		l_mem_bzero(str+len,s-len);
		return len;
	}
	return s;	
}

l_bool l_str_toi(const l_str str,l_int32 *ret)
{
	if(! l_str_isdigit(str))
		return FALSE;
	(*ret)=atol(str);
	return TRUE;	
}

void l_str_compact(l_str str,l_str *cstr)
{
	l_buf buf=l_buf_new();
	l_index i=0,j=0;
	l_size len=l_str_len(str);
	while(j<len)
	{
		if(str[j] != ' ' && str[j] != '\t')
		{	
			l_buf_fwrite(buf,i,"b",(l_byte) str[j]);	
			i++;
		}
		j++;
	}
	l_str_cpy(cstr,(l_str)l_buf_mem(buf,0));
	l_buf_free(&buf);	
}

l_split l_str_split(const l_str str,const l_str delimiter)
{
	l_split split = (l_split)l_mem_malloc0(sizeof(struct split));
	split->nsplit_=0;
	if(str == 0 || delimiter == 0)
	{	
		split->psplit_=l_mem_malloc0(1);
		(split->psplit_)[0] = l_mem_malloc0(1);
	}
	else
		split->psplit_=g_strsplit(str,delimiter,0);
	while((split->psplit_)[(split->nsplit_)] != 0)
		(split->nsplit_)++;
	return split;
}	

l_num l_str_nsplit(const l_split split)
{
	return split->nsplit_;	
}

l_str l_str_rsplit(const l_split split,l_num n)
{
	if(n>= split->nsplit_)
		return 0;
	return (split->psplit_)[n];	
}

void l_str_freesplit(l_split *psplit)
{
	g_strfreev((*psplit)->psplit_);
	l_mem_freezero((void **)psplit,sizeof(struct split));	
}

/*end extern*/
