/* -*- Mode:C; c-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella <vasta@ragnu.it>
 */
 
 #include "../core.h"
 
 #define LUDPDGSIZE 1024
 #define MAXTCPREAD 4096
 
 /*internal*/
 
 l_udp l_sock_udp()
{
	l_udp udp=l_mem_malloc0(sizeof(struct udp));
	udp->sockfd_=0;
	udp->laddr_=0;
	udp->raddr_=0;
	udp->bsend_ = l_buf_new();
	udp->brecv_  =l_buf_new();
	return udp;	
}
 
l_udp l_sock_udpdup(const l_udp udp)
{
	l_udp ret;
	if(udp == 0)
		return 0;
	ret=l_sock_udp();
	ret->sockfd_=udp->sockfd_;
	ret->laddr_=l_ip_dupaddrinfo(udp->laddr_);
	ret->raddr_=l_ip_dupaddrinfo(udp->raddr_);
	ret->bsend_ = l_buf_new();
	ret->brecv_  =l_buf_new();
	l_buf_copy(ret->bsend_,udp->bsend_);
	l_buf_copy(ret->brecv_,udp->brecv_);
	return ret;
}

void l_sock_udpsetraddr(const l_str host,const l_str port,l_udp udp,l_out out)
{
	udp->raddr_=l_ip_newaddrinfo(host,port,SOCK_DGRAM,0,out);	
}
 
l_tcp l_sock_tcp()
{
	l_tcp tcp=l_mem_malloc0(sizeof(struct tcp));
	tcp->sockfd_=0;
	tcp->laddr_=0;
	tcp->raddr_=0;
	tcp->bsend_ = l_buf_new();
	tcp->brecv_  =l_buf_new();
	return tcp;		
}

l_raw  l_sock_raw()
{
	l_raw raw=l_mem_malloc0(sizeof(struct raw));
	raw->handle_=0;
	raw->bsend_ = l_buf_new();
	raw->brecv_  =l_buf_new();
	return raw;			
}

 /*end internal*/
 
/*extern*/

l_udp l_sock_udplisten(const l_str ip,const l_str port,l_out out)
{
	l_udp udp = l_sock_udp();
	udp->laddr_=l_ip_newaddrinfo(ip,port,SOCK_DGRAM,0,out);
	if(l_out_iserr(out))
	{
		l_sock_udpfree(&udp);
		return 0;
	}
	if((udp->sockfd_ =socket((udp->laddr_)->ai_family,(udp-> laddr_)->ai_socktype,(udp->laddr_)->ai_protocol)) == -1)
	{
		l_sock_udpfree(&udp);
		l_out_err(out,"l_sock_udplisten:%s",l_str_error(errno));
		return 0;
	}		
	if(bind(udp->sockfd_,(udp->laddr_)->ai_addr,(udp->laddr_)->ai_addrlen) !=0)
	{
		l_sock_udpfree(&udp);
		l_out_err(out,"l_sock_udplisten:%s",l_str_error(errno));
		return 0;
	}
	return udp;
}
 
l_udp l_sock_udpaccept(const l_udp udp,l_out out)
{
	l_udp ret=l_sock_udpdup(udp);
	l_sock_udprecv(ret,out);
	if(l_out_iserr(out))
	{
		l_sock_udpfree(&ret);
		return 0;
	}
	return ret;	
}

l_size l_sock_udprecv(l_udp udp,l_out out)
{
	l_size size;
	l_str host=l_str_new(),port=l_str_new();
	struct sockaddr_storage peeraddr;
	socklen_t peeraddr_len=sizeof(struct sockaddr_storage);
	if(udp == 0)
	{
		l_out_err(out,"l_sock_udprecv:udp connection not associated");
		return 0;
	}	
	if(udp->raddr_ != 0)
		l_ip_freeaddrinfo(&(udp->raddr_));
	size =recvfrom(udp->sockfd_,l_buf_alloc(udp->brecv_,LUDPDGSIZE),LUDPDGSIZE,0,(struct sockaddr *) &peeraddr, &peeraddr_len);
	size=l_buf_newsize(udp->brecv_,size);
	l_ip_getstraddrinfo(&peeraddr,peeraddr_len,&host,&port,out);
	if(l_out_iserr(out))
		size=0;
	else
	{
		l_sock_udpsetraddr(host,port,udp,out);
		if(l_out_iserr(out))
			size=0;
	}
	l_str_free(&host);
	l_str_free(&port);
	return size;
}

void l_sock_udpsend(l_udp udp,l_out out)
{
	l_size size;
	if(udp == 0)
	{	
		l_out_err(out,"l_sock_udpsend:udp connection not associated");
		return;
	}		
	if((size=sendto(udp->sockfd_,l_buf_mem(udp->bsend_,0),l_buf_size(udp->bsend_),0,udp->raddr_->ai_addr,udp->raddr_->ai_addrlen))==-1)
		l_out_err(out,l_str_error(errno));
	else	
		fsync(udp->sockfd_);	
}

void l_sock_udpfree(l_udp *pudp)
{
	if((*pudp) != 0)
	{
		if((*pudp)->laddr_ != 0)
			l_ip_freeaddrinfo(&((*pudp)->laddr_));
		if((*pudp)->raddr_ != 0)
			l_ip_freeaddrinfo(&((*pudp)->raddr_));
		l_buf_free(&((*pudp)->bsend_));
		l_buf_free(&((*pudp)->brecv_));
		l_mem_freezero((void **)pudp,sizeof(struct udp));
	}
}

void l_sock_udpclose(l_udp *pudp)
{
	if((*pudp) != 0)
	{
		close((*pudp)->sockfd_);
		l_sock_udpfree(pudp);
	}
}

void l_sock_tcpbsendreset(l_tcp tcp)
{
	l_buf_reset(tcp->bsend_);	
}

void l_sock_tcpbrecvreset(l_tcp tcp)
{
	l_buf_reset(tcp->brecv_);	
}

l_tcp l_sock_tcpclient(const l_str rhost,const l_str rport,l_out out)
{
	l_tcp tcp;
	l_str ip = l_ip_resolver(rhost,out);
	if(l_out_iserr(out))
	{
		l_str_free(&ip);
		return 0;
	}
	tcp=l_sock_tcp();	
	tcp->raddr_=l_ip_newaddrinfo(ip,rport,SOCK_STREAM,0,out);
	if(l_out_iserr(out))
	{
		l_sock_tcpfree(&tcp);
		return 0;
	}		
	if((tcp->sockfd_ = socket(tcp->raddr_->ai_family,tcp->raddr_->ai_socktype,0)) == -1)
	{	
		l_out_err(out,"l_sock_tcpclient:%s",(l_str) l_str_error(errno));
		l_sock_tcpfree(&tcp);
		return 0;
	}
	if(connect(tcp->sockfd_,tcp->raddr_->ai_addr,tcp->raddr_->ai_addrlen) == -1)
	{	
		l_out_err(out,"l_sock_tcpclient:%s",(l_str) l_str_error(errno));
		l_sock_tcpfree(&tcp);
		return 0;
	}
	return tcp;	
}

void l_sock_tcpTimeout(l_tcp tcp,l_time time,l_out out)
{
	l_int32 err;
	struct timeval timeout;      
	timeout.tv_sec = (l_long)time;
	timeout.tv_usec = (l_long)((time-((l_time)timeout.tv_sec))*1000000);
	if ((err=setsockopt (tcp->sockfd_, SOL_SOCKET, SO_RCVTIMEO, (l_char *)&timeout,sizeof(timeout))) < 0)
		l_out_err(out,"l_sock_tcpTimeout:%s\n",l_str_error(err));
}

void l_sock_tcpsend(l_tcp tcp,l_out out)
{
	l_int32 tmp;
	l_int32 i=0;
	if(tcp == 0)
	{	
		l_out_err(out,"l_sock_tcpsend:tcp connection not associated");
		return;
		
	}		
	while((tmp=write(tcp->sockfd_,l_buf_mem(tcp->bsend_,0),l_buf_size(tcp->bsend_))) == -1 && i < 5)
		i++;
	if(i==5 || tmp==0)
		l_out_err(out,"l_sock_tcpsend:error send message");
}

l_size l_sock_tcprecv(l_tcp tcp,l_str end,l_out out)
{
	l_int32 retcode;
	l_num point=0;
	l_char tmp[MAXTCPREAD];
	l_bool go=TRUE;
	if(tcp == 0)
	{
		l_out_err(out,"l_sock_tcprecv:tcp connection not associated");
		return 0;
	}	
	l_mem_bzero(tmp,MAXTCPREAD);
	while(go && ((retcode = read(tcp->sockfd_,(void *)tmp,MAXTCPREAD)) >= 0))
	{
		point += retcode;
		l_buf_cwrite(tcp->brecv_,tmp,retcode);
		l_mem_bzero(tmp,MAXTCPREAD);
		if(end == 0)  
			go=FALSE;
		if(l_buf_tokenendstr(tcp->brecv_,end))
		{
			go=FALSE;
			l_buf_newsize(tcp->brecv_,l_buf_size(tcp->brecv_)-l_str_len(end));
		}		
	}
	if(point == 0)
		l_out_err(out,"l_sock_tcprecv:error recv message");
	return (l_size) point;
}

void l_sock_tcpfree(l_tcp *ptcp)
{
	if((*ptcp) != 0)
	{
		if((*ptcp)->laddr_ != 0)
			l_ip_freeaddrinfo(&((*ptcp)->laddr_));
		if((*ptcp)->raddr_ != 0)
			l_ip_freeaddrinfo(&((*ptcp)->raddr_));
		l_buf_free(&((*ptcp)->bsend_));
		l_buf_free(&((*ptcp)->brecv_));
		l_mem_freezero((void **)ptcp,sizeof(struct tcp));
	}
}

void l_sock_tcpclose(l_tcp *ptcp)
{
	if((*ptcp) != 0)
	{
		close((*ptcp)->sockfd_);
		l_sock_tcpfree(ptcp);
	}
}

l_raw l_sock_rawopen(const l_str dev,const l_str filter,l_out out)
{
	struct bpf_program fp;
	l_char errpcap[PCAP_ERRBUF_SIZE];
	l_raw raw= l_sock_raw();
	raw->handle_=(void *) pcap_open_live(dev,BUFSIZ,1,1000,errpcap);
	if(raw->handle_ == 0)
	{
		l_sock_rawfree(&raw);
		l_out_err(out,"%s",errpcap);
		return 0;
	}
	if(filter != 0)
	{
		if (pcap_compile((pcap_t *)raw->handle_, &fp,filter,0, 0) == -1 || pcap_setfilter((pcap_t *)raw->handle_, &fp) == -1)  
		{
			l_out_err(out,"l_sock_rawopen:%s",pcap_geterr((pcap_t *)raw->handle_));
			l_sock_rawclose(&raw);
			return 0;
		}	
	}		
	return raw;
}

void l_sock_rawclose(l_raw *praw)
{
	if((*praw) != 0)
	{
		pcap_close((pcap_t *)((*praw)->handle_));
		l_sock_rawfree(praw);
	}		
}

void l_sock_rawfree(l_raw *praw)
{
	if((*praw) != 0)
	{
		l_buf_free(&((*praw)->bsend_));
		l_buf_free(&((*praw)->brecv_));
		l_mem_freezero((void **)praw,sizeof(struct raw));
	}		
}

void l_sock_rawsend(l_raw raw,l_out out)
{
	if(raw==0)
	{
		l_out_err(out,"l_sock_rawsend:no raw connection associated");
		return;
	}
	if(pcap_inject((pcap_t *)raw->handle_,l_buf_mem(raw->bsend_,0),l_buf_size(raw->bsend_)) == -1)
	{
		l_out_err(out,"l_sock_rawsend:%s",pcap_geterr((pcap_t *)raw->handle_));
		return;
	}	
}

l_size l_sock_rawrecv(l_raw raw,l_out out)
{
	struct pcap_pkthdr pkthdr;
	const l_uchar *pack;
	l_num p=0;
	if(raw==0)
	{
		l_out_err(out,"l_sock_rawrecv:no raw connection associated");
		return 0;
	}
	do
	{	
		if((pack=pcap_next((pcap_t *)raw->handle_,&pkthdr)) == 0)
			return 0;
		l_buf_nwrite(raw->brecv_,p,(void *)pack,pkthdr.len);
		p+=pkthdr.len;
	}while(pkthdr.caplen != p);
	return l_buf_size(raw->brecv_);
}

void l_sock_rawroutine(l_raw raw,void *callback,l_uchar *user ,l_out out)
{
	if(raw==0)
		l_out_err(out,"l_sock_rawroutine:no raw connection associated");
	else if(pcap_loop((pcap_t *)raw->handle_,0, (pcap_handler) callback,user) == -1)
			l_out_err(out,pcap_geterr(raw->handle_));
}

void l_sock_rawroutineclose(l_raw *praw)
{
	if((*praw) != 0)
	{
		pcap_breakloop((pcap_t *)((*praw)->handle_));
		l_sock_rawclose(praw);
	}		
}

 /*end extern*/
 