/* -*- Mode:C; c-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella <vasta@ragnu.it>
 */
 
#include "../core.h"

#define FRAGIPV6ERR 65535

/*internal*/
l_str l_ip_4ntoa(l_uint32 ip)
{
	struct in_addr addr;
	addr.s_addr = ip;
	return inet_ntoa(addr);
}

l_bool l_ip_4strtobuf(const l_str ip,l_buf obuf,l_num point)
{
	l_split split=l_str_split(ip,".");
	l_size n = l_str_nsplit(split),i;
	l_buf ret;
	l_int32 v;
	l_str tmp;
	if(n!=4)
	{
		l_str_freesplit(&split);
		return FALSE;
	}
	ret=l_buf_new();
	for(i=0;i<n;i++)
	{
		tmp=l_str_rsplit(split,i);	
		if(l_str_toirange(tmp,0,255,&v))
			l_buf_fwrite(ret,i,"b",(l_byte)v);
		else
		{
			l_str_freesplit(&split);
			l_buf_free(&ret);
			return FALSE;
		}		
	}
	l_buf_copymem(obuf,point,ret,0,4);
	l_str_freesplit(&split);
	l_buf_free(&ret);
	return TRUE;
}

l_bool l_ip_4isstr(const l_str ip)
{
	l_buf obuf=l_buf_new();
	l_bool ret=l_ip_4strtobuf(ip,obuf,0);
	l_buf_free(&obuf);
	return ret;
}

l_bool l_ip_4tobuf(struct addrinfo *ip,l_buf buf,l_num point)
{
	l_uint32 tmp;
	if(ip == 0 || ip->ai_family != AF_INET)
		return FALSE;
	tmp=((struct sockaddr_in *) (ip->ai_addr))->sin_addr.s_addr;
	l_buf_fwrite(buf,point,"b:b:b:b",tmp&0xFF,tmp>>8&0xFF,tmp>>16&0xFF,tmp>>24);
	return TRUE;	
}

l_bool l_ip_4tostr(struct addrinfo *ip,l_str *str)
{
	l_buf buf=l_buf_new();
	l_str tmp;
	if(ip == 0 || ip->ai_family != AF_INET || ! l_ip_4tobuf(ip,buf,0) || (tmp=l_ip_4buftostr(buf,0))==0)
	{	
		l_buf_free(&buf);
		return FALSE;
	}
	l_str_cpy(str,tmp);
	l_buf_free(&buf);
	return TRUE;
}

l_str l_ip_4buftostr(const l_buf buf,l_num point)
{
	l_byte i[4];
	l_str str;
	if(l_buf_size(buf)-point < 4 )
		return 0;
	l_buf_fread(buf,point,"b:b:b:b",&i[0],&i[1],&i[2],&i[3]);
	str=l_str_new();
	l_str_write(&str,"%d.%d.%d.%d",i[0],i[1],i[2],i[3]);
	return str;
}

l_num l_ip_6strfragment(const l_str fipv6,l_buf obuf)
{
	l_split split = l_str_split(fipv6,":");
	l_num n=l_str_nsplit(split),i;
	l_int32 c;
	l_str tmp;
	l_size l;
	l_buf_reset(obuf);
	if(n > 8)
		n=FRAGIPV6ERR;
	else
	{	
		for(i=0;i<n;i++)
		{
			tmp=l_str_rsplit(split,i);
			l=l_str_len(tmp);
			if(l>0 && l<=4 && l_str_tox(tmp,&c))
				l_buf_fwrite(obuf,2*i,"w",(l_word)c);
			else
			{	
				l_buf_reset(obuf);
				n=FRAGIPV6ERR;
				break;
			}
		}
	}	
	if(n!=FRAGIPV6ERR)
		n *=2;
	l_str_freesplit(&split);	
	return n;
}

l_bool l_ip_6strtobuf(const l_str ip,l_buf obuf,l_num point)
{
	l_split split=l_str_split(ip,"::");
	l_num n=l_str_nsplit(split);
	l_buf buf1=l_buf_new(),buf2=l_buf_new(),buf=l_buf_new();
	l_num p1,p2;
	l_bool ret;
	l_str tmp,tmp2;
	l_buf_alloc(buf,16);
	if(n==1)
	{
		tmp=l_str_rsplit(split,0);
		if(l_ip_6strfragment(tmp,buf1) == 16)
		{	
			p1=0;
			p2=FRAGIPV6ERR;
			ret=TRUE;
		}	
		else
			ret=FALSE;
	}
	else if(n==2)
	{
		tmp=l_str_rsplit(split,0);
		tmp2=l_str_rsplit(split,1);
		p1=l_ip_6strfragment(tmp,buf1);
		p2=l_ip_6strfragment(tmp2,buf2);
		if(p1 == 0)
		{
			p1=FRAGIPV6ERR;
			if(p2==0)
			{
				p2=FRAGIPV6ERR;
				ret=TRUE;
			}
			else if(p2 == FRAGIPV6ERR && l_ip_4strtobuf(tmp2,buf2,0))
			{
				p2=12;
				ret=TRUE;
			}
			else if(p2 > 0 && p2 <=14)
			{
				p2 =(16-p2);
				ret=TRUE;
			}
			else
				ret=FALSE;		
		}
		else if(p1 > 0 && p1<=16)
		{
			if(p2==0)
			{
				p2=FRAGIPV6ERR;
				p1=0;
				ret=TRUE;
			}
			else if(p2!=FRAGIPV6ERR && ((p1+p2) <= 14))
			{
				p1=0;
				p2=0+(16-p2);
				ret=TRUE;
			}
			else
				ret=FALSE;	
		}
		else
			ret=FALSE;
	}
	else
		ret=FALSE;
	if(ret==TRUE)
	{
		if(p1 != FRAGIPV6ERR)
			l_buf_copymem(buf,p1,buf1,0,l_buf_size(buf1));
		if(p2 != FRAGIPV6ERR)
			l_buf_copymem(buf,p2,buf2,0,l_buf_size(buf2));	
		l_buf_copymem(obuf,point,buf,0,16);
	}		
	l_str_freesplit(&split);
	l_buf_free(&buf);
	l_buf_free(&buf1);
	l_buf_free(&buf2);
	return ret;
}

l_bool l_ip_6isstr(const l_str ip)
{
	l_buf obuf=l_buf_new();
	l_bool ret=l_ip_6strtobuf(ip,obuf,0);
	l_buf_free(&obuf);
	return ret;
}

l_int32 l_ip_type(const l_str ip,l_out out)
{
	if(l_ip_4isstr(ip))
		return AF_INET;
	if(l_ip_6isstr(ip))
		return AF_INET6;
	l_out_err(out,"l_ip_type:(%s) not bad format",ip);
	return -1;
}

l_str l_ip_ntop(l_int32 af,void *src,l_out out)
{
	l_size size;
	l_str ret;
	l_buf buf;
	if(af == AF_INET)
		size=INET_ADDRSTRLEN;
	else if(af == AF_INET6)
		size=INET6_ADDRSTRLEN;
	else
	{
		l_out_err(out,"l_ip_ntop:address family not valid");
		return 0;
	}
	buf=l_buf_new();
	if(inet_ntop(af,src,(l_str) l_buf_alloc(buf,size),size) == 0)
	{
		l_out_err(out,"l_ip_ntop:%s",l_str_error(errno));
		l_buf_free(&buf);
		return 0;
	}		
	ret = l_str_new();
	l_buf_tostr(buf,0,&ret,l_buf_size(buf));
	l_buf_free(&buf);
	return ret;
}

l_str l_ip_top(struct sockaddr *sa,l_out out)
{
	l_str ret=l_str_new();
	struct sockaddr_in *ip4;
	struct sockaddr_in6 *ip6;
	if(sa->sa_family == AF_INET)
	{	
		ip4=(struct sockaddr_in *) sa;
		ret= l_ip_ntop(sa->sa_family,(void *)&(ip4->sin_addr),out);	
	}
	else if(sa->sa_family == AF_INET6)
	{
		ip6=(struct sockaddr_in6 *) sa;
		ret= l_ip_ntop(sa->sa_family,(void *)&(ip6->sin6_addr),out);
	}
	else
		l_out_err(out,"l_ip_top:not sa_family valid");
	return ret;	
}

l_str l_ip_addrtop(struct addrinfo *ai,l_out out)
{
	if(ai->ai_family == AF_INET)
		return l_ip_ntop(ai->ai_family,&(((struct sockaddr_in *)(ai->ai_addr))->sin_addr),out);
	if(ai->ai_family == AF_INET6)
		return l_ip_ntop(ai->ai_family,&(((struct sockaddr_in6 *)(ai->ai_addr))->sin6_addr),out);
	l_out_err(out,"l_ip_top:address family not valid");
	return 0;
}

void l_ip_getstraddrinfo(void *addr,l_size saddr,l_str *host,l_str *port,l_out out)
{
	#ifdef  _POSIX_SOURCE
	l_int32 s;
	l_buf hbuf=l_buf_new();
	l_buf pbuf=l_buf_new();
	if((s=getnameinfo((struct sockaddr *) addr,saddr,l_buf_alloc(hbuf, NI_MAXHOST), NI_MAXHOST,l_buf_alloc(pbuf, NI_MAXSERV), NI_MAXSERV, NI_NUMERICHOST|NI_NUMERICSERV)) != 0)
		l_out_err(out,"%s",l_str_gaierror(s));
	else
	{
		l_buf_tostr(hbuf,0,host,l_buf_size(hbuf));
		l_buf_tostr(pbuf,0,port,l_buf_size(pbuf));
	}
	l_buf_free(&hbuf);
	l_buf_free(&pbuf);
	#else
	l_err(out,"no macro _POSIX_SOURCE found");
	return 0;
	#endif
}

void l_ip_getaddrinfo(struct addrinfo *ai,l_str *host,l_str *port,l_out out)
{
	switch(ai->ai_family)
	{
		case AF_INET:l_ip_getstraddrinfo((void *) ai->ai_addr,sizeof(struct sockaddr_in),host,port,out);
				      break;
		case AF_INET6:l_ip_getstraddrinfo((void *) ai->ai_addr,sizeof(struct sockaddr_in6),host,port,out);
					break;
		default:
				l_out_err(out,"l_ip_getaddrinfo:address family not valid");
				break;
	}	
}	

struct addrinfo *l_ip_newaddrinfo(const l_str ip,const l_str port,l_int32 socktype,l_int32 flags,l_out out)
{
	#ifdef  _POSIX_SOURCE
	l_int32 s,i;
	struct addrinfo hints,*result;
	i=l_ip_type(ip,out);
	if(i==-1)
		return 0;
	l_mem_bzero(&hints, sizeof(struct addrinfo));
	hints.ai_family = i;    
	hints.ai_socktype = socktype; 
	hints.ai_flags = flags;    
	hints.ai_protocol = 0;          
	hints.ai_canonname = NULL;
	hints.ai_addr = NULL;
	hints.ai_next = NULL;
	if((s=getaddrinfo(ip,port, &hints, &result)) != 0)
	{	
		l_out_err(out,"l_ip_newaddrinfo:%s",l_str_gaierror(s));
		return 0;
	}
	return result;
	#else
	l_out_err(out,"no macro _POSIX_SOURCE found");
	return 0;
	#endif
}

struct addrinfo *l_ip_dupaddrinfo(struct addrinfo *ai)
{
	struct addrinfo *ret;
	l_size s=sizeof(struct addrinfo);
	if(ai==0)
		return 0;
	ret = (struct addrinfo *) l_mem_malloc0(s);
	l_mem_cpy(ret,ai,sizeof(s));
	return ret;
}

void l_ip_freeaddrinfo(struct addrinfo **addr)
{
	if((*addr) != 0)
	{	
		l_mem_bzero((*addr),sizeof(struct addrinfo));
		freeaddrinfo((*addr));
	}
	(*addr)=0;
}

struct addrinfo *l_ip_getsubnetaddrinfo(const l_str ip,const l_str netmask,l_num *nhost,l_out out)
{
	struct addrinfo *aip=0,*anetmask=0,*subnet=0;
	l_uint32 ndst,ntmp,nbit=0;
	if((aip=l_ip_newaddrinfo(ip,"0",0,0,out)) != 0 && (anetmask=l_ip_newaddrinfo(netmask,"0",0,0,out)) != 0)
	{
		if(aip->ai_family == AF_INET && anetmask->ai_family == AF_INET)
		{
			ntmp=((struct sockaddr_in *) (anetmask->ai_addr))->sin_addr.s_addr;
			ndst=((struct sockaddr_in *)(aip->ai_addr))->sin_addr.s_addr & ntmp;
			while(ntmp != 0)
			{
				ntmp >>= 1;
				nbit++;
			}
			nbit=32-nbit;
			(*nhost)=(l_num) pow(2.0,nbit)-1;	
			subnet=l_ip_newaddrinfo(l_ip_4ntoa(ndst),"0",0,0,out);
		}
		else if(aip->ai_family == AF_INET6 && anetmask->ai_family == AF_INET6)
		{
			l_out_err(out,"l_ip_getsubnetaddrinfo:ipv6 subnet not implemented");
		}
		else
			l_out_err(out,"l_ip_getsubnetaddrinfo:ip %s and netmask %s incompatible type",ip,netmask);		
	}
	l_ip_freeaddrinfo(&aip);	
	l_ip_freeaddrinfo(&anetmask);
	return subnet;
}

void l_ip_incraddrinfo(struct addrinfo *addr)
{
	l_uint32 tmp;
	struct sockaddr_in *i4;
	if(addr->ai_family==AF_INET)
	{	
		i4=((struct sockaddr_in *) (addr->ai_addr));
		tmp=i4->sin_addr.s_addr;
		if((tmp >> 24) != 0xFF)	
			i4->sin_addr.s_addr +=0x1000000;
		else if(((tmp & 0xFFFFFF) >> 16) != 0xFF)
		{	
			i4->sin_addr.s_addr = i4->sin_addr.s_addr & 0xFFFFFF; 
			i4->sin_addr.s_addr+= 0x10000;
		}
		else if(((tmp & 0xFFFF) >> 8) != 0xFF)
		{	
			i4->sin_addr.s_addr = i4->sin_addr.s_addr & 0xFFFF;
			i4->sin_addr.s_addr+= 0x100;
		}
		else if(((tmp & 0xFF)) != 0xFF)
		{	
			i4->sin_addr.s_addr = i4->sin_addr.s_addr & 0xFF;
			i4->sin_addr.s_addr+= 0x1;
		}
	}
	else if(addr->ai_family==AF_INET6)
	{
		/*not implemented*/
	}		
}

/*end internal*/

/*extern*/

l_str l_ip_resolver(const l_str host,l_out out)
{
	#ifdef  _POSIX_SOURCE
	l_int32 s;
	l_str ret=l_str_new();
	l_str p=l_str_new();
	struct addrinfo hints,*result;
	l_mem_bzero(&hints, sizeof(struct addrinfo));
	hints.ai_family = PF_UNSPEC;    
	hints.ai_socktype = 0; 
	hints.ai_flags = 0;    
	hints.ai_protocol = 0;          
	hints.ai_canonname = NULL;
	hints.ai_addr = NULL;
	hints.ai_next = NULL;
	if((s=getaddrinfo(host,0, &hints, &result)) != 0)
	{	
		l_out_err(out,"l_ip_resolver:%s",l_str_gaierror(s));
		return 0;
	}
	l_ip_getaddrinfo(result,&ret,&p,out);
	l_str_free(&p);
	l_ip_freeaddrinfo(&result);
	if(l_out_iserr(out))
	{
		l_str_free(&ret);
		return 0;	
	}
	return ret;	
	#else
	l_out_err(out,"no macro _POSIX_SOURCE found");
	return 0;
	#endif	
}

void l_ip_dns(const l_str host,l_str *domain,l_str *ip,l_out out)
{
	#ifdef  _POSIX_SOURCE
	l_int32 s;
	l_str p=l_str_new();
	char hostname[NI_MAXHOST];
	struct addrinfo hints,*result;
	l_mem_bzero(&hints, sizeof(struct addrinfo));
	hints.ai_family = PF_UNSPEC;    
	hints.ai_socktype = 0; 
	hints.ai_flags = 0;    
	hints.ai_protocol = 0;          
	hints.ai_canonname = NULL;
	hints.ai_addr = NULL;
	hints.ai_next = NULL;
	if((s=getaddrinfo(host,0, &hints, &result)) != 0)
		l_out_err(out,"l_ip_resolver:%s",l_str_gaierror(s));
	else	
	{
		l_ip_getaddrinfo(result,ip,&p,out);
		getnameinfo(result->ai_addr, result->ai_addrlen, hostname, NI_MAXHOST, NULL, 0, 0);
		l_str_cpy(domain,hostname);
		l_ip_freeaddrinfo(&result);
	}
	l_str_free(&p);
	#else
	l_out_err(out,"no macro _POSIX_SOURCE found");
	#endif	
}

void l_ip_expand(const l_str ip1,const l_str ip2,l_buf obuf,l_bool valid,l_out out)
{
	struct addrinfo *aip1=0,*aip2=0;
	l_uint32 tmp1,tmp2,tmp3,tmp4;
	if((aip1=l_ip_newaddrinfo(ip1,"0",0,0,out)) != 0 && (aip2=l_ip_newaddrinfo(ip2,"0",0,0,out)) != 0)
	{
		if(aip1->ai_family == AF_INET && aip2->ai_family == AF_INET)
		{
			tmp1=((struct sockaddr_in *) (aip1->ai_addr))->sin_addr.s_addr;
			tmp2 =((struct sockaddr_in *) (aip2->ai_addr))->sin_addr.s_addr;
			tmp3=(tmp1 >> 24) + ((tmp1 & 0xFF0000) >> 8) + ((tmp1 & 0xFF00) << 8) + ((tmp1 & 0xFF) << 24); 
			tmp4=(tmp2 >> 24) + ((tmp2 & 0xFF0000) >> 8) + ((tmp2 & 0xFF00) << 8) + ((tmp2 & 0xFF) << 24);
			while(tmp3 <= tmp4)
			{
				if(valid)
				{
					if((tmp1 >> 24) != 0x00)
						l_buf_fwrite(obuf,l_buf_size(obuf),"s:b", l_ip_4ntoa(tmp1),'\n');
				}
				else
					l_buf_fwrite(obuf,l_buf_size(obuf),"s:b", l_ip_4ntoa(tmp1),'\n');
				l_ip_incraddrinfo(aip1);
				tmp1=((struct sockaddr_in *) (aip1->ai_addr))->sin_addr.s_addr;
				tmp3=(tmp1 >> 24) + ((tmp1 & 0xFF0000) >> 8) + ((tmp1 & 0xFF00) << 8) + ((tmp1 & 0xFF) << 24);
			}
		}
		else if(aip1->ai_family == AF_INET6 && aip2->ai_family == AF_INET6)
		{
			l_out_err(out,"l_ip_expand:ipv6 not implemented");	
		}
		else
			l_out_err(out,"l_ip_expand:ip1 %s and ip2 %s incompatible type",ip1,ip2);	
		l_ip_freeaddrinfo(&aip1);
		l_ip_freeaddrinfo(&aip2);
	}
}

/*end extern*/
