/* -*- Mode:C; c-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella <vasta@ragnu.it>
 */

#include "../core.h"

#define LMSIZEBUFFERDEFAULT 64

/*internal*/

void *l_buf_alloc(l_buf buf,l_size size)
{
	l_size tmp=l_buf_size(buf);
	void *mem=l_buf_mem(buf,0);
	if(buf->msize_ < size)
	{
		while(buf->msize_ < size)
			buf->msize_ *=2;
		l_mem_freezero(&mem,tmp);
		buf->mem_ = l_mem_malloc0(buf->msize_);
		buf->size_=size;	
	}
	else
	{
		l_mem_bzero(mem,tmp);
		buf->size_=size;
	}
	return l_buf_mem(buf,0);	
}

void *l_buf_ext(l_buf buf,l_size size)
{
	l_size k=l_buf_size(buf);
	void *tmp,*mem;
	if(buf->msize_<size)
	{
		while(buf->msize_<size)
			buf->msize_ *=2;
		tmp=l_mem_malloc0(buf->msize_);
		mem=l_buf_mem(buf,0);
		l_mem_cpy(tmp,mem,k);
		l_mem_freezero(&mem,k);
		buf->mem_=tmp;
		buf->size_=size;
	}
	else
		buf->size_=size;
	return l_buf_mem(buf,k);
}

l_bool l_buf_sub(const l_buf buf,l_index index,const l_buf sbuf)
{
	l_index i;
	l_bool ret=FALSE;
	if((index+l_buf_size(sbuf)) <= l_buf_size(buf))
	{
		for(i=0;i<l_buf_size(sbuf);i++)
		{
			ret=TRUE;
			if(l_buf_byte(buf,index+i) != l_buf_byte(sbuf,i))
			{		
				ret=FALSE;
				break;
			}	
		}	
	}
	return ret;	
}

l_bool l_buf_substr(l_buf buf,l_index index,const l_str str)
{
	l_bool ret;
	l_buf tmp=l_buf_new();
	l_buf_fwrite(tmp,0,"s",str);
	ret=l_buf_sub(buf,index,tmp);
	l_buf_free(&tmp);
	return ret;
}

l_bool l_buf_tokenend(const l_buf buf,const l_buf token)
{
	if(l_buf_size(token) <= l_buf_size(buf))
		return l_buf_sub(buf,l_buf_size(buf)-l_buf_size(token),token);	
	return FALSE;
}

l_bool l_buf_tokenendstr(const l_buf buf,const l_str token)
{
	l_buf tmp;
	l_bool ret;
	if(token == 0 || l_str_cmp(token,"") == 0)
		return TRUE;
	tmp=l_buf_new();
	l_buf_fwrite(tmp,0,"s",token);
	ret=l_buf_tokenend(buf,tmp);
	l_buf_free(&tmp);
	return ret;
}

/*end internal*/

/*extern*/

void *l_buf_mem(l_buf buf,l_index point)
{
	if(l_buf_size(buf) <= point)
		l_buf_ext(buf,point+1);
	return (buf->mem_)+point;	
}

l_buf l_buf_new()
{
	l_buf ret = l_mem_malloc0(sizeof(struct buffer));
	ret->mem_=l_mem_malloc0(LMSIZEBUFFERDEFAULT);
	ret->size_=0;
	ret->msize_=LMSIZEBUFFERDEFAULT;
	return ret;
}

void l_buf_free(l_buf *pbuf)
{
	void *mem=l_buf_mem((*pbuf),0);
	l_mem_freezero(&mem,l_buf_size((*pbuf)));
	l_mem_bzero((*pbuf),sizeof(struct buffer));
	l_mem_free((void **)pbuf);	
}

l_size l_buf_size(const l_buf buf)
{
	return buf->size_;
}

void l_buf_reset(l_buf buf)
{
	l_mem_bzero(l_buf_mem(buf,0),l_buf_size(buf));
	buf->size_=0;
}

l_size l_buf_newsize(l_buf buf,l_size ns)
{
	if(ns < l_buf_size(buf))
	{
		l_mem_bzero(l_buf_mem(buf,ns),l_buf_size(buf)-ns);
		buf->size_=ns;
	}
	else if(ns > l_buf_size(buf))
		l_buf_ext(buf,ns);
	return l_buf_size(buf);
}

l_byte l_buf_byte(const l_buf buf,l_index nbyte)
{
	l_byte *ret;
	if(nbyte >= l_buf_size(buf))
		return '\0';
	ret=l_buf_mem(buf,nbyte);
	return (*ret);	
}

l_byte l_buf_bitseq(const l_buf buf,l_index a,l_index nbits)
{
	l_byte byte=0x00;
	l_num b=a+nbits-1;
	l_num num1=a/8,num2=b/8,num3=a%8,num4=b%8;
	if(nbits <= 8 && b/8<l_buf_size(buf))
	{
		byte=(l_buf_byte(buf,num1) << num3);
		if(num2==num1)
			byte=byte>>(8-nbits);
		else
			byte=(byte>> (8-nbits)) + (l_buf_byte(buf,num2) >> (7-num4));	
	}		
	return byte;	
}


l_size l_buf_write(l_buf buf,l_index point,void *mem,l_size len)
{
	l_size tlen=point+len;
	if(l_buf_size(buf) < tlen)
		l_buf_ext(buf,tlen);
	l_mem_cpy(l_buf_mem(buf,point),mem,len);
	return l_buf_size(buf);
}

l_size l_buf_nwrite(l_buf buf,l_index point,void *mem,l_size len)
{
	l_size tlen=point+len;
	l_buf_alloc(buf,tlen);
	l_mem_cpy(l_buf_mem(buf,point),mem,len);
	return l_buf_size(buf);	
}

l_size l_buf_cwrite(l_buf buf,void *mem,l_size len)
{
	l_size tlen=l_buf_size(buf)+len;
	l_mem_cpy(l_buf_ext(buf,tlen),mem,len);	
	return l_buf_size(buf);
}

l_size l_buf_fwrite(l_buf buf,l_index point,const l_str fmt,...)
{
	l_buf tbuf;
	l_split split=l_str_split(fmt,":");
	l_byte t,bt[8];
	l_word w;
	l_dword dv;
	l_str tmp,tmp2;
	l_size n=l_str_nsplit(split),i,len;
	va_list args;
	va_start(args,fmt);
	tbuf=l_buf_new();
	for(i=0;i<n;i++)
	{
		tmp=l_str_rsplit(split,i);
		if(l_str_cmp(tmp,"b") == 0)
		{	
			t=(l_byte)va_arg(args,int);
			l_buf_cwrite(tbuf,&t,1);
		}	
		else if(l_str_cmp(tmp,"w") == 0)
		{	
			w=(l_word)va_arg(args,int);
			bt[0]= (l_byte)(w>>8);
			bt[1]=(l_byte)(w&0xFF);
			l_buf_cwrite(tbuf,bt,2);
		}	
		else if(l_str_cmp(tmp,"dw") == 0)
		{
			dv=(l_dword)va_arg(args,int);
			bt[0]=(l_byte) (dv >> 24);
			bt[1]=(l_byte) (dv >> 16);
			bt[2]=(l_byte) (dv >> 8);
			bt[3]=(l_byte)(dv&0xFF);
			l_buf_cwrite(tbuf,bt,4);
		}
		else if(l_str_cmp(tmp,"s") == 0)
		{	
			tmp2=(l_str)va_arg(args,unsigned char *);
			len=l_str_len(tmp2);
			l_buf_cwrite(tbuf,tmp2,len);
		}
		else	
		{
			l_mem_bzero(bt,8);
			l_buf_free(&tbuf);
			l_str_freesplit(&split);
			va_end(args);
			return l_buf_size(buf);
		}
	}
	l_buf_copymem(buf,point,tbuf,0,l_buf_size(tbuf));
	l_mem_bzero(bt,8);
	l_buf_free(&tbuf);
	l_str_freesplit(&split);
	va_end(args);
	return l_buf_size(buf);	
}

l_size l_buf_fread(const l_buf buf,l_index point,const l_str fmt,...)
{
	l_split split=l_str_split(fmt,":"),split2;
	l_num n=l_str_nsplit(split),i;
	l_int32 n2;
	l_size db,t=point;
	l_byte *tb;
	l_word *tw;
	l_dword *tdw;
	l_str tmp,tmp2,*ts;
	va_list args;
	va_start(args,fmt);
	for(i=0;i<n;i++)
	{
		tmp=l_str_rsplit(split,i);
		if(l_str_cmp(tmp,"b") == 0)
		{	
			tb=(l_byte *)va_arg(args,int *);
			(*tb)=l_buf_byte(buf,t++);
		}
		else if(l_str_cmp(tmp,"w") == 0)
		{	
			tw = (l_word *)va_arg(args,int *);
			(*tw) = (l_word)l_buf_byte(buf,t++); 
			(*tw) = ((*tw) << 8)+ (l_word) l_buf_byte(buf,t++);
		}
		else if(l_str_cmp(tmp,"dw") == 0)
		{
			tdw=(l_dword *)va_arg(args,int *);
			(*tdw) =((l_word)l_buf_byte(buf,t++) << 24);
			(*tdw) += ((l_word) l_buf_byte(buf,t++) << 16);
			(*tdw) += ((l_word) l_buf_byte(buf,t++) << 8);
			(*tdw) += (l_word) l_buf_byte(buf,t++);
		}
		else	
		{
			split2 =l_str_split(tmp,".");
			n2=l_str_nsplit(split2);
			if(n2==2)
			{
				tmp2=l_str_rsplit(split2,0);
				if(l_str_cmp(tmp2,"s") == 0)
				{
					tmp2=l_str_rsplit(split2,1);
					if(l_str_toi(tmp2,&n2) == FALSE)
					{
						l_str_freesplit(&split2);
						va_end(args);
						l_str_freesplit(&split);
						return (t-point);
					}
					ts=(l_str *)va_arg(args,char *);	
					if((db=l_buf_tostr(buf,t,ts,n2)) != n2)
					{
						t += db;
						l_str_freesplit(&split2);
						va_end(args);
						l_str_freesplit(&split);
						return (t-point);
					}		
					t += n2;
				}		
				else
				{
					l_str_freesplit(&split2);
					va_end(args);
					l_str_freesplit(&split);
					return (t-point);
				}
			}
			else
			{
				l_str_freesplit(&split2);
				va_end(args);
				l_str_freesplit(&split);
				return (t-point);
			}
		}	
	}
	va_end(args);
	l_str_freesplit(&split);
	return (t-point);
}

l_size l_buf_copy(l_buf dbuf,const l_buf sbuf)
{
	if(l_buf_size(sbuf) != 0)
		l_mem_cpy(l_buf_alloc(dbuf,l_buf_size(sbuf)),l_buf_mem(sbuf,0),l_buf_size(sbuf));
	return l_buf_size(dbuf);
}

l_size l_buf_cat(l_buf dbuf,const l_buf sbuf)
{
	l_size tsize=l_buf_size(dbuf)+l_buf_size(sbuf);
	l_mem_cpy(l_buf_ext(dbuf,tsize),l_buf_mem(sbuf,0),l_buf_size(sbuf));
	return l_buf_size(dbuf);
}


void l_buf_copymem(l_buf dbuf,l_index pd,const l_buf sbuf,l_index ps,l_size len)
{
	l_size tolen=pd+len,tslen=ps+len,nbc,tmp;
	if(tolen > l_buf_size(dbuf))
		l_buf_ext(dbuf,tolen);
	if(tslen > (tmp=l_buf_size(sbuf)))
		nbc=len-(tslen-tmp);
	else
		nbc=len;
	l_mem_cpy(l_buf_mem(dbuf,pd),l_buf_mem(sbuf,ps),nbc);
}

void l_buf_out(const l_buf buf)
{
	l_size i,t=l_buf_size(buf);
	char *tmp=(char *) l_buf_mem(buf,0);
	for (i=0;i<t;i++)
		l_std_out("%c",tmp[i]);
}

void l_buf_err(const l_buf buf)
{
	l_size i,t=l_buf_size(buf);
	char *tmp=(char *) l_buf_mem(buf,0);
	for (i=0;i<t;i++)
		l_std_err("%c",tmp[i]);
}

void l_buf_exa(const l_buf buf)
{
	l_size i,t=l_buf_size(buf);
	l_byte *tmp=(l_byte *) l_buf_mem(buf,0);
	for (i=0;i<t;i++)
		l_std_out("%02x",tmp[i]);	
}

l_size l_buf_tostr(const l_buf buf,l_num point,l_str *pstr,l_size len)
{
	l_size tdim=l_buf_size(buf)-point,dim;
	if(tdim <= 0)
		return 0;
	else if (tdim < len)
		dim=tdim;
	else
		dim=len;
	l_str_alloc(pstr,dim+1);
	l_mem_cpy((*pstr),l_buf_mem(buf,point),dim);
	return dim;
}

l_bool l_buf_strexato(const l_str str,l_buf buf,l_num point)
{
	l_size s;
	l_num n=0,p=0;
	l_uchar vect[L_MAXSTRWRITE];
	l_mem_bzero((void *)vect,L_MAXSTRWRITE);
	if(str==0 || ! l_str_isxdigit(str))
		return FALSE;
	s=l_str_len(str);
	while(n<s && n < L_MAXSTRWRITE)
	{
		if(n%2 == 0)
			vect[p]=(l_byte)  l_digit_asciix(str[n]);
		else
			vect[p]=(vect[p] << 4) + (l_byte)  l_digit_asciix(str[n]);	
		if(n%2 == 1 || n==s-1)
			l_buf_fwrite(buf,point++,"b",vect[p++]);	
		n++;	
	}		
	return TRUE;
}

l_int32 l_buf_cmp(const l_buf buf1,const l_buf buf2)
{
	l_str tmp1 =l_str_new();
	l_str tmp2 =l_str_new();
	l_int32 ret;
	l_buf_tostr (buf1,0,&tmp1,l_buf_size(buf1));
	l_buf_tostr (buf2,0,&tmp2,l_buf_size(buf2));
	ret=l_str_cmp(tmp1,tmp2);
	l_str_free(&tmp1);
	l_str_free(&tmp2);
	return ret;
}

l_int32 l_buf_cmpstr(const l_buf buf,const l_str str)
{
	l_str tmp =l_str_new();
	l_int32 ret;
	l_buf_tostr (buf,0,&tmp,l_buf_size(buf));
	ret=l_str_cmp(tmp,str);	
	l_str_free(&tmp);
	return ret;
}

/*end extern*/

