/* -*- Mode:C; c-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella <vasta@ragnu.it>
 */
 
/**
	*\file lupin-header.h
	*\author Marco Guastella (contact:vasta@ragnu.it)
	*\brief Header file to define the header functions
*/

#ifndef	_LIBLUPINHEADER_H
#define	_LIBLUPINHEADER_H	1

/**
	\defgroup hdrtype hdrtype section
*/

/**
	*\ingroup hdrtype
	*\brief	service access protocol type
*/
typedef enum saptype {SAPUNDEF,IPV4,ARP4} l_sap;

/**
	*\ingroup hdrtype
	*\brief hardware type
*/
typedef enum hwtype {HWUNDEF,ETH} l_hw;

/**
	*\ingroup hdrtype
	*\brief hardware type
*/
typedef enum arpcodetype {ARPCODEUNDEF,REQUEST,REPLY,RREQUEST,RREPLY,DRARPREQUEST,DRARPREPLY,DRARPERROR,
						INARPREQUEST,INARPREPLY,ARPNAK,MARSREQUEST,MARSMULTI,MARSMSERV,MARSJOIN,
						MARSLEAVE,MARSNAK,MARSUNSERV,MARSSJOIN,MARSSLEAVE,MARSGRREQUEST,MARSGRREPLY,
						MARSREDIRECTMAP,MAPOSUNARP}l_arpcode;

/**
	\defgroup hmac3 hmac3 section
	
*/

/**
	*\ingroup hmac3
	*\brief	mac802.3 header type
*/
typedef struct mac3
{
	/**
		*\ingroup hmac3
		*\brief	destination mac
	*/
	l_str dmac_;
	
	/**
		*\ingroup hmac3
		*\brief	source mac
	*/
	l_str smac_;
	
	/**
		*\ingroup hmac.3
		*\brief	the type of protocol being carried
	*/
	l_sap sap_;	
}*l_hmac3;

/**
	*\ingroup hmac3
	*\return new struct mac802.3
	*\brief create struct mac802.3
*/
l_hmac3 l_hmac3_new();

/**
	*\ingroup hmac3
	*\param[in] phmac3 mac802.3 structure pointer
	*\brief frees the memory allocated
*/
void l_hmac3_free(l_hmac3 *phmac3);

/**
	*\ingroup hmac3
	*\param[in] hmac3 mac802.3 structure
	*\brief reset the struct mac802.3
*/
void l_hmac3_reset(l_hmac3 hmac3);

/**
	*\ingroup hmac3
	*\param[in] dmac destination mac address
	*\param[in] smac source mac address
	*\param[in] sap the type of protocol being carried
	*\param[out] hmac3 mac802.3 structure in which to insert the values
	*\param[out] out output structure
	*\brief set the struct mac802.3
*/
void l_hmac3_set(const l_str dmac,const l_str smac,l_sap sap,l_hmac3 hmac3,l_out out);

/**
	*\ingroup hmac3
	*\param[in] hmac3 mac802.3 structure as input
	*\param[out] buf write buffer where the values
	*\param[in]  point point in the buffer where to write the values
	*\param[out] out output structure
	*\return index of the buffer where to write the next header
	 *\brief  writes the values of the structure in the buffer hmac3 buf.In case of the error
		   *On error it returns it out 	
*/
l_index l_hmac3_write(l_hmac3 hmac3,l_buf buf,l_index point,l_out out);

/**
	*\ingroup hmac3
	*\param[in] buf input buffer
	*\param[in] point point buffer from where to read the values
	*\param[out] hmac3 mac802.3, where property values to write
	*\param[out] out output structure
	*\return index of the buffer where to read the next header
	*\brief reads the values of the buffer buf from a point and writes them into the structure mac802.3.
		 *On error it returns it out	
*/	
l_index l_hmac3_read(l_buf buf,l_index point,l_hmac3 hmac3,l_out out);

/**
	\defgroup harp4 harp4 section
	
*/

/**
	*\ingroup harp4
	*\brief      arp4 header type
*/
typedef struct arp4
{
	/**
		*\ingroup harp4
		*\brief      hardware type
	*/
	l_hw hw_;
	
	/**
		*\ingroup harp4
		*\brief      protocol type
	*/
	l_sap ptype_;
	
	/**
		*\ingroup harp4
		*\brief      length of the hardware address in bytes
	*/
	l_byte hwsize_;
	
	/**
		*\ingroup harp4
		*\brief      length of the protocol address in bytes
	*/
	l_byte psize_;
	
	/**
		*\ingroup harp4
		*\brief      operation code type
	*/
	l_arpcode opcode_;
	
	/**
		*\ingroup harp4
		*\brief      source mac address
	*/
	l_str smac_;
	
	/**
		*\ingroup harp4
		*\brief      source ip
	*/
	l_str sip_;
	
	/**
		*\ingroup harp4
		*\brief      destination mac address
	*/
	l_str dmac_;
	
	/**
		*\ingroup harp4
		*\brief      destination ip
	*/
	l_str dip_;
	
}*l_harp4;

/**
	*\ingroup harp4
	*\return new struct arp4
	*\brief create struct arp4
*/
l_harp4 l_harp4_new();

/**
	*\ingroup harp4
	*\param[in] pharp4 arp4 structure pointer
	*\brief frees the memory allocated
*/
void l_harp4_free(l_harp4 *pharp4);

/**
	*\ingroup harp4
	*\param[in] harp4 arp4 struct
	*\brief reset the struct arp4
*/
void l_harp4_reset(l_harp4 harp4);

/**
	*\ingroup harp4
	*\param[in] hw hardware type
	*\param[in] ptype protocol type
	*\param[in] opcode operation code
	*\param[in] smac source mac address
	*\param[in] sip source ip address
	*\param[in] dmac destination mac address
	*\param[in] dip destination ip address
	*\param[out] harp4 arp4 structure in which to insert the values
	*\param[out] out output structure
	*\brief set the struct arp4
*/
void l_harp4_set(l_hw hw,l_sap ptype,l_arpcode opcode,const l_str smac,const l_str sip,const l_str dmac,const l_str dip,l_harp4 harp4,l_out out);

/**
	*\ingroup harp4
	*\param[in] harp4 arp4 structure as input
	*\param[out] buf write buffer where the values
	*\param[in]  point point in the buffer where to write the values
	*\param[out] out output structure
	*\return index of the buffer where to write the next header
	*\brief  writes the values of the structure in the buffer hmac3 buf.In case of the error
		   *On error it returns it out 	
*/
l_index l_harp4_write(l_harp4 harp4,l_buf buf,l_index point,l_out out);

/**
	*\ingroup harp4
	*\param[in] buf input buffer
	*\param[in] point point buffer from where to read the values
	*\param[out] harp4 arp4, where property values to write
	*\param[out] out output structure
	*\return index of the buffer where to read the next header
	*\brief reads the values of the buffer buf from a point and writes them into the structure arp4.
		 *On error it returns it out	
*/	
l_index l_harp4_read(l_buf buf,l_index point,l_harp4 harp4,l_out out);

/**
	\defgroup hftp hftp section
	
*/

/**
	*\ingroup hftp
	*\param[in] user user to authenticate
	*\param[out] obuf ouput buffer
	*\param[in] point point in the buffer where to write the values
	*\brief obuf writes from the point header ftp user
*/
void l_hftp_user(l_str user,l_buf obuf,l_index point);

/**
	*\ingroup hftp
	*\param[in] pass password to authenticate
	*\param[out] obuf output buffer
	*\param[in] point point in the buffer where to write the values
	*\brief obuf writes from the point header ftp pass
*/
void l_hftp_pass(l_str pass,l_buf obuf,l_index point);


#endif /*end lupin-header.h*/ 
