/* -*- Mode:C; c-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella <vasta@ragnu.it>
 */
 
/**
	*\file lupin-core.h
	*\author Marco Guastella (contact:vasta@ragnu.it)
	*\brief Header file to define the basic functions
*/

#ifndef	_LIBLUPINCORE_H
#define	_LIBLUPINCORE_H	1

/**
	\defgroup string string section
	
*/

/**
	*\ingroup string
	*\brief  string type
*/
typedef char *l_str;

/** 	
	\ingroup string
	\return null string	
	\brief initializes the string
*/	
l_str l_str_new();

/** 	
	\ingroup string
	\param[in] pstr string to deallocate pointer	
	\brief deallocates the string pointed to by pdstr
*/
void l_str_free(l_str *pstr);


/** 	
	\ingroup string
	\param[in] str input string
	\return size str input string	
	\brief calculates the file size of string str
*/
l_size l_str_len(const l_str str);

/** 	
	\ingroup string
	\param[in] str input string
	\return the size of the str in bytes	
	\brief calculates the size of str and makes the casting of the value in bytes
*/
l_byte l_str_lentobyte(l_str str);

/** 	
	\ingroup string
	\param[in] s1 input string
	\param[in] s2 input string
	\return an integer less than, equal to, or greater than zero if  s1  is  found,
        *respectively, to be less than, to match, or be greater than s2.
	\brief compares the two strings s1 and s2
*/
l_int32 l_str_cmp(const l_str s1,const l_str s2);

/** 	
	\ingroup string
	\param[out] pdst pointer to destination string
	\param[in] src string to be copied
	\return the file size of the string pointed to by pdst
	\brief copies the string src to the string pointed to by pdst
*/
l_size l_str_cpy(l_str *pdst,const l_str src);

/** 	
	\ingroup string
	\param[out] dst destination string
	\param[in] src string to be copied
        \param[in] len the maximum number of bytes copied	
        \return the file size of the string dst
	\brief copies the string src in dst. dst must be allocated before
*/
l_size l_str_cpy2(l_str dst,const l_str src,l_size len);

/** 	
	\ingroup string
	\param[out] pdst pointer to the string to be concatenated
	\param[in] src string to be concatenated
	\return the file size of the string pointed to by pdst
	\brief concatenates the string src to the string pointed to by pdst
*/
l_size l_str_cat(l_str *pdst,const l_str src);

/** 	
	\ingroup string
	\param[in] str input string 
	\return TRUE se str is null string otherwise return FALSE
	\brief checks whether the string str and NULL
*/
l_bool l_str_isnull(const l_str str);

/** 	
	\ingroup string
	\param[in] str string to verify prefix
	\param[in] prefix prefix to verify
	\return TRUE if prefix is prefix str FALSE otherwise
	\brief verify if prefix is prefix str
*/
l_bool l_str_isprefix(const l_str str,const l_str prefix);

/** 	
	\ingroup string
	\param[in] str input string to cut
	\param[in] pos position string to cut
	\return string to tail str
	\brief return tail str from pos position
*/	
l_str l_str_tail(const l_str str,l_num pos);

/** 	
	\ingroup string
	\param[out] pstr string to write pointer
	\param[in] fmt format string 
	\param[in] ...  	arguments
	\return new size string pointed to by pstr
	\brief produces an output based on the description of the 
	* format string and writes it to in the string pointed to by pstr
*/
l_size l_str_write(l_str *pstr,const l_str fmt,...);

/** 	
	\ingroup string
	\param[out] pstr string to write pointer
	\param[in] fmt format string 
	\param[in] ...  	arguments
	\return new size string pointed to by pstr
	\brief produces an output based on the description of the 
	* format string and concatenates it to in the string pointed to by pstr
*/
l_size l_str_cwrite(l_str *pstr,const l_str fmt,...);

/** 	
	\ingroup string
	\param[in] str string to reduce
	\param[in] len size to reduce str
	\return the size str
	\brief if the file size of str is' more slowly sets its file size in len 
*/
l_size l_str_reduce(l_str str,l_size len);

/**
	\ingroup string
	\param[in] str string to convert in integer
	\param[out] ret pointer integer to write value
	\return TRUE if convert is valid FALSE otherwise
	\brief convert str value in format integer to integer
*/	
l_bool l_str_toi(const l_str str,l_int32 *ret);

/**
	\ingroup string
	\param[in] str input string
	\param[out] cstr output string
	\brief cstr copy the string str to which the spaces are removed
*/
void l_str_compact(l_str str,l_str *cstr);

/**
	*\ingroup string
	*\brief	string split type
*/
typedef struct split
{
	/**
		*\ingroup string
		*\brief	number of strings split
	*/
	l_num nsplit_;
	
	/**
		*\ingroup string
		*\brief 	split strings
	*/
	l_str *psplit_;
	
}*l_split;

/** 	
	\ingroup string
	\param[in] str string to splip
	\param[in] delimiter delimiter split 
	\return new struct split
	\brief  split the string str based on the delimiter delimiter 
	*and returns the structure derived from this
*/
l_split l_str_split(const l_str str,const l_str delimiter);

/** 	
	\ingroup string
	\param[in] split struct split
	\return the number of split string; 0 on error
	\brief  return the number of split string in split
*/
l_num l_str_nsplit(const l_split split);

/** 	
	\ingroup string
	\param[in] split struct split
	\param[in] n the number of the string to read
	\return the corresponding string n 
	\brief  return the corresponding string n
*/
l_str l_str_rsplit(const l_split split,l_num n);

/**
	\ingroup string
	\param[in] psplit struct split pointer
	\brief  deallocates a struct split
*/
void l_str_freesplit(l_split *psplit);

/**
	\defgroup buffer buffer section
	
*/

/**
	*\ingroup buffer 
	*\brief  buffer  type
*/
typedef struct buffer
{
	/**
	*\ingroup buffer 
	*\brief  pointer to memory buffer
	*/
	void *mem_;
	
	/**
	*\ingroup buffer 
	*\brief  buffer size
	*/
	l_size size_;
	
	/**
	*\ingroup buffer
	*\brief  allocated memory mem
	*/
	l_size msize_;
}*l_buf;

/**
	\ingroup buffer
	\return return memory buffer init + point 
	\brief return memory buffer init + point
*/
void *l_buf_mem(l_buf buf,l_index point);

/**
	\ingroup buffer
	\return a new buffer
	\brief creates a new buffer
*/
l_buf l_buf_new();

/**
	\ingroup buffer
	\param[in] pbuf buffer pointer
	\brief eliminates the buffer pointed to by pbuf and free the memory allocated to it
*/
void l_buf_free(l_buf *pbuf);

/**
	\ingroup buffer
	\param[in] buf input buffer
	\return size input buffer
	\brief returns the size of the input buffer
*/
l_size l_buf_size(const l_buf buf);

/**
	\ingroup buffer
	\param[in] buf input buffer
	\brief reset the input buffer and initializes the file size zero
*/
void l_buf_reset(l_buf buf);

/**
	\ingroup buffer
	\param[out] buf buffer to change size
	\param[in] ns new size buf
	\return new buffer size
	\brief change the buffer size if ns < size buf reset memory
*/
l_size l_buf_newsize(l_buf buf,l_size ns);

/**
	\ingroup buffer
	\param[in] buf input buffer
	\param[in] nbyte the position of bytes returned
	\return byte position nbyte
	\brief returns the value of the byte position of the input buffer nbyte
*/
l_byte l_buf_byte(const l_buf buf,l_index nbyte);

/**
	\ingroup buffer
	\param[in] buf input buffer
	\param[in] a initial point from where to draw the bit
	\param[in] nbits number of bits to charge(MAX 8)
	\return the byte format with the bits taken
	\brief  download the buffer buf nbits starting from point a and returns the byte format from these bits
*/
l_byte l_buf_bitseq(const l_buf buf,l_index a,l_index nbits);

/**
	\ingroup buffer
	\param[out] buf write buffer
	\param[in] point point buffer to write
	\param[in] mem memory to be written
	\param[in] len length of the message to be written
	\return the new buffer size
	\brief writes the memory mem of size len into the buffer buf starting at the position point
*/
l_size l_buf_write(l_buf buf,l_index point,void *mem,l_size len);

/**
	\ingroup buffer
	\param[out] buf write buffer
	\param[in] point point buffer to write
	\param[in] mem memory to be written
	\param[in] len length of the message to be written
	\return the new buffer size
	\brief reset buffer and writes the memory mem of size len into the buffer buf starting at the position point
*/
l_size l_buf_nwrite(l_buf buf,l_index point,void *mem,l_size len);

/**
	\ingroup buffer
	\param[out] buf write buffer
	\param[in] mem memory to be written
	\param[in] len length of the message to be written
	\return the new buffer size
	\brief  concatenates the memory mem of size len into the buffer buf
*/
l_size l_buf_cwrite(l_buf buf,void *mem,l_size len);

/**
	\ingroup buffer
	\param[out] buf write buffer
	\param[in] point point buffer to write
	\param[in] fmt fmt format string
	\param[in] ... arguments
	\return the new buffer size
	\brief produces an output based on the description of the 
	* format string and writes it to buffer buf.
	*The format string has the format "v [: v]" where the values of v are:
	*b = l_byte;
	*w = l_word;
	*dw= l_dword;
	*s=string
*/
l_size l_buf_fwrite(l_buf buf,l_index point,const l_str fmt,...);

/**
	\ingroup buffer
	\param[in] buf read buffer
	\param[in] point point buffer to read
	\param[in] fmt fmt format string
	\param[out] ... arguments
	\return the number byte read
	\brief produces an input based on the description of the 
	* format string and reads the values from the buffer buf and copies them in the arguments. 
	*Attention arguments must be pointers.
	*The format string has the format "v [: v]" where the values of v are:
	*b = l_byte;
	*w = l_word;
	*dw= l_dword;
	*s.n= string of length n
*/
l_size l_buf_fread(const l_buf buf,l_size point,const l_str fmt,...);


/**
	\ingroup buffer
	\param[out] dbuf destination buffer
	\param[in]   sbuf source buffer
	\return  the new destination buffer size
	\brief copy the buffer sbuf into the buffer dbuf 
*/	
l_size l_buf_copy(l_buf dbuf,const l_buf sbuf);

/**
	\ingroup buffer
	\param[out] dbuf destination buffer
	\param[in]   sbuf source buffer
	\return  the new destination buffer size
	\brief concatenates the buffer sbuf into the buffer dbuf 
*/	
l_size l_buf_cat(l_buf dbuf,const l_buf sbuf);

/**
	\ingroup buffer
	\param[out] dbuf destination buffer
	\param[in]   pd point buffer sbuf to write
	\param[in]   sbuf source buffer
	\param[in]   ps  point buffer sbuf to copy
	\param[in]   len number of bytes to copy
	\return  the new destination buffer size
	\brief copy len bytes from the buffer sbuf ps from the point buffer at the point dbuf pd
*/
void l_buf_copymem(l_buf dbuf,l_index pd,const l_buf sbuf,l_index ps,l_size len);

/**
	\ingroup buffer
	\param[in] buf print buffer
	\brief print the contents of the buffer to standard output (type string)
*/
void l_buf_out(const l_buf buf);

/**
	\ingroup buffer
	\param[in] buf print buffer
	\brief print the contents of the buffer to standard error (type string)
*/
void l_buf_err(const l_buf buf);

/**
	\ingroup buffer
	\param[in] buf print buffer
	\brief print the contents of the buffer to standard output (sequence of hexadecimal characters)
*/
void l_buf_exa(const l_buf buf);

/**
	\ingroup buffer
	\param[in] buf buffer to copy
	\param[in] point  point from which the conversion
	\param[out] pstr pointer to output string
	\param[in] len number of bytes to copy in the string
	\brief  copy len bytes from buffer buf point point in the string pointed to by pstr
*/
l_size l_buf_tostr(const l_buf buf,l_num point,l_str *pstr,l_size len);

/**
	\ingroup buffer
	\param[in] str string to convert
	\param[out] buf buffer to copy value hexadecimal
	\param[in] point point buf to copy value hexadecimal
	\return True if the string is the hexadecimal otherwise False
	\brief  copy the values of a hexadecimal string in a buffer
*/
l_bool l_buf_strexato(const l_str str,l_buf buf,l_num point);

/**
	\ingroup buffer
	\param[in] buf1 input buffer
	\param[in] buf2 input buffer
	\return an integer less than, equal to, or greater than zero if  buf1  is  found,
        *respectively, to be less than, to match, or be greater than buf2.
	\brief compares the two buffer buf1 and buf2
*/
l_int32 l_buf_cmp(const l_buf buf1,const l_buf buf2);

/**
	\ingroup buffer
	\param[in] buf input buffer
	\param[in] str input string
	\return an integer less than, equal to, or greater than zero if  buf  is  found,
        *respectively, to be less than, to match, or be greater than str.
	\brief compares the memory buf and str string
*/
l_int32 l_buf_cmpstr(const l_buf buf,const l_str str);

/**
	\defgroup arraybuffer arraybuffer section
	
*/

/**
	*\ingroup arraybufffer
	*\brief	array buffer type
*/
typedef struct arraybuffer
{
	/**
		*\ingroup arraybuffer
		*\brief memory array buffer
	*/
	l_buf *mem_;
	
	/**
		*\ingroup arraybuffer
		*\brief size array buffer
	*/
	l_size size_;
	
	/**
		*\ingroup arraybuffer
		*\brief	max size array buffer
	*/
	l_size msize_;
}*l_abuf;

/**
	\ingroup arraybuffer
	\return a new buffer
	\brief creates a new array buffer
*/
l_abuf l_abuf_new();

/**
	\ingroup arraybauffer
	\param[in] abuf input array buffer
	\return number elements array buffer
	\brief returns the number elements of the input buffer
*/
l_size l_abuf_size(const l_abuf abuf);

/**
	\ingroup arraybuffer
	\param[in] abuf input array buffer
	\brief reset the input array buffer and initializes the file size zero
*/
void l_abuf_reset(l_abuf abuf);

/**
	\ingroup arraybuffer
	\param[in] abuf input array buffer
	\param[in] index array buffer index to reset buf
	\brief resets the index index in the array buffer buf
*/
void l_abuf_resetbuf(l_abuf abuf,l_index index);

/**
	\ingroup arraybuffer
	\param[in] abuf input array buffer
	\param[in] size new size array buffer
	\return new size array buffer
	\brief  change the buffer size 
*/
l_size l_abuf_newsize(l_abuf abuf,l_size size);

/**
	\ingroup arraybuffer
	\param[in] abuf input array buffer
	\param[in] index buffer index
	\param[out] obuf the buffer copy
	\brief  buffer copy the abuf[index] in obuf
*/
void l_abuf_get(const l_abuf abuf,l_index index,l_buf obuf);

/**
	\ingroup arraybuffer
	\param[in] abuf input array buffer
	\param[in] index buffer index
	\param[out] obuf the buffer copy
	\brief  buffer copy the abuf[index] in obuf and buffer removes from abuf
*/
void l_abuf_getrm(l_abuf abuf,l_index index,l_buf obuf);

/**
	\ingroup arraybuffer
	\param[in] abuf input array buffer
	\param[in] index buffer index
	\param[in] buf the buffer copy
	\brief  abuf[index] copy in buf
*/

void l_abuf_set(l_abuf abuf,l_index index,const l_buf buf);

/**
	\ingroup arraybuffer
	\param[in] pabuf array buffer pointer
	\brief eliminates the buffer pointed to by pabuf and free the memory allocated to it
*/
void l_abuf_free(l_abuf *pabuf);

/**
	\ingroup arraybuffer
	\param[in] abuf input array buffer
	\param[in] index buffer index
	\return the buffer index
	\brief  abuf returns the buffer index
*/
l_buf l_abuf_getbuf(const l_abuf abuf,l_index index);

/**
	\ingroup arraybuffer
	\param[in] abuf input array buffer
	\param[in] index buffer index
	\param[in] point point in abuf[index]
	\return memory abuf[index]->mem+point
	\brief  return  memory abuf[index]->mem+point
*/
void *l_abuf_getmem(const l_abuf abuf,l_index index,l_index point);

/**
	\ingroup arraybuffer
	\param[in] abuf input array buffer
	\param[in] index buffer index
	\param[in] point point in abuf[index]
	\param[in] mem memory to write abuf[index]
	\param[in] len size memory mem
	\return new size  abuf[index]
	\brief  writes the memory mem of size len into the buffer abuf[index] starting at the position point
*/
l_size l_abuf_write(l_abuf abuf,l_index index,l_index point,void *mem,l_size len);

/**
	\ingroup arraybuffer
	\param[in] abuf input array buffer
	\param[in] index buffer index
	\param[in] point point in abuf[index]
	\param[in] mem memory to write abuf[index]
	\param[in] len size memory mem
	\return new size  abuf[index]
	\brief   reset buffer and writes the memory mem of size len into the buffer abuf[index] starting at the position point
*/
l_size l_abuf_nwrite(l_abuf abuf,l_index index,l_index point,void *mem,l_size len);

/**
	\ingroup arraybuffer
	\param[in] abuf input array buffer
	\param[in] index buffer index
	\param[in] mem memory to write abuf[index]
	\param[in] len size memory mem
	\return new size  abuf[index]
	\brief   concatenates the memory mem of size len into the buffer abuf[index]
*/
l_size l_abuf_cwrite(l_abuf abuf,l_index index,void *mem,l_size len);

/**
	\defgroup buffersplit buffersplit section
	
*/

/**
	*\ingroup buffersplit
	*\brief	buffer split type
*/
typedef struct bsplit
{
	/**
		*\ingroup buffersplit
		*\brief	number of bufs split
	*/
	l_num nbsplit_;
	
	/**
		*\ingroup buffersplit
		*\brief 	split buf
	*/
	l_abuf pbsplit_;
	
}*l_bsplit;

/** 	
	\ingroup buffersplit
	\param[in] buf buffer to splip
	\param[in] point index of the buffer to split
	\param[in] delimiter delimiter split 
	\return new struct bsplit
	\brief  split the buffer buf based on the delimiter delimiter 
	*and returns the structure derived from this
*/
l_bsplit l_bsplit_new(const l_buf buf,l_index point,const l_str delimiter);

/** 	
	\ingroup buffersplit
	\param[in] bsplit struct bsplit
	\return the number of split buffer; 0 on error
	\brief  return the number of split buffer in bsplit
*/
l_num l_bsplit_nsplit(const l_bsplit bsplit);

/** 	
	\ingroup buffersplit
	\param[in] bsplit struct bsplit
	\param[in] n the number of the buffer to read
	\return the corresponding buffer n 
	\brief  return the corresponding buffer n
*/
l_buf l_bsplit_read(const l_bsplit bsplit,l_num n);

/** 	
	\ingroup buffersplit
	\param[in] bsplit struct bsplit
	\param[in] n the number of the buffer to read
	\return the corresponding memory buffer n 
	\brief  return the corresponding memory buffer n
*/
void *l_bsplit_memsplit(const l_bsplit bsplit,l_num n);

/**
	\ingroup buffersplit
	\param[in] pbsplit bsplit pointer
	\brief  deallocates a struct bsplit
*/
void l_bsplit_free(l_bsplit *pbsplit);

/**
	\defgroup table table section
	
*/

/**
	*\ingroup table
	*\brief table type
*/
typedef struct table
{
	/**
		*\ingroup table
		*\brief names table columns
	*/
	l_abuf named_;
	
	/**
		*\ingroup table
		*\brief memory table
	*/
	l_abuf *mem_;
	
	/**
		*\ingroup table
		*\brief size row table
	*/
	l_size size_;
	
	/**
		*\ingroup table
		*\brief max size row table
	*/
	l_size msize_;
}*l_tab;

/**
	\ingroup table
	\return a new table
	\brief creates a new table
*/
l_tab l_tab_new();

/**
	\ingroup table
	\param[in] tab input table 
	\brief reset the input table and initializes the file size zero
*/
void l_tab_reset(l_tab tab);

/**
	\ingroup table
	\param[in] tab input table
	\return number row table
	\brief returns the number row of the input table
*/
l_size l_tab_sizerow(const l_tab tab);

/**
	\ingroup table
	\param[in] tab input table
	\param[in] rindex row index
	\return number column tab[rindex](row index)
	\brief returns the number of columns of the row in the table rindex
*/
l_size l_tab_sizecol(const l_tab tab,l_index rindex);

/**
	\ingroup table
	\param[in] tab input table
	\param[in] size new row tab
	\return new row in tab
	\brief  change the row table size 
*/
l_size l_tab_newsize(l_tab tab,l_size size);

/**
	\ingroup table
	\param[in] tab input table
	\param[in] name column name
	\param[in] col column number
	\brief sets the name of the column col
*/
void l_tab_name(l_tab tab,const l_str name,l_index col);

/**
	\ingroup table
	\param[in] tab input table
	\param[in] col column number 
	\brief resets the name of the column col	
*/
void l_tab_rname(l_tab tab,l_index col);

/**
	\ingroup table
	\param[in] tab input table
	\param[in] row row table index
	\param[in] col  column table index
	\param[out] obuf the buffer copy
	\brief  buffer copy the table[row][col] in obuf
*/
void l_tab_get(const l_tab tab,l_index row,l_index col,l_buf obuf);

/**
	\ingroup table
	\param[in] tab input table
	\param[in] row row table index
	\param[in] name  column table name
	\param[out] obuf the buffer copy
	\brief  buffer copy the table[row]["name"] in obuf
*/
void l_tab_getn(const l_tab tab,l_index row,const l_str name,l_buf obuf);

/**
	\ingroup table
	\param[in] tab input table
	\param[in] row row table index
	\param[in] col  column table index
	\param[out] obuf the buffer copy
	\brief  buffer copy the table[row][col] in obuf and buffer removes from tab
*/
void l_tab_getrm(l_tab tab,l_index row,l_index col,l_buf obuf);

/**
	\ingroup table
	\param[in] tab input table
	\param[in] row row table index
	\param[in] name  column table name
	\param[out] obuf the buffer copy
	\brief  buffer copy the table[row]["name"] in obuf and buffer removes from tab
*/
void l_tab_getrmn(l_tab tab,l_index row,const l_str name,l_buf obuf);

/**
	\ingroup table
	\param[in] tab input table
	\param[in] row row table index
	\param[in] col column table index
	\param[in] buf the buffer copy
	\brief  table[row][col] copy in buf
*/
void l_tab_set(l_tab tab,l_index row,l_index col,const l_buf buf);

/**
	\ingroup table
	\param[in] tab input table
	\param[in] row row table index
	\param[in] name column table name
	\param[in] buf the buffer copy
	\brief  table[row]["name"] copy in buf
*/
void l_tab_setn(l_tab tab,l_index row,l_str name,const l_buf buf);

/**
	\ingroup table
	\param[in] tab input table
	\param[in] row row table index	
	\param[in] col column table index
	\return the buffer tab[row][col]
	\brief return the buffer tab[row][col]
*/
l_buf l_tab_getbuf(const l_tab tab,l_index row,l_index col);

/**
	\ingroup table
	\param[in] tab input table
	\param[in] row row table index	
	\param[in] name column table name
	\return the buffer tab[row]["name"]
	\brief return the buffer tab[row]["name"]
*/
l_buf l_tab_getbufn(const l_tab tab,l_index row,const l_str name);

/**
	\ingroup table
	\param[in] tab input table
	\param[in] row row table index	
	\param[in] col column table index
	\param[in] point point memory buffer
	\return the buffer tab[row][col]
	\brief return the memory buffer + point from tab[row][col]
*/
void *l_tab_getmem(const l_tab tab,l_index row,l_index col,l_index point);

/**
	\ingroup table
	\param[in] tab input table
	\param[in] row row table index	
	\param[in] name column table name
	\param[in] point point memory buffer
	\return the buffer tab[row][col]
	\brief return the memory buffer + point from tab[row]["name"]
*/
void *l_tab_getmemn(const l_tab tab,l_index row,const l_str name,l_index point);

/**
	\ingroup table
	\param[in] tab input table
	\param[in] row row table index
	\param[in] col column table index
	\param[in] point point in memory buffer tab[row][col]
	\param[in] mem memory to write tab[row][col]
	\param[in] len size memory mem
	\return new size  tab[row][col]
	\brief  writes the memory mem of size len into the buffer tab[row][col] starting at the position point
*/
l_size l_tab_write(l_tab tab,l_index row,l_index col,l_index point,void *mem,l_size len);

/**
	\ingroup table
	\param[in] tab input table
	\param[in] row row table index
	\param[in] name column table name
	\param[in] point point in memory buffer tab[row]["name"]
	\param[in] mem memory to write tab[row]["name"]
	\param[in] len size memory mem
	\return new size  tab[row]["name"]
	\brief  writes the memory mem of size len into the buffer tab[row]["name"] starting at the position point
*/
l_size l_tab_writen(l_tab tab,l_index row,const l_str name,l_index point,void *mem,l_size len);

/**
	\ingroup table
	\param[in] tab input table
	\param[in] row row table index
	\param[in] col column table index
	\param[in] point point in memory buffer tab[row][col]
	\param[in] mem memory to write tab[row][col]
	\param[in] len size memory mem
	\return new size  tab[row][col]
	\brief set buffer and writes the memory mem of size len into the buffer tab[index][col] starting at the position point
*/
l_size l_tab_nwrite(l_tab tab,l_index  row,l_index col,l_index point,void *mem,l_size len);

/**
	\ingroup table
	\param[in] tab input table
	\param[in] row row table index
	\param[in] name column table name
	\param[in] point point in memory buffer tab[row]["name"]
	\param[in] mem memory to write tab[row]["name"]
	\param[in] len size memory mem
	\return new size  tab[row]["name"]
	\brief set buffer and writes the memory mem of size len into the buffer tab[index]["name"] starting at the position point
*/
l_size l_tab_nwriten(l_tab tab,l_index  row,const l_str name,l_index point,void *mem,l_size len);

/**
	\ingroup table
	\param[in] tab input table
	\param[in] row row table index
	\param[in] col column table index
	\param[in] mem memory to write tab[row][col]
	\param[in] len size memory mem
	\return new size  tab[row][col]
	\brief concatenates the memory mem of size len into the buffer tab[row][col]
*/
l_size l_tab_cwrite(l_tab tab,l_index row,l_index col,void *mem,l_size len);

/**
	\ingroup table
	\param[in] tab input table
	\param[in] row row table index
	\param[in] name column table name
	\param[in] mem memory to write tab[row]["name"]
	\param[in] len size memory mem
	\return new size  tab[row]["name"]
	\brief concatenates the memory mem of size len into the buffer tab[row][col]
*/
l_size l_tab_cwriten(l_tab tab,l_index row,const l_str name,void *mem,l_size len);

/**
	\ingroup table
	\param[in] ptab table pointer
	\brief eliminates the table pointed to by pabuf and free the memory allocated to it
*/
void l_tab_free(l_tab *ptab);

/**
	\defgroup std std section
	
*/

/**
	\ingroup std
	\brief function exit type
*/
typedef void (*l_fexit)();

/**
	\ingroup std
	\brief autotools function to call to verify the existence of the library
*/

void l_lupin();

/**
	\ingroup std
	\param[in] fmt format string 
	\param[in] ...  	arguments
	\brief  produces an output based on the description of the 
	* format string and writes it to standard output
*/
void l_std_out(const l_str fmt,...);

/**
	\ingroup std
	\param[in] fmt format string 
	\param[in] ...  	arguments
	\brief  produces an output based on the description of the 
	* format string and writes it to standard error
*/
void l_std_err(const l_str fmt,...);


/**
	\ingroup std
	\param[out] buf input buffer
	\brief reads one line from standard input and write in buf 	
*/
void l_std_in(l_buf buf);

/**
	\ingroup std
	\param[in] failure TRUE failure if otherwise FALSE
	\param[in] e function to exec before to exit
	\brief exit performs the function e the input process and ends with failure if the parameter value 
		        *and failure 'equal to the value TRUE, otherwise exit with success	
*/
void l_std_exit(l_bool failure,l_fexit e);

/**
	\defgroup output output section
	
*/

/**
	*\ingroup output 
	*\brief  return value type 
*/
typedef enum ret{LOK,LERR,LUNDEF} l_ret;

/**
	*\ingroup output
	*\brief  output type 
*/
typedef struct out
{
	/**
		*\ingroup output
		*\brief	standard error buffer
	*/
	l_buf stderr_;
	
	/**
		*\ingroup output
		*\brief	standard output buffer
	*/
	l_buf stdout_;
	
	/**
		*\ingroup output
		*\brief	return value of functions
	*/
	l_ret retvalue_;
	
}*l_out;

/**
	\ingroup std
	\param[in] msg message to write
	\brief eaving the process with the error message written to standard error
*/
void l_std_error(const l_str msg);

/**
	\ingroup std
	\param[in] path path name file
	\param[in] mode open mode
	\param[out] out output return value
	\return descriptor or zero on error (error reported in out)
	\brief opens a file
*/
l_file l_std_openfile(const l_str path,const l_str mode,l_out out);

/**
	\ingroup std
	\param[in] pfile file pointer
	\brief closes a file
*/
void l_std_fileclose(l_file *pfile);

/**
	\ingroup std
	\param[in] file file descriptor
	\param[out] obuf output buffer
	\param[in] point point where it writes the buffer read line
	\param[out] out output structure
	\return the number of bytes read
	\brief reads a line from a file

*/
l_size l_std_filereadline(l_file file,l_buf obuf,l_index point,l_out out);

/**
	\ingroup output
	\return structure out 
	\brief  creates a structure out
*/	
l_out l_out_new();

/**
	\ingroup output
	\param[in] pout output pointer
	\brief  pointer to the output structure to deallocate
*/
void l_out_free(l_out *pout);

/**
	\ingroup output
	\param[in] out the structure to be printed out
	\brief check out the condition of the structure as input. 
	*If LOK prints the value stdout of the standard output, 
	*if LERR prints the value stderr of the standard error  
	*else if LUNDEF not print anything. 
*/
void l_out_print(const l_out out);

/**
	\ingroup output
	\param[in] out the structure to be copied out
	\param[out] str the pointer to string should be copied to the value
	\brief check out the condition of the structure as input. 
	*If LOK copies stdout value in str , 
	*if LERR copies stderr value in str  
	*else if LUNDEF not copies anything. 
*/
void l_out_sprint(const l_out out,l_str *str);

/**
	\ingroup output
	\param[in] out the structure to be verified
	\return out and if TRUE 'state LERR, FALSE otherwise
	\brief return out and if true 'state LERR, FALSE otherwise	
*/
l_bool l_out_iserr(const l_out out);

/**
	\ingroup output
	\param[in] out the structure to be verified
	\param[in] e function to exec
	\brief print out error if the error comes out and performs the function e
*/
void l_out_assert(l_out out,l_fexit e);

/**
	\defgroup filesystem filesystem section
*/

/**
	\ingroup filesystem
	\param[in] path absolute path to create
	\param[out] out output
	\brief creates a directory
*/
void l_fs_mkdir(const l_str path,l_out out);

/**
	\ingroup filesystem
	\param[in] file file name
	\brief remove file name	
*/
void l_fs_remove(const l_str file);

/**
	\defgroup ip ip section
	
*/

/**
	\ingroup ip
	\param[in] host hosts to know the ip
	\param[out] out output return value
	\return the ip address in string format
	\brief resolves the address of a host via the local resolver
*/
l_str l_ip_resolver(const l_str host,l_out out);

/**
	\ingroup ip
	\param[in] host hosts to know the ip and domain
	\param[out] domain the host domain
	\param[out] ip the host ip
	\param[out] out output return value
	\brief resolves the address and reverse lookup of a host and via the local resolver
*/

void l_ip_dns(const l_str host,l_str *domain,l_str *ip,l_out out);

/**
	\ingroup ip
	\param[in] ip1 range start
	\param[in] ip2 range stop
	\param[in] obuf output buffer
	\param[in] valid TRUE if delete addresses that are not associated
	\param[out] out output return value
	\brief generates IP addresses in the range ip1 <-> ip2, and copies obuf
*/
void l_ip_expand(const l_str ip1,const l_str ip2,l_buf obuf,l_bool valid,l_out out);

/**
	\defgroup mac mac section
	
*/

/**
	\ingroup mac
	\param[in] str mac address(00:11:22:33:44:55)
	\param[out] buf buffer to write mac address in byte format
	\param[in] point point in buffer to write mac 
	\return TRUE if mac format is bad FALSE otherwise
	\brief copy str in mac format to buf	
	
*/
l_bool l_mac_strtobuf(const l_str str,l_buf buf,l_num point);


/**
	\defgroup socket socket section
*/

/**
	*\ingroup socket
	*\brief socket udp type
*/
typedef struct udp
{
	/**
		*\ingroup socket
		*\brief socket descriptor
	*/
	l_int32 sockfd_;
	
	/**
		*\ingroup socket
		*\brief local machine network address format
	*/
	struct addrinfo *laddr_;
	
	/**
		*\ingroup socket
		*\brief remote machine network address format
	*/
	struct addrinfo *raddr_;
	
	/**
		*\ingroup socket
		*\brief send buffer
	*/
	l_buf bsend_;
	
	/**
		*\ingroup socket
		*\brief receive buffer
	*/
	l_buf brecv_;
}*l_udp;

/**
	*\ingroup socket
	*\brief socket tcp type
*/
typedef struct tcp
{
	/**
		*\ingroup socket
		*\brief socket descriptor
	*/
	l_int32 sockfd_;
	
	/**
		*\ingroup socket
		*\brief local machine network address format
	*/
	struct addrinfo *laddr_;
	
	/**
		*\ingroup socket
		*\brief remote machine network address format
	*/
	struct addrinfo *raddr_;
	
	/**
		*\ingroup socket
		*\brief send buffer
	*/
	l_buf bsend_;
	
	/**
		*\ingroup socket
		*\brief receive buffer
	*/
	l_buf brecv_;
	
}*l_tcp;

/**
	\ingroup socket
	\brief socket tcp type
*/
typedef struct raw
{
	/**
		\ingroup socket
		\brief pcap handle
	*/
	void *handle_;

	/**
		\ingroup socket
		\brief send buffer
	*/
	l_buf bsend_;
	
	/**
		\ingroup socket
		\brief receive buffer
	*/
	l_buf brecv_;
	
}*l_raw;

/**
	\ingroup socket
	\param[in] ip address the server udp
	\param[in] port port the server udp 
	\param[out] out output return value
	\return new struct socket connection listen udp,0 if error
	\brief listen a udp connection and returns the structure refers to the connection
*/
l_udp l_sock_udplisten(const l_str ip,const l_str port,l_out out);

/**
	\ingroup socket
	\param[in] udp struct  socket connection udp
	\param[out] out output return value
	\return new struct socket connection listen udp,0 if error
	\brief  accepts a connection and returns a new struct udp socket
*/
l_udp l_sock_udpaccept(const l_udp udp,l_out out);

/**
	\ingroup socket
	\param[in] udp struct  socket connection udp
	\param[out] out output return value
	\return the number of bytes received
	\brief  receives a message from remote machine (message in udp->brecv)
*/
l_size l_sock_udprecv(l_udp udp,l_out out);

/**
	\ingroup socket
	\param[in] udp  struct  socket connection udp
	\param[out] out output return value
	\brief  send a message to remote machine (message in udp -> bsend)
*/
void l_sock_udpsend(l_udp udp,l_out out);

/**
	\ingroup socket
	\param[in] pudp udp structure pointer
	\brief  free struct connection udp
*/
void l_sock_udpfree(l_udp *pudp);

/**
	\ingroup socket
	\param[in] tcp struct tcp
	\brief reset the send buffer structure tcp
*/
void l_sock_tcpbsendreset(l_tcp tcp);

/**
	\ingroup socket
	\param[in] tcp struct tcp
	\brief reset the receive buffer structure tcp
*/
void l_sock_tcpbrecvreset(l_tcp tcp);

/**
	\ingroup socket
	\param[in] pudp udp structure pointer 
	\brief  close udp socket and free struct connection udp
*/
void l_sock_udpclose(l_udp *pudp);

/**
	\ingroup socket
	\param[in] rhost remote host
	\param[in] rport remote port
	\param[out] out output return value
	\return structure associated with the tcp connection, 0 if error
	\brief  opens a TCP client
*/

l_tcp l_sock_tcpclient(const l_str rhost,const l_str rport,l_out out);

/**
	\ingroup socket
	\param[in] tcp structure associated with tcp connection
	\param[in] time timeout value
	\param[out] out output return value
	\brief set receive socket timeout
*/
void l_sock_tcpTimeout(l_tcp tcp,l_time time,l_out out);

/**
	\ingroup socket
	\param[in] tcp structure associated with the tcp connection
	\param[out] out output return value
	\brief  sends a TCP packet in remote

*/
void l_sock_tcpsend(l_tcp tcp,l_out out);

/**
	\ingroup socket
	\param[in] tcp structure associated with the tcp connection
	\param[in] end end stream receive to exit
	\param[out] out output return value
	\return the number of bytes received
	\brief recvs a TCP packet from remote,if end == 0 otherwise law until the law once 'end and the final string' received.
	The buffer stcp->brecv not reset and concatenates message

*/
l_size l_sock_tcprecv(l_tcp tcp,l_str end,l_out out);

/**
	\ingroup socket
	\param[in] ptcp tcp structure pointer
	\brief  free struct connection tcp
*/
void l_sock_tcpfree(l_tcp *ptcp);

/**
	\ingroup socket
	\param[in] ptcp tcp structure pointer 
	\brief  close tcp socket and free struct connection tcp
*/
void l_sock_tcpclose(l_tcp *ptcp);

/**
	\ingroup socket
	\param[in] dev device to open raw connection
	\param[in] filter pcap filter string
	\param[out] out output return value
	\return raw structure, if error 0
	\brief open a raw connection in dev applying the pcap filter 
*/
l_raw l_sock_rawopen(const l_str dev,const l_str filter,l_out out);

/**
	\ingroup socket
	\param[in] praw pointer to raw connection 
	\brief  free struct connection raw
*/
void l_sock_rawfree(l_raw *praw);

/**
	\ingroup socket
	\param[in] praw raw structure pointer 
	\brief  close raw connection and free struct connection raw
*/
void l_sock_rawclose(l_raw *praw);

/**
	\ingroup socket
	\param[in] raw raw connection
	\param[out] out output return value
	\brief  sends a packet via a raw
*/
void l_sock_rawsend(l_raw raw,l_out out);

/**
	\ingroup socket
	\param[in] raw raw connection
	\param[out] out output return value
	\brief recv a packet via raw 
*/	
l_size l_sock_rawrecv(l_raw raw,l_out out);

/**
	\ingroup socket
	\param[in] raw raw connection
	\param[in] callback function to process packets (void *callback)(l_uchar *args, const struct pcap_pkthdr *header,const l_uchar *packet)
			  args = args callback , header header libpcap type, packet the packet to process
	\param[in] user input args callback
	\param[in] out output return value
	\brief receives raw packets from one connection and processes them through the callback function
*/
void l_sock_rawroutine(l_raw raw,void *callback,l_uchar *user ,l_out out);

/**
	\ingroup socket
	\param[in] praw pointer to raw connection
	\brief break routine l_srawroutine,close raw connection and free struct to by praw
*/
void l_sock_rawroutineclose(l_raw *praw);

/**
	\defgroup crypt crypt section
	
*/

/**
	\ingroup crypt
	\param[in] input buffer to apply the function md4
	\param[out] obuf output buffer
	\brief apply the function md4 to the buffer of input and to write result in obuf
*/
void l_crypt_md4(const l_buf input,l_buf obuf);

/**
	\ingroup crypt
	\param[in] input buffer to apply the function md5
	\param[out] obuf output buffer
	\brief apply the function md5 to the buffer of input and to write result in obuf
*/	
void l_crypt_md5(const l_buf input,l_buf obuf);

/**
	\ingroup crypt
	\param[in] input buffer to apply the function sha256
	\param[out] obuf output buffer
	\brief apply the function sha256 to the buffer of input and to write result in obuf
*/	
void l_crypt_sha256(const l_buf input,l_buf obuf);

/**
	\ingroup crypt
	\param[in] key buffer to apply the function md5
	\param[in] text text to crypt
	\param[out] obuf output buffer
	\brief apply the function rc4 to the buffer of input and to write result in obuf
*/
void l_crypt_rc4(const l_buf key,const l_buf text,l_buf obuf);

/**
	\defgroup database database section
	
*/

/**
	*\ingroup database 
	*\brief  database type 
*/
typedef struct db
{
	/**
		*\ingroup database
		*\brief database file name
	*/
	l_str file_;
	
	/**
		*\ingroup database
		*\brief database sqlite3 descriptor
	*/
	void *dbase_ ;
}*l_db;

/**
	*\ingroup database
	*\param[in] name database name
	*\param[in] path database path
	*\param[in] newdb True if you want to create a new database otherwise false
	*\param[out] out output value
	*\return new struct db
	*\brief opens a database
*/
l_db l_db_open(const l_str name,l_str path,l_bool newdb,l_out out);

/**
	*\ingroup database
	*\param[in] db database to send query
	*\param[in] query string query(no select)
	*\param[out] out output result
	*\brief send to query to db
*/
void l_db_query(l_db db,const l_str query,l_out out);

/**
	*\ingroup database
	*\param[in] db database to send select
	*\param[in] squery string select query
	*\param[in] table result table
	*\param[out] out output result
	*\brief send to select query to db and return results in table
*/
void l_db_select(l_db db,const l_str squery,l_tab table,l_out out);

/**
	*\ingroup database
	*\param[in] pdb database structure pointer
	*\param[in] remove TRUE removes the file associated with it otherwise FALSE
	*\brief close database and eliminates the database pointed to by pdb and free the memory allocated to it.
	*if remove == TRUE removes the file associated with it
	
*/
void l_db_close(l_db *pdb,l_bool remove);

/**
	\defgroup hashtable hashtable section
	
*/

/**
	\ingroup hashtable 
	\brief hash data type
*/	
typedef void *l_hash;

/**
	\ingroup hashtable 
	\brief hash function type
*/
typedef void (*l_fhash)(l_abuf data,l_abuf args);

/**
	\ingroup hashtable 
	\brief create hash table
	\return hash table to create
*/
l_hash l_hash_new();

/**
	\ingroup hashtable 
	\param[in] phash hash table pointer
	\brief destroys the hash table and frees its memory
*/
void l_hash_free(l_hash *phash);

/**
	\ingroup hashtable 
	\param[in] hash hash table in which to insert element
	\param[in] key  key element
	\param[in] data data element
	\param[in] funz function associated with the element	
	\brief inserts an item in the hash table
*/
void l_hash_insert(l_hash hash,l_str key,l_abuf data,l_fhash funz);

/**
	\ingroup hashtable 
	\param[in] hash hash table
	\return sizes hash table
	\brief sizes returns the hash table
*/
l_size l_hash_size(l_hash hash);

/**
	\ingroup hashtable 
	\param[in] hash hash table
	\param[in] key key element
	\return  data from hash table
	\brief   return data from the hash table corresponding to the key, if not the key
		    and 'good returns 0
*/
l_abuf l_hash_search(l_hash hash,l_str key);

/**
	\ingroup hashtable 
	\param[in] hash hash table
	\param[in] key key element
	\param[in] args econd argument of the function
	\return TRUE if the function and 'was successful, otherwise FALSE
	\brief performs the function corresponding to the key with the first argument data stored in 
		*the hash table and as a second args argument
*/
l_bool l_hash_exec(l_hash hash,l_str key,l_abuf args);

/**
	\ingroup hashtable 
	\param[in] hash hash table
	\param[in] key key element
	\brief removes the element from the hash table corresponding to the key
*/
void l_hash_remove(l_hash hash,l_str key);

/**
	\ingroup hashtable 
	\param[in] hash hash table
	\brief reset hash table
*/
void l_hash_reset(l_hash hash);

/**
	\ingroup hashtable 
	\param[in] hash hash table
	\param[out] okey buffer array of all keys of the hash table
	\brief okey writes all the keys of the hash table
*/
void l_hash_allkey(l_hash hash,l_abuf okey);

#endif /*end lupin-core.h*/ 

