/* -*- Mode:C; c-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella <vasta@ragnu.it>
 */

/**
	*\file lupin-apps.h
	*\author Marco Guastella (contact:vasta@ragnu.it)
	*\brief Header file to application functions
*/

#ifndef	_LIBLUPINAPPS_H
#define	_LIBLUPINAPPS_H	1

/**
	\defgroup http http section
	
*/

/**
	\ingroup http
	\param[in] host address web server
	\param[in] port  port web server
	\param[out] out  http output options or error associated with the connection
	\brief  sends an HTTP request to a remote web server options and print out the result
*/
void l_http_options(const l_str host,const l_str port,l_out out);

/**
	\defgroup tcp tcp section
	
*/

/**
	\ingroup tcp
	\param[in] host host to connect
	\param[in] port port to connect
	\param[in] info if TRUE receive the welcome message
	\param[out] out output returned
	\brief  connects to a server using the tcp protocol and if the parameter info and 'equal 
	*to TRUE receive any welcome message
*/
void l_tcp_scan(l_str host,l_str port,l_bool info,l_out out);

/**
	\defgroup ftp ftp section
	
*/

/**
	\ingroup ftp
	\param[in] host host to connect
	\param[in] port port to connect
	\param[out] out output returned
	\brief connects to a ftp server, receives the welcome message and verifies
	*and if 'configure mode anonymous
*/
void l_ftp_scan(l_str host,l_str port,l_out out);

/**
	\defgroup arp arp section
	
*/

/**
	\ingroup arp
	\param[in] dev the local device to scan the local network
	\param[in] ipsrc the local IP address, 0 if you want to use one assigned to the local device
	\param[in] netmask  the local netmask address, 0 if you want to use one assigned to the local device
	\param[in] maclocal the local mac address, 0 if you want to use one assigned to the local device
	\param[in] pvideo TRUE if you want to print the results while scanning, FALSE
			             the results are returned only through out
	\param[out] out scan result or error associated with the scan
	\brief scanning a local network information of the host and connected to each of them gives 
		  us the ip address and mac address of the device
*/
void l_arp_scanlan(const l_str dev,const l_str ipsrc,const l_str netmask,const l_str maclocal,l_bool pvideo,l_out out);

/**
	\ingroup arp
	\param[in] dev the local device to research host
	\param[in] ipsrc the local IP address, 0 if you want to use one assigned to the local device
	\param[in] ipdst host IP address to be found in local network
	\param[in] maclocal the local mac address, 0 if you want to use one assigned to the local device
	\param[out] out output return value
	\return the mac address associated with your device connected otherwise return 0
	\brief Search ipdst ip host on a local network
*/
l_str l_arp_getremotemac(const l_str dev,const l_str ipsrc,const l_str ipdst,const l_str maclocal,l_out out);

/**
	\ingroup arp
	\param[in] dev the local device to research host
	\param[in] ipsrc the local IP address, 0 if you want to use one assigned to the local device
	\param[in] ipdst host IP address to be found in local network
	\param[in] maclocal the local mac address, 0 if you want to use one assigned to the local device
	\param[out] out research result or error associated with the research
	\brief Search ipdst ip host on a local network, if found returns the mac address associated with 
		 your device connected otherwise, it returns not found
*/
void l_arp_getremotemac2(const l_str dev,const l_str ipsrc,const l_str ipdst,const l_str maclocal,l_out out);

/**
	\ingroup arp
	\param[in] dev  the local device to research hosts
	\param[in] ip1 ip address the first host to search
	\param[in] ip2 ip address the second host to search
	\param[out] mac1 mac address first host
	\param[out] mac2 mac address second host
	\return TRUE if the two hosts have been found, FALSE otherwise and error out stores
	\param[out]  out research result or error associated with the research
	\brief must search the network associated with the host with IP address ip1 and ip2, 
		 and if he finds them both stores and their mac address in mac1 mac2
*/
l_bool l_arp_macdouble(const l_str dev,const l_str ip1,const l_str ip2,l_str *mac1,l_str *mac2,l_out out);

/**
	\ingroup arp
	\param[in] dev arp spoofing device associated
	\param[in] ipvict ip host associated with victim
	\param[in] ipspoof ip spoofing referred to
	\param[in] macspoof mac address to associate with ipspoof,if 0 use the mac address associated with dev
	\return the structure to be passed as input to l_arpspoof_exec,l_arpspoof_err and l_arpspoof_close
	\brief creates the structure associated with the implementation of an IP spoofing attack
*/

void *l_arp_spoof_init(const l_str dev,const l_str ipvict,const l_str ipspoof,const l_str macspoof);

/**
	\ingroup arp
	\param[in] data data structure associated ip spoofing attack
	\return TRUE If and 'associated with an error, FALSE otherwise
	\brief check whether the structure and initialized in init 'error associated with a
*/
l_bool l_arp_spoof_err(void *data);

/** 
	\ingroup arp
	\param[in] data structure associated ip spoofing attack
	\return 0
	\brief do ip spoofing attack by sending an ARP request every 5 seconds
		  the victim so that in its ARP cache entry associated with the all'ipspoof 
		  (see l_arpspoof_init) matches the macspoof
*/
void *l_arp_spoof_exec(void *data);

/**
	\ingroup arp
	\param[in] pdata structure associated ip spoofing attack pointer
	\return output ip spoofing attack
	\brief closes the connection and free the memory associated with raw ip spoofing attack
*/
l_out l_arp_spoof_close(void **pdata);

/**
	\ingroup arp
	\param[in] dev arp deny device associated
	\param[in] ipvict ip host associated with victim
	\param[in] ipdeny ip address of the host to which they do not connect
	\param[in] macspoof mac spoof associated to ipdeny, 0 to generate automatic
	\return the structure to be passed as input to l_arpdeny_exec,l_arpdeny_err and l_arpdeny_close
	\brief creates the structure associated with the implementation of an IP deny attack
*/
void *l_arp_deny_init(const l_str dev,const l_str ipvict,const l_str ipdeny,l_str macspoof);

/**
	\ingroup arp
	\param[in] data structure associated ip deny attack
	\return TRUE If and 'associated with an error, FALSE otherwise
	\brief check whether the structure and initialized in init 'error associated with a
*/
l_bool l_arp_deny_err(void *data);

/** 
	\ingroup arp
	\param[in] data structure associated ip deny attack
	\return 0
	\brief prevent a victim host to communicate with another host
*/
void *l_arp_deny_exec(void *data);

/**
	\ingroup arp
	\param[in] pdata pointer to the structure associated ip deny attack
	\return output ip deny attack
	\brief closes the connection and free the memory associated with raw ip deny attack
*/
l_out l_arp_deny_close(void **pdata);

/**
	\ingroup arp
	\param[in] dev arp direct device associated
	\param[in] ipsrc ip source host associated
	\param[in] ipdst ip destination host associated
	\param[in] maclocal  mac address local,if 0 use the mac address associated with dev
	\param[in] filter optional filter(pcap mode) to receive packets from redirect,0 if not optional filter
	\return the structure to be passed as input to l_arpdirect_exec,l_arpdirect_err and l_arpdirect_close
	\brief creates the structure associated with the implementation of an IP direct
*/
void *l_arp_direct_init(const l_str dev,const l_str ipsrc,const l_str ipdst,const l_str maclocal,const l_str filter);

/**
	\ingroup arp
	\param[in] data structure associated ip direct
	\return TRUE If and 'associated with an error, FALSE otherwise
	\brief check whether the structure and initialized in init 'error associated with a
*/
l_bool l_arp_direct_err(void *data);

/** 
	\ingroup arp
	\param[in] data structure associated ip direct
	\return 0
	\brief reads the packets received by routing them to and from ipsrc ipdst.
		 To receive packages locally should combine this function with l_arpspoof by thread
*/
void *l_arp_direct_exec(void *data);

/**
	\ingroup arp
	\param[in] pdata structure associated ip direct pointer
	\return output ip direct
	\brief closes the connection and free the memory associated with raw ip direct
*/
l_out l_arp_direct_close(void **pdata);

/**
	\defgroup socks socks section
	
*/

/**
	*\ingroup socks
	*\brief socks struct type(incomplete)
*/
typedef struct socks
{
	/**
		*\ingroup socks
		*\brief socks server version
	*/
	l_byte version_;	
	
	/**
		*\ingroup socks
		*\brief struct tcp socks connection 
	*/
	l_tcp tcp_;
}*l_socks;

/**
	\defgroup tor tor section
	
*/

/**
	*\ingroup tor
	*\brief tor struct type
*/
typedef struct tor
{
	/**
		*\ingroup tor
		*\brief      ip address tor servers
	*/
	l_str ip_;
	
	/**
		*\ingroup tor
		*\brief      port tor control
	*/
	l_str portc_;
	
	/**
		*\ingroup tor
		*\brief      password tor control
	*/
	l_str passc_;
	
	/**
		*\ingroup tor
		*\brief      port socks proxy
	*/
	l_str ports_;
	
	/**
		*\ingroup tor
		*\brief      struct tcp tor control
	*/
	l_tcp torc_;
	
	/**
		*\ingroup tor
		*\brief      struct socks proxy
	*/
	l_socks socks_;
}*l_tor;

/** 
	\ingroup tor
	\param[in] ip address tor servers
	\param[in] portc port tor control
	\param[in] passc password tor control
	\param[in] ports port socks proxy
	\return tor structure
	\brief create tor structure
*/
l_tor l_tor_new(const l_str ip,const l_str portc,const l_str passc,const l_str ports);

/** 
	\ingroup tor
	\param[in] ptor tor structure pointer
	\brief close open connection tor service's and free tor structure
*/
void l_tor_free(l_tor *ptor);

/** 
	\ingroup tor
	\param[in] tor tor structure
	\param[out] out return value
	\brief tor connects to tor control
*/
void l_tor_control(l_tor tor,l_out out);

/** 
	\ingroup tor
	\param[in] tor tor structure
	\brief tor closes tor control
*/
void l_tor_controlclose(l_tor tor);

/** 
	\ingroup tor
	\param[in] tor tor structure
	\param[out] out return value
	\brief  tor creates a new circuit for the next connection
*/
void l_tor_newidentity(l_tor tor,l_out out);

/** 
	\ingroup tor
	\param[in] tor tor structure
	\param[in] host host address to resolve
	\param[out] ip return value if valid
	\param[out] out return value
	\brief dns query through the Tor network and returns the value to ip if found otherwise
	*returns the error out
*/
void l_tor_resolve(l_tor tor,const l_str host,l_str *ip,l_out out);

#endif /*end lupin-apps.h*/ 
