/* -*- Mode:C; c-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella <vasta@ragnu.it>
 */
 
 #include "../core.h"
 
 /*digit section*/
 
 l_int32 l_asciixdigit(l_char b)
{
	return g_ascii_xdigit_value(b);
}

l_bool l_isdigit(l_char c)
{
	if(g_ascii_isdigit(c))
		return TRUE;
	return FALSE;
}

l_bool l_isxdigit(l_char c)
{
	if(g_ascii_isxdigit(c))
		return TRUE;
	return FALSE;
}

l_bool l_incrxdigit(l_char *c)
{
	l_str tmp=l_newstr();
	l_int32 v = g_ascii_xdigit_value((*c));
	if(v != -1)
	{	
		v = (v+1)%16;
		l_writestr(&tmp,"%x",v);
		(*c) = tmp[0];
		l_freestr(&tmp);
		return TRUE;
	}
	return FALSE;
}

l_int32 l_bytetodigit(l_byte b)
{
	l_int32 ret=0;
	ret = b;
	return ret;	
}

l_bool l_isstrdigit(const l_str str)
{
	l_size len=l_strlen(str),i;
	if(len == 0)
		return FALSE;
	for(i=0;i<len;i++)
		if(! l_isdigit(str[i]))
			return FALSE;
	return TRUE;
}

l_str l_digittostr(l_int32 d)
{
	l_str str = l_newstr();
	l_writestr(&str,"%d",d);
	return str;
}

l_bool l_isstrxdigit(const l_str str)
{
	l_size len=l_strlen(str),i;
	if(len == 0)
		return FALSE;
	for(i=0;i<len;i++)
	{
		for(i=0;i<len;i++)
			if(! l_isxdigit(str[i]))
				return FALSE;
	}
	return TRUE;
}

/*external functions*/

l_bool l_atoi(const l_str str,l_int32 *ret)
{
	if(! l_isstrdigit(str))
		return FALSE;
	(*ret)=atol(str);
	return TRUE;	
}

/*end external functions*/

l_bool l_atoirange(const l_str str,l_int32 low,l_int32 up,l_int32 *ret)
{
	if(l_atoi(str,ret) && (*ret) >= low && (*ret) <=up)
		return TRUE;
	(*ret)=0;
	return FALSE;		
}

l_bool l_axtoi(const l_str str,l_int32 *ret)
{
	l_size len,i;
	if(! l_isstrxdigit(str))
		return FALSE;
	len=l_strlen(str);
	(*ret)=0;
	for(i=0;i<len;i++)
	{
		(*ret) = (*ret) << 4;
		(*ret) += g_ascii_xdigit_value(str[i]);
	}
	return TRUE;	
}

/*end digit section*/
 
/*string section*/

/*external*/

l_size l_buftostr(l_buf buf,l_num point,l_str *pstr,l_size len)
{
	l_size tdim=l_sizebuf(buf)-point,dim;
	if(tdim <= 0)
		return 0;
	else if (tdim < len)
		dim=tdim;
	else
		dim=len;
	l_allocstr(pstr,dim+1);
	l_memcpy((*pstr),l_membuf(buf,point),dim);
	return dim;
}

l_bool l_strexatobuf(const l_str str,l_buf buf,l_num point)
{
	l_size s;
	l_num n=0,p=0;
	l_uchar vect[L_MAXSTRWRITE];
	l_bzero((void *)vect,L_MAXSTRWRITE);
	if(str==0 || ! l_isstrxdigit(str))
		return FALSE;
	s=l_strlen(str);
	while(n<s && n < L_MAXSTRWRITE)
	{
		if(n%2 == 0)
			vect[p]=(l_byte) l_asciixdigit(str[n]);
		else
			vect[p]=(vect[p] << 4) + (l_byte) l_asciixdigit(str[n]);	
		if(n%2 == 1 || n==s-1)
			l_fwritebuf(buf,point++,"b",vect[p++]);	
		n++;	
	}		
	return TRUE;
}

/*end external*/

/*end string section*/

/*ip section*/

l_str l_ipv4ntoa(l_uint32 ip)
{
	struct in_addr addr;
	addr.s_addr = ip;
	return inet_ntoa(addr);
}

l_bool l_stripv4tobuf(const l_str ip,l_buf obuf,l_num point)
{
	l_split split=l_newsplit(ip,".");
	l_size n = l_nsplit(split),i;
	l_buf ret;
	l_int32 v;
	l_str tmp;
	if(n!=4)
	{
		l_freesplit(&split);
		return FALSE;
	}
	ret=l_newbuf();
	for(i=0;i<n;i++)
	{
		tmp=l_readsplit(split,i);	
		if(l_atoirange(tmp,0,255,&v))
			l_fwritebuf(ret,i,"b",(l_byte)v);
		else
		{
			l_freesplit(&split);
			l_freebuf(&ret);
			return FALSE;
		}		
	}
	l_copymembuf(obuf,point,ret,0,4);
	l_freesplit(&split);
	l_freebuf(&ret);
	return TRUE;
}

l_bool l_isstripv4(const l_str ip)
{
	l_buf obuf=l_newbuf();
	l_bool ret=l_stripv4tobuf(ip,obuf,0);
	l_freebuf(&obuf);
	return ret;
}

l_bool l_ipv4tobuf(struct addrinfo *ip,l_buf buf,l_num point)
{
	l_uint32 tmp;
	if(ip == 0 || ip->ai_family != AF_INET)
		return FALSE;
	tmp=((struct sockaddr_in *) (ip->ai_addr))->sin_addr.s_addr;
	l_fwritebuf(buf,point,"b:b:b:b",tmp&0xFF,tmp>>8&0xFF,tmp>>16&0xFF,tmp>>24);
	return TRUE;	
}

l_str l_readipv4(l_buf buf,l_num point)
{
	l_byte i[4];
	l_str str;
	if(l_sizebuf(buf)-point < 6)
		return 0;
	l_freadbuf(buf,point,"b:b:b:b",&i[0],&i[1],&i[2],&i[3]);
	str=l_newstr();
	l_writestr(&str,"%d.%d.%d.%d",i[0],i[1],i[2],i[3]);
	return str;
}

l_num l_stripv6fragment(const l_str fipv6,l_buf buf)
{
	l_split split = l_newsplit(fipv6,":");
	l_num n=l_nsplit(split),i;
	l_int32 c;
	l_str tmp;
	l_size l;
	l_resetbuf(buf);
	if(n > 8)
		n=FRAGIPV6ERR;
	else
	{	
		for(i=0;i<n;i++)
		{
			tmp=l_readsplit(split,i);
			l=l_strlen(tmp);
			if(l>0 && l<=4 && l_axtoi(tmp,&c))
				l_fwritebuf(buf,2*i,"w",(l_word)c);
			else
			{	
				l_resetbuf(buf);
				n=FRAGIPV6ERR;
				break;
			}
		}
	}	
	if(n!=FRAGIPV6ERR)
		n *=2;
	l_freesplit(&split);	
	return n;
}

l_bool l_stripv6tobuf(const l_str ip,l_buf obuf,l_num point)
{
	l_split split=l_newsplit(ip,"::");
	l_num n=l_nsplit(split);
	l_buf buf1=l_newbuf(),buf2=l_newbuf(),buf=l_newbuf();
	l_num p1,p2;
	l_bool ret;
	l_str tmp,tmp2;
	l_allocbuf(buf,16);
	if(n==1)
	{
		tmp=l_readsplit(split,0);
		if(l_stripv6fragment(tmp,buf1) == 16)
		{	
			p1=0;
			p2=FRAGIPV6ERR;
			ret=TRUE;
		}	
		else
			ret=FALSE;
	}
	else if(n==2)
	{
		tmp=l_readsplit(split,0);
		tmp2=l_readsplit(split,1);
		p1=l_stripv6fragment(tmp,buf1);
		p2=l_stripv6fragment(tmp2,buf2);
		if(p1 == 0)
		{
			p1=FRAGIPV6ERR;
			if(p2==0)
			{
				p2=FRAGIPV6ERR;
				ret=TRUE;
			}
			else if(p2 == FRAGIPV6ERR && l_stripv4tobuf(tmp2,buf2,0))
			{
				p2=12;
				ret=TRUE;
			}
			else if(p2 > 0 && p2 <=14)
			{
				p2 =(16-p2);
				ret=TRUE;
			}
			else
				ret=FALSE;		
		}
		else if(p1 > 0 && p1<=16)
		{
			if(p2==0)
			{
				p2=FRAGIPV6ERR;
				p1=0;
				ret=TRUE;
			}
			else if(p2!=FRAGIPV6ERR && ((p1+p2) <= 14))
			{
				p1=0;
				p2=0+(16-p2);
				ret=TRUE;
			}
			else
				ret=FALSE;	
		}
		else
			ret=FALSE;
	}
	else
		ret=FALSE;
	if(ret==TRUE)
	{
		if(p1 != FRAGIPV6ERR)
			l_copymembuf(buf,p1,buf1,0,l_sizebuf(buf1));
		if(p2 != FRAGIPV6ERR)
			l_copymembuf(buf,p2,buf2,0,l_sizebuf(buf2));	
		l_copymembuf(obuf,point,buf,0,16);
	}		
	l_freesplit(&split);
	l_freebuf(&buf);
	l_freebuf(&buf1);
	l_freebuf(&buf2);
	return ret;
}

l_bool l_isstripv6(const l_str ip)
{
	l_buf obuf=l_newbuf();
	l_bool ret=l_stripv6tobuf(ip,obuf,0);
	l_freebuf(&obuf);
	return ret;
}

l_int32 l_ipstrtype(const l_str ip,l_out out)
{
	if(l_isstripv4(ip))
		return AF_INET;
	if(l_isstripv6(ip))
		return AF_INET6;
	l_err(out,"l_ipstrtype:(%s) not bad format",ip);
	return -1;
}

l_str l_ipntop(l_int32 af,void *src,l_out out)
{
	l_size size;
	l_str ret;
	l_buf buf;
	if(af == AF_INET)
		size=INET_ADDRSTRLEN;
	else if(af == AF_INET6)
		size=INET6_ADDRSTRLEN;
	else
	{
		l_err(out,"l_ipntop:address family not valid");
		return 0;
	}
	buf=l_newbuf();
	if(inet_ntop(af,src,(l_str) l_allocbuf(buf,size),size) == 0)
	{
		l_err(out,"l_ipntop:%s",l_strerror(errno));
		l_freebuf(&buf);
		return 0;
	}		
	ret = l_newstr();
	l_buftostr(buf,0,&ret,l_sizebuf(buf));
	l_freebuf(&buf);
	return ret;
}

l_str l_ipsockaddrtop(struct sockaddr *sa,l_out out)
{
	l_str ret=l_newstr();
	struct sockaddr_in *ip4;
	struct sockaddr_in6 *ip6;
	if(sa->sa_family == AF_INET)
	{	
		ip4=(struct sockaddr_in *) sa;
		ret= l_ipntop(sa->sa_family,(void *)&(ip4->sin_addr),out);	
	}
	else if(sa->sa_family == AF_INET6)
	{
		ip6=(struct sockaddr_in6 *) sa;
		ret= l_ipntop(sa->sa_family,(void *)&(ip6->sin6_addr),out);
	}
	else
		l_err(out,"l_ipsockaddrtop:not sa_family valid");
	return ret;	
}

l_str l_ipaddrtop(struct addrinfo *ai,l_out out)
{
	if(ai->ai_family == AF_INET)
		return l_ipntop(ai->ai_family,&(((struct sockaddr_in *)(ai->ai_addr))->sin_addr),out);
	if(ai->ai_family == AF_INET6)
		return l_ipntop(ai->ai_family,&(((struct sockaddr_in6 *)(ai->ai_addr))->sin6_addr),out);
	l_err(out,"l_ipaddrtop:address family not valid");
	return 0;
}

void l_getstraddrinfo(void *addr,l_size saddr,l_str *host,l_str *port,l_out out)
{
	#ifdef  _POSIX_SOURCE
	l_int32 s;
	l_buf hbuf=l_newbuf();
	l_buf pbuf=l_newbuf();
	if((s=getnameinfo((struct sockaddr *) addr,saddr,l_allocbuf(hbuf, NI_MAXHOST), NI_MAXHOST,l_allocbuf(pbuf, NI_MAXSERV), NI_MAXSERV, NI_NUMERICHOST|NI_NUMERICSERV)) != 0)
		l_err(out,"%s",l_gaistrerror(s));
	else
	{
		l_buftostr(hbuf,0,host,l_sizebuf(hbuf));
		l_buftostr(pbuf,0,port,l_sizebuf(pbuf));
	}
	l_freebuf(&hbuf);
	l_freebuf(&pbuf);
	#else
	l_err(out,"no macro _POSIX_SOURCE found");
	return 0;
	#endif
}

void l_getaddrinfo(struct addrinfo *ai,l_str *host,l_str *port,l_out out)
{
	switch(ai->ai_family)
	{
		case AF_INET:l_getstraddrinfo((void *) ai->ai_addr,sizeof(struct sockaddr_in),host,port,out);
				      break;
		case AF_INET6:l_getstraddrinfo((void *) ai->ai_addr,sizeof(struct sockaddr_in6),host,port,out);
					break;
		default:
				l_err(out,"l_getaddrinfo:address family not valid");
				break;
	}	
}	

struct addrinfo *l_newaddrinfo(const l_str ip,const l_str port,l_int32 socktype,l_int32 flags,l_out out)
{
	#ifdef  _POSIX_SOURCE
	l_int32 s,i;
	struct addrinfo hints,*result;
	i=l_ipstrtype(ip,out);
	if(i==-1)
		return 0;
	l_bzero(&hints, sizeof(struct addrinfo));
	hints.ai_family = i;    
	hints.ai_socktype = socktype; 
	hints.ai_flags = flags;    
	hints.ai_protocol = 0;          
	hints.ai_canonname = NULL;
	hints.ai_addr = NULL;
	hints.ai_next = NULL;
	if((s=getaddrinfo(ip,port, &hints, &result)) != 0)
	{	
		l_err(out,"l_newaddrinfo:%s\n",l_gaistrerror(s));
		return 0;
	}
	return result;
	#else
	l_err(out,"no macro _POSIX_SOURCE found");
	return 0;
	#endif
}

struct addrinfo *l_dupaddrinfo(struct addrinfo *ai)
{
	struct addrinfo *ret;
	l_size s=sizeof(struct addrinfo);
	if(ai==0)
		return 0;
	ret = (struct addrinfo *) l_malloc0(s);
	l_memcpy(ret,ai,sizeof(s));
	return ret;
}

void l_freeaddrinfo(struct addrinfo **addr)
{
	if((*addr) != 0)
	{	
		l_bzero((*addr),sizeof(struct addrinfo));
		freeaddrinfo((*addr));
	}
	(*addr)=0;
}

struct addrinfo *l_getsubnetaddrinfo(const l_str ip,const l_str netmask,l_num *nhost,l_out out)
{
	struct addrinfo *aip=0,*anetmask=0,*subnet=0;
	l_uint32 ndst,ntmp,nbit=0;
	if((aip=l_newaddrinfo(ip,"0",0,0,out)) != 0 && (anetmask=l_newaddrinfo(netmask,"0",0,0,out)) != 0)
	{
		if(aip->ai_family == AF_INET && anetmask->ai_family == AF_INET)
		{
			ntmp=((struct sockaddr_in *) (anetmask->ai_addr))->sin_addr.s_addr;
			ndst=((struct sockaddr_in *)(aip->ai_addr))->sin_addr.s_addr & ntmp;
			while(ntmp != 0)
			{
				ntmp >>= 1;
				nbit++;
			}
			nbit=32-nbit;
			(*nhost)=(l_num) pow(2.0,nbit)-1;	
			subnet=l_newaddrinfo(l_ipv4ntoa(ndst),"0",0,0,out);
		}
		else if(aip->ai_family == AF_INET6 && anetmask->ai_family == AF_INET6)
		{
			l_err(out,"l_getsubnetaddrinfo:ipv6 subnet not implemented");
		}
		else
			l_err(out,"l_getsubnetaddrinfo:ip %s and netmask %s incompatible type",ip,netmask);		
	}
	l_freeaddrinfo(&aip);	
	l_freeaddrinfo(&anetmask);
	return subnet;
}

void l_incraddrinfo(struct addrinfo *addr)
{
	if(addr->ai_family==AF_INET)
		((struct sockaddr_in *) (addr->ai_addr))->sin_addr.s_addr+= 0x1000000;
	else if(addr->ai_family==AF_INET6)
	{
		/*not implemented*/
	}		
}

/*external*/

l_str l_resolver(const l_str host,l_out out)
{
	#ifdef  _POSIX_SOURCE
	l_int32 s;
	l_str ret=l_newstr();
	l_str p=l_newstr();
	struct addrinfo hints,*result;
	l_bzero(&hints, sizeof(struct addrinfo));
	hints.ai_family = PF_UNSPEC;    
	hints.ai_socktype = 0; 
	hints.ai_flags = 0;    
	hints.ai_protocol = 0;          
	hints.ai_canonname = NULL;
	hints.ai_addr = NULL;
	hints.ai_next = NULL;
	if((s=getaddrinfo(host,0, &hints, &result)) != 0)
	{	
		l_err(out,"l_resolver:%s\n",l_gaistrerror(s));
		return 0;
	}
	l_getaddrinfo(result,&ret,&p,out);
	l_freestr(&p);
	l_freeaddrinfo(&result);
	if(l_isouterr(out))
	{
		l_freestr(&ret);
		return 0;	
	}
	return ret;	
	#else
	l_err(out,"no macro _POSIX_SOURCE found");
	return 0;
	#endif	
}

/*end external*/

/*end ip section*/

/*datalink section*/

l_bool l_isstrmac(const l_str str)
{
	l_num i,len;
	l_bool ret=TRUE;
	l_str s;
	l_split split =l_newsplit(str,":");
	if(l_nsplit(split) !=  MACADDRLEN)
		ret=FALSE;
	else
	{
		for(i=0;i<MACADDRLEN;i++)
		{
			s=l_readsplit(split,i);
			len=l_strlen(s);
			if(len > 2 || len==0)
			{
				ret=FALSE;
				break;	
			}
			if(! l_isstrxdigit(s))
			{
				ret=FALSE;
				l_freesplit(&split);	
				break;	
			}	
		}
	}
	l_freesplit(&split);
	return ret;
}

l_str l_incrdigitmac(const l_str mac,l_num pos)
{
	l_num i,len,ns,ps;
	l_bool ret=TRUE;
	l_str s;
	l_str sret=l_newstr();
	if(pos > 11)
		return FALSE;
	else
	{
		ns=(l_num) pos/2;
		ps=(l_num) pos%2;
	}		
	l_split split =l_newsplit(mac,":");
	if(l_nsplit(split) !=  MACADDRLEN)
		ret=FALSE;
	else
	{
		for(i=0;i<MACADDRLEN;i++)
		{
			s=l_readsplit(split,i);
			len=l_strlen(s);
			if(len > 2 || len==0)
			{
				ret=FALSE;
				break;	
			}
			if(! l_isstrxdigit(s))
			{
				ret=FALSE;
				l_freesplit(&split);	
				break;	
			}	
			if(i!=ns)
			{	
				if(len == 1)
					l_strcat(&sret,"0");
			}
			else
			{
				if(len==2)
				{
					if(ps==0)
						l_incrxdigit(&s[0]);
					else
						l_incrxdigit(&s[1]);
						
				}
				else
				{
					if(ps == 0)
						l_strcat(&sret,"1");
					else
					{	
						l_strcat(&sret,"0");
						l_incrxdigit(&s[1]);
					}
				}		
			}
			l_strcat(&sret,s);
			if(i!=5)
				l_strcat(&sret,":");
		}
	}
	l_freesplit(&split);
	if(!ret)
		l_freestr(&sret);
	return sret;
}

/*external*/

l_bool l_strmactobuf(const l_str str,l_buf buf,l_num point)
{
	l_num i,len,j;
	l_str s;
	l_uchar vect[MACADDRLEN];
	l_bzero(vect,6);
	l_split split =l_newsplit(str,":");
	if(l_nsplit(split) !=  MACADDRLEN)
	{
		l_freesplit(&split);	
		return FALSE;		
	}
	else
	{
		for(i=0;i<MACADDRLEN;i++)
		{
			s=l_readsplit(split,i);
			len=l_strlen(s);
			if(len > 2 || len==0)
			{
				l_freesplit(&split);	
				return FALSE;		
			}
			if(! l_isstrxdigit(s))
			{
				l_freesplit(&split);	
				return FALSE;	
			}	
			for(j=0;j<len;j++)
				vect[i] = (vect[i] << 4) + (l_byte) l_asciixdigit(s[j]);
		}
	}
	l_fwritebuf(buf,point,"b:b:b:b:b:b",vect[0],vect[1],vect[2],vect[3],vect[4],vect[5]);
	l_freesplit(&split);
	return TRUE;
}

/*end external*/

l_str l_readmac(l_buf buf,l_num point)
{
	l_byte m[6];
	l_str str;
	if(l_sizebuf(buf)-point < 6)
		return 0;	
	l_freadbuf(buf,point,"b:b:b:b:b:b",&m[0],&m[1],&m[2],&m[3],&m[4],&m[5]);
	str=l_newstr();
	l_writestr(&str,"%02x:%02x:%02x:%02x:%02x:%02x",m[0],m[1],m[2],m[3],m[4],m[5]);
	return str;
}

/*end datalink section*/

/*net interfaces section*/

l_devtype l_getdevtype(const l_str dev,l_out out)
{
	l_devtype dt;
	l_sraw sraw = l_opensraw(dev,0,out);	
	if(sraw == 0)
		return NOTYPE;
	switch(pcap_datalink(sraw->handle))
	{
		case DLT_EN10MB: dt=LETH;
					      break;
		default:  l_err(out,"l_getdevtype:device not valid type");
			       dt=NOTYPE;
			       break;
	}
	l_closesraw(&sraw);
	return dt;	
}

l_bool l_isethdev(const l_str dev,l_out out)
{
	l_devtype dt=l_getdevtype(dev,out);
	if(dt == NOTYPE)
		return FALSE;
	if(dt != LETH)
	{		
		l_err(out,"l_isethdev:not ethernet netcard");
		return FALSE;
	}
	return TRUE;
}

l_bool l_ifinfo(const l_str dev,struct ifreq *ifreq,l_ifparinfo par,l_out out)
{
	l_int32 val;
	l_int32 fd;
	switch(par)
	{
		case HW:	val=SIOCGIFHWADDR;
					break;
		case IP:		val=SIOCGIFADDR;
					break;
		case MASK:	val=SIOCGIFNETMASK;
					break;
		default:		l_err(out,"l_ifinfo:not parinfo valid");
					return FALSE;
	}	
	fd=socket(AF_INET,SOCK_DGRAM,0);
	if(fd < 0)
	{
		fd=socket(AF_INET6,SOCK_DGRAM,0);
		if(fd < 0)
		{
			l_err(out,"l_ifinfo:%s",l_strerror(errno));
			return FALSE;
		}		
	
	}
	l_strcpy2(ifreq->ifr_name,dev,IFNAMSIZ);
	if(ioctl(fd,val,ifreq) < 0)
	{	
		l_err(out,"l_ifinfo:%s",l_strerror(errno));
		close(fd);
		return FALSE;
	}	
	close(fd);
	return TRUE;
}

l_size l_getdevhw(const l_str dev,l_buf buf,l_num point,l_out out)
{
	struct ifreq ifreq;
	l_bzero(&ifreq,sizeof(struct ifreq));
	if(l_ifinfo(dev,&ifreq,HW,out))
		l_writebuf(buf,0,&ifreq.ifr_hwaddr.sa_data,MACADDRLEN);
	return buf->size;	
}

l_str l_getstrdevhw(const l_str dev,l_out out)
{
	struct ifreq ifreq;
	l_str ret=l_newstr();
	l_bzero(&ifreq,sizeof(struct ifreq));
	if(l_ifinfo(dev,&ifreq,HW,out))
		l_writestr(&ret,"%02x:%02x:%02x:%02x:%02x:%02x",(l_uchar)ifreq.ifr_hwaddr.sa_data[0],(l_uchar)ifreq.ifr_hwaddr.sa_data[1],(l_uchar)ifreq.ifr_hwaddr.sa_data[2],(l_uchar)ifreq.ifr_hwaddr.sa_data[3],(l_uchar)ifreq.ifr_hwaddr.sa_data[4],(l_uchar)ifreq.ifr_hwaddr.sa_data[5]);
	return ret;	
}

l_str l_getstrdevip(const l_str dev,l_out out)
{
	struct ifreq ifreq;
	struct sockaddr *sa;
	l_str ret=l_newstr();
	l_bzero(&ifreq,sizeof(struct ifreq));
	if(l_ifinfo(dev,&ifreq,IP,out))
	{	
		sa=(struct sockaddr *)&(ifreq.ifr_addr);
		ret=l_ipsockaddrtop(sa,out);
	}
	return ret;
}

l_str l_getstrdevmask(const l_str dev,l_out out)
{
	struct ifreq ifreq;
	struct sockaddr *sa;
	l_str ret=l_newstr();
	l_bzero(&ifreq,sizeof(struct ifreq));
	if(l_ifinfo(dev,&ifreq,MASK,out))
	{	
		sa=(struct sockaddr *)&(ifreq.ifr_addr);
		ret=l_ipsockaddrtop(sa,out);
	}
	return ret;
}

/*end net interfaces section*/
