/* -*- Mode:C; c-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella <vasta@ragnu.it>
 */
 
 #include "../core.h"
 
 /*udp section*/
 
 l_sudp l_newsudp()
{
	l_sudp sudp=l_malloc0(sizeof(struct sudp));
	sudp->sockfd=0;
	sudp->laddr=0;
	sudp->raddr=0;
	sudp->bsend = l_newbuf();
	sudp->brecv  =l_newbuf();
	return sudp;	
}
 
l_sudp l_dupsudp(l_sudp sudp)
{
	l_sudp ret;
	if(sudp == 0)
		return 0;
	ret=l_newsudp();
	ret->sockfd=sudp->sockfd;
	ret->laddr=l_dupaddrinfo(sudp->laddr);
	ret->raddr=l_dupaddrinfo(sudp->raddr);
	ret->bsend = l_newbuf();
	ret->brecv  =l_newbuf();
	l_copybuf(ret->bsend,sudp->bsend);
	l_copybuf(ret->brecv,sudp->brecv);
	return ret;
}

void l_udpsetraddr(const l_str host,const l_str port,l_sudp udp,l_out out)
{
	udp->raddr=l_newaddrinfo(host,port,SOCK_DGRAM,0,out);	
}

/*external*/

l_sudp l_udpsocklisten(const l_str ip,const l_str port,l_out out)
{
	l_sudp sudp = l_newsudp();
	sudp->laddr=l_newaddrinfo(ip,port,SOCK_DGRAM,0,out);
	if(l_isouterr(out))
	{
		l_freesudp(&sudp);
		return 0;
	}
	if((sudp->sockfd =socket((sudp->laddr)->ai_family,(sudp-> laddr)->ai_socktype,(sudp->laddr)->ai_protocol)) == -1)
	{
		l_freesudp(&sudp);
		l_err(out,"l_udpsocklisten:%s",l_strerror(errno));
		return 0;
	}		
	if(bind(sudp->sockfd,(sudp->laddr)->ai_addr,(sudp->laddr)->ai_addrlen) !=0)
	{
		l_freesudp(&sudp);
		l_err(out,"l_udpsocklisten:%s",l_strerror(errno));
		return 0;
	}
	#ifdef L_DEBUG
		l_out out2 = l_newout();
		l_str s= l_ipaddrtop(sudp->laddr,out2);
		if(s == 0)
			l_debug("l_udpsocklisten:listen udp socket %d in 0:%s",sudp->sockfd,port);	
		else
			l_debug("l_udpsocklisten:listen udp socket %d in %s:%s",sudp->sockfd,s,port);
		l_freestr(&s);
		l_freeout(&out2);
	#endif
	return sudp;
}
 
l_sudp l_udpsockaccept(l_sudp udp,l_out out)
{
	l_sudp ret=l_dupsudp(udp);
	l_udpsockrecv(ret,out);
	if(l_isouterr(out))
	{
		l_freesudp(&ret);
		return 0;
	}
	return ret;	
}

l_size l_udpsockrecv(l_sudp udp,l_out out)
{
	l_size size;
	l_str host=l_newstr(),port=l_newstr();
	struct sockaddr_storage peeraddr;
	socklen_t peeraddr_len=sizeof(struct sockaddr_storage);
	if(udp == 0)
	{
		l_err(out,"l_udpsockrecv:udp connection not associated");
		return 0;
	}	
	if(udp->raddr != 0)
		l_freeaddrinfo(&(udp->raddr));
	size =recvfrom(udp->sockfd,l_allocbuf(udp->brecv,LUDPDGSIZE),LUDPDGSIZE,0,(struct sockaddr *) &peeraddr, &peeraddr_len);
	size=l_newsizebuf(udp->brecv,size);
	l_getstraddrinfo(&peeraddr,peeraddr_len,&host,&port,out);
	if(l_isouterr(out))
		size=0;
	else
	{
		l_udpsetraddr(host,port,udp,out);
		if(l_isouterr(out))
			size=0;
	}
	#ifdef L_DEBUG
		l_debug("l_udpsockrecv:recv udp datagram (%d byte) from client %s:%s",size,host,port);
	#endif
	l_freestr(&host);
	l_freestr(&port);
	return size;
}

void l_udpsocksend(l_sudp udp,l_out out)
{
	l_size size;
	if(udp == 0)
	{	
		l_err(out,"l_udpsocksend:udp connection not associated");
		return;
	}		
	if((size=sendto(udp->sockfd,l_membuf(udp->bsend,0),l_sizebuf(udp->bsend),0,udp->raddr->ai_addr,udp->raddr->ai_addrlen))==-1)
		l_err(out,l_strerror(errno));
	else	
	{	
		fsync(udp->sockfd);	
		#ifdef L_DEBUG
		l_str host=l_newstr();
		l_str port=l_newstr();
		l_getaddrinfo(udp->raddr,&host,&port,out);
		l_debug("l_udpsocksend:send udp datagram (%d byte) to client %s:%s",size,host,port);
		l_freestr(&port);
		l_freestr(&host);
		#endif
	}
}

void l_freesudp(l_sudp *pudp)
{
	if((*pudp) != 0)
	{
		if((*pudp)->laddr != 0)
			l_freeaddrinfo(&((*pudp)->laddr));
		if((*pudp)->raddr != 0)
			l_freeaddrinfo(&((*pudp)->raddr));
		l_freebuf(&((*pudp)->bsend));
		l_freebuf(&((*pudp)->brecv));
		l_memfree((void **)pudp,sizeof(struct sudp));
	}
}

void l_closesudp(l_sudp *pudp)
{
	if((*pudp) != 0)
	{
		close((*pudp)->sockfd);
		#ifdef L_DEBUG
			l_debug("l_closesudp:close udp sock %d",(*pudp)->sockfd);
		#endif
		l_freesudp(pudp);
	}
}

 /*end external*/
 
 /*end udp section*/

/*tcp section*/

l_stcp l_newstcp()
{
	l_stcp stcp=l_malloc0(sizeof(struct stcp));
	stcp->sockfd=0;
	stcp->laddr=0;
	stcp->raddr=0;
	stcp->bsend = l_newbuf();
	stcp->brecv  =l_newbuf();
	return stcp;		
}

/*external*/

l_stcp l_tcpsockclient(const l_str rhost,const l_str rport,l_out out)
{
	l_stcp stcp;
	l_str ip = l_resolver(rhost,out);
	if(l_isouterr(out))
	{
		l_freestr(&ip);
		return 0;
	}
	stcp=l_newstcp();	
	stcp->raddr=l_newaddrinfo(ip,rport,SOCK_STREAM,0,out);
	if((stcp->sockfd = socket(stcp->raddr->ai_family,stcp->raddr->ai_socktype,0)) == -1)
	{	
		l_err(out,"%s",(l_str) l_strerror(errno));
		l_freestcp(&stcp);
		return 0;
	}
	if(connect(stcp->sockfd,stcp->raddr->ai_addr,stcp->raddr->ai_addrlen) == -1)
	{	
		l_err(out,"%s",(l_str) l_strerror(errno));
		l_freestcp(&stcp);
		return 0;
	}
	#ifdef L_DEBUG
		l_debug("l_tcpsockclient:open socket %d (%s:%s)",stcp->sockfd,rhost,rport);
	#endif
	return stcp;	
}

void l_tcpsocksend(l_stcp tcp,l_out out)
{
	l_int32 tmp;
	l_int32 i=0;
	if(tcp == 0)
	{	
		l_err(out,"l_tcpsocksend:tcp connection not associated");
		return;
		
	}		
	while((tmp=write(tcp->sockfd,l_membuf(tcp->bsend,0),l_sizebuf(tcp->bsend))) == -1 && i < 5)
		i++;
	if(i==5 || tmp==0)
		l_err(out,"l_tcpsocksend:error send message");
	#ifdef L_DEBUG
	else	
	{	
		l_str host=l_newstr();
		l_str port=l_newstr();
		l_getaddrinfo(tcp->raddr,&host,&port,out);
		l_debug("l_tcpsocksend:send tcp message (%d byte) to remote machine %s:%s",tmp,host,port);
		l_freestr(&port);
		l_freestr(&host);
	}
	#endif	
}

l_size l_tcpsockrecv(l_stcp tcp,l_out out)
{
	l_int32 i=10,retcode;
	l_num point=0;
	l_char tmp[MAXTCPREAD];
	if(tcp == 0)
	{
		l_err(out,"l_tcpsockrecv:tcp connection not associated");
		return 0;
	}	
	l_resetbuf(tcp->brecv);
	l_bzero(tmp,MAXTCPREAD);
	while((retcode = read(tcp->sockfd,(void *)tmp,MAXTCPREAD)) >= 0 && i != 0)
	{
		point += retcode;
		l_cwritebuf(tcp->brecv,tmp,retcode);
		i--;
		l_bzero(tmp,MAXTCPREAD);
	}
	if(point == 0)
		l_err(out,"l_tcpsockrecv:error recv message");
	#ifdef L_DEBUG
	else	
	{	
		l_str host=l_newstr();
		l_str port=l_newstr();
		l_getaddrinfo(tcp->raddr,&host,&port,out);
		l_debug("l_tcpsockrecv:recv tcp message (%d byte) from remote machine %s:%s",point,host,port);
		l_freestr(&port);
		l_freestr(&host);
	}
	#endif	
	return (l_size) point;
}	

void l_freestcp(l_stcp *ptcp)
{
	if((*ptcp) != 0)
	{
		if((*ptcp)->laddr != 0)
			l_freeaddrinfo(&((*ptcp)->laddr));
		if((*ptcp)->raddr != 0)
			l_freeaddrinfo(&((*ptcp)->raddr));
		l_freebuf(&((*ptcp)->bsend));
		l_freebuf(&((*ptcp)->brecv));
		l_memfree((void **)ptcp,sizeof(struct stcp));
	}
}

void l_closestcp(l_stcp *ptcp)
{
	if((*ptcp) != 0)
	{
		close((*ptcp)->sockfd);
		#ifdef L_DEBUG
			l_debug("l_closestcp:close tcp socket %d",(*ptcp)->sockfd);
		#endif
		l_freestcp(ptcp);
	}
}

/*end external*/

/*end tcp section*/

/*raw section*/

l_sraw  l_newsraw()
{
	l_sraw sraw=l_malloc0(sizeof(struct sraw));
	sraw->handle=0;
	sraw->bsend = l_newbuf();
	sraw->brecv  =l_newbuf();
	return sraw;			
}

/*external*/

l_sraw l_opensraw(const l_str dev,const l_str filter,l_out out)
{
	struct bpf_program fp;
	l_char errpcap[PCAP_ERRBUF_SIZE];
	l_sraw sraw= l_newsraw();
	sraw->handle=(void *) pcap_open_live(dev,BUFSIZ,1,1000,errpcap);
	if(sraw->handle == 0)
	{
		l_freesraw(&sraw);
		l_err(out,"%s",errpcap);
		return 0;
	}
	if(filter != 0)
	{
		if (pcap_compile((pcap_t *)sraw->handle, &fp,filter,0, 0) == -1 || pcap_setfilter((pcap_t *)sraw->handle, &fp) == -1)  
		{
			l_err(out,"l_opensraw:%s",pcap_geterr((pcap_t *)sraw->handle));
			l_closesraw(&sraw);
			return 0;
		}	
	}		
	#ifdef L_DEBUG
		l_debug("l_opensraw:open raw connection (handle %d)",(pcap_t *)sraw->handle);
	#endif
	return sraw;
}

void l_closesraw(l_sraw *praw)
{
	if((*praw) != 0)
	{
		#ifdef L_DEBUG
			l_debug("l_closesraw:close raw connection (handle %d)",(pcap_t *)((*praw)->handle));
		#endif
		pcap_close((pcap_t *)((*praw)->handle));
		l_freesraw(praw);
	}		
}

void l_freesraw(l_sraw *praw)
{
	if((*praw) != 0)
	{
		l_freebuf(&((*praw)->bsend));
		l_freebuf(&((*praw)->brecv));
		l_memfree((void **)praw,sizeof(struct sraw));
	}		
}

void l_srawsend(l_sraw sraw,l_out out)
{
	if(sraw==0)
	{
		l_err(out,"l_srawsend:no raw connection associated");
		return;
	}
	if(pcap_inject((pcap_t *)sraw->handle,l_membuf(sraw->bsend,0),l_sizebuf(sraw->bsend)) == -1)
	{
		l_err(out,"l_srawsend:%s",pcap_geterr((pcap_t *)sraw->handle));
		return;
	}	
}

l_size srawrecv(l_sraw sraw,l_out out)
{
	struct pcap_pkthdr pkthdr;
	const l_uchar *pack;
	l_num p=0;
	if(sraw==0)
	{
		l_err(out,"l_srawrecv:no raw connection associated");
		return 0;
	}
	do
	{	
		if((pack=pcap_next((pcap_t *)sraw->handle,&pkthdr)) == 0)
			return 0;
		l_nwritebuf(sraw->brecv,p,(void *)pack,pkthdr.len);
		p+=pkthdr.len;
	}while(pkthdr.caplen != p);
	return l_sizebuf(sraw->brecv);
}

void l_srawroutine(l_sraw sraw,void *callback,l_uchar *user ,l_out out)
{
	if(sraw==0)
		l_err(out,"l_srawroutine:no raw connection associated");
	else if(pcap_loop((pcap_t *)sraw->handle,0, (pcap_handler) callback,user) == -1)
			l_err(out,pcap_geterr(sraw->handle));
}

void l_closesrawroutine(l_sraw *praw)
{
	if((*praw) != 0)
	{
		#ifdef L_DEBUG
			l_debug("l_closesrawroutine:break raw routine (handle %d)",(pcap_t *)((*praw)->handle));
		#endif
		pcap_breakloop((pcap_t *)((*praw)->handle));
		l_closesraw(praw);
	}		
}

/*end external*/

/*end raw section*/

