/* -*- Mode:C; c-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella <vasta@ragnu.it>
 */
 
 /**
	*\file lupin-str.h
	*\author Marco Guastella (contact:vasta@ragnu.it)
	*\brief Header file to define the string functions
*/

#ifndef	_LIBLUPINSTR_H
#define	_LIBLUPINSTR_H	1

/**
	\defgroup str str section
	
*/

/**
	*\ingroup str 
	*\brief  string type
*/
typedef char *l_str;

/** 	
	\ingroup str
	\return null string	
	\brief initializes the string
*/	
l_str l_newstr();

/** 	
	\ingroup str
	\param[in] pstr pointer to the string to deallocate	
	\brief deallocates the string pointed to by pdstr
*/
void l_freestr(l_str *pstr);


/** 	
	\ingroup str
	\param[in] str input string
	\return size str input string	
	\brief calculates the file size of string str
*/
l_size l_strlen(l_str str);

/** 	
	\ingroup str
	\param[in] s1 input string
	\param[in] s2 input string
	\return an integer less than, equal to, or greater than zero if  s1  is  found,
        *respectively, to be less than, to match, or be greater than s2.
	\brief compares the two strings s1 and s2
*/
l_int32 l_strcmp(const l_str s1,const l_str s2);

/** 	
	\ingroup str
	\param[out] pdst pointer to destination string
	\param[in] src string to be copied
	\return the file size of the string pointed to by pdst
	\brief copies the string src to the string pointed to by pdst
*/
l_size l_strcpy(l_str *pdst,const l_str src);

/** 	
	\ingroup str
	\param[out] dst destination string
	\param[in] src string to be copied
        \param[in] len the maximum number of bytes copied	
        \return the file size of the string dst
	\brief copies the string src in dst. dst must be allocated before
*/
l_size l_strcpy2(l_str dst,l_str src,l_size len);

/** 	
	\ingroup str
	\param[out] pdst pointer to the string to be concatenated
	\param[in] src string to be concatenated
	\return the file size of the string pointed to by pdst
	\brief concatenates the string src to the string pointed to by pdst
*/
l_size l_strcat(l_str *pdst,const l_str src);

/** 	
	\ingroup str
	\param[in] str string to verify prefix
	\param[in] prefix prefix to verify
	\return TRUE if prefix is prefix str FALSE otherwise
	\brief verify if prefix is prefix str
*/
l_bool l_isstrprefix(l_str str,l_str prefix);

/** 	
	\ingroup str
	\param[in] str input string to cut
	\param[in] pos position string to cut
	\return string to tail str
	\brief return tail str from pos position
*/	
l_str l_strtail(l_str str,l_num pos);

/** 	
	\ingroup str
	\param[out] pstr pointer to the string to write
	\param[in] fmt format string 
	\param[in] ...  	arguments
	\return new size string pointed to by pstr
	\brief produces an output based on the description of the 
	* format string and writes it to in the string pointed to by pstr
*/
l_size l_writestr(l_str *pstr,const l_str fmt,...);

/** 	
	\ingroup str
	\param[out] pstr pointer to the string to write
	\param[in] fmt format string 
	\param[in] ...  	arguments
	\return new size string pointed to by pstr
	\brief produces an output based on the description of the 
	* format string and concatenates it to in the string pointed to by pstr
*/
l_size l_cwritestr(l_str *pstr,const l_str fmt,...);

/** 	
	\ingroup str
	\param[in] str string to reduce
	\param[in] len size to reduce str
	\return the size str
	\brief if the file size of str is' more slowly sets its file size in len 
*/
l_size l_reducestr(l_str str,l_size len);

/**
	*\ingroup str
	*\brief	string split type
*/

typedef struct split
{
	/**
		*\ingroup str
		*\brief	number of strings split
	*/
	l_num nsplit;
	
	/**
		*\ingroup str
		*\brief 	split strings
	*/
	l_str *psplit;
	
}*l_split;

/** 	
	\ingroup str
	\param[in] str string to splip
	\param[in] delimiter delimiter split 
	\return new struct split
	\brief  split the string str based on the delimiter delimiter 
	*and returns the structure derived from this
*/
l_split l_newsplit(const l_str str,const l_str delimiter);

/** 	
	\ingroup str
	\param[in] split struct split
	\return the number of split string; 0 on error
	\brief  return the number of split string in split
*/
l_num l_nsplit(l_split split);

/** 	
	\ingroup str
	\param[in] split struct split
	\param[in] n the number of the string to read
	\return the corresponding string n 
	\brief  return the corresponding string n
*/
l_str l_readsplit(l_split split,l_num n);

/**
	\ingroup str
	\param[in] psplit pointer to a struct split
	\brief  deallocates a struct split
*/
void l_freesplit(l_split *psplit);

#endif /*end lupin-str.h*/ 
