/* -*- Mode:C; c-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella <vasta@ragnu.it>
 */

/**
	*\file lupin-proto.h
	*\author Marco Guastella (contact:vasta@ragnu.it)
	*\brief Header file to protocols functions
*/

#ifndef	_LIBLUPINPROTO_H
#define	_LIBLUPINPROTO_H	1

/**
	\defgroup http http section
	
*/

/**
	\ingroup http
	\param[in] host address web server
	\param[in] port  port web server
	\param[out] out  http output options or error associated with the connection
	\brief  sends an HTTP request to a remote web server options and print out the result
*/
void l_httpoptions(const l_str host,const l_str port,l_out out);

/**
	\defgroup arp arp section
	
*/

/**
	\ingroup arp
	\param[in] dev the local device to scan the local network
	\param[in] ipsrc the local IP address, 0 if you want to use one assigned to the local device
	\param[in] netmask  the local netmask address, 0 if you want to use one assigned to the local device
	\param[in] maclocal the local mac address, 0 if you want to use one assigned to the local device
	\param[in] pvideo TRUE if you want to print the results while scanning, FALSE
			             the results are returned only through out
	\param[out] out scan result or error associated with the scan
	\brief scanning a local network information of the host and connected to each of them gives 
		  us the ip address and mac address of the device
*/
void l_scanlan(const l_str dev,const l_str ipsrc,const l_str netmask,const l_str maclocal,l_bool pvideo,l_out out);

/**
	\ingroup arp
	\param[in] dev the local device to research host
	\param[in] ipsrc the local IP address, 0 if you want to use one assigned to the local device
	\param[in] ipdst host IP address to be found in local network
	\param[in] maclocal the local mac address, 0 if you want to use one assigned to the local device
	\param[out] out output return value
	\return the mac address associated with your device connected otherwise return 0
	\brief Search ipdst ip host on a local network
*/
l_str l_getremotemac(const l_str dev,const l_str ipsrc,const l_str ipdst,const l_str maclocal,l_out out);

/**
	\ingroup arp
	\param[in] dev the local device to research host
	\param[in] ipsrc the local IP address, 0 if you want to use one assigned to the local device
	\param[in] ipdst host IP address to be found in local network
	\param[in] maclocal the local mac address, 0 if you want to use one assigned to the local device
	\param[out] out research result or error associated with the research
	\brief Search ipdst ip host on a local network, if found returns the mac address associated with 
		 your device connected otherwise, it returns not found
*/
void l_getremotemac2(const l_str dev,const l_str ipsrc,const l_str ipdst,const l_str maclocal,l_out out);

/**
	\ingroup arp
	\param[in] dev  the local device to research hosts
	\param[in] ip1 ip address the first host to search
	\param[in] ip2 ip address the second host to search
	\param[out] mac1 mac address first host
	\param[out] mac2 mac address second host
	\return TRUE if the two hosts have been found, FALSE otherwise and error out stores
	\param[out]  out research result or error associated with the research
	\brief must search the network associated with the host with IP address ip1 and ip2, 
		 and if he finds them both stores and their mac address in mac1 mac2
*/
l_bool l_macdouble(const l_str dev,const l_str ip1,const l_str ip2,l_str *mac1,l_str *mac2,l_out out);

/**
	\ingroup arp
	\param[in] dev arp spoofing device associated
	\param[in] ipvict ip host associated with victim
	\param[in] ipspoof ip spoofing referred to
	\param[in] macspoof mac address to associate with ipspoof,if 0 use the mac address associated with dev
	\return the structure to be passed as input to l_arpspoof_exec,l_arpspoof_err and l_arpspoof_close
	\brief creates the structure associated with the implementation of an IP spoofing attack
*/

void *l_arpspoof_init(const l_str dev,const l_str ipvict,const l_str ipspoof,const l_str macspoof);

/**
	\ingroup arp
	\param[in] data data structure associated ip spoofing attack
	\return TRUE If and 'associated with an error, FALSE otherwise
	\brief check whether the structure and initialized in init 'error associated with a
*/
l_bool l_arpspoof_err(void *data);

/** 
	\ingroup arp
	\param[in] data structure associated ip spoofing attack
	\return 0
	\brief do ip spoofing attack by sending an ARP request every 5 seconds
		  the victim so that in its ARP cache entry associated with the all'ipspoof 
		  (see l_arpspoof_init) matches the macspoof
*/
void *l_arpspoof_exec(void *data);

/**
	\ingroup arp
	\param[in] pdata pointer to the structure associated ip spoofing attack
	\return output ip spoofing attack
	\brief closes the connection and free the memory associated with raw ip spoofing attack
*/
l_out l_arpspoof_close(void **pdata);

/**
	\ingroup arp
	\param[in] dev arp deny device associated
	\param[in] ipvict ip host associated with victim
	\param[in] ipdeny ip address of the host to which they do not connect
	\param[in] macspoof mac spoof associated to ipdeny, 0 to generate automatic
	\return the structure to be passed as input to l_arpdeny_exec,l_arpdeny_err and l_arpdeny_close
	\brief creates the structure associated with the implementation of an IP deny attack
*/
void *l_arpdeny_init(const l_str dev,const l_str ipvict,const l_str ipdeny,l_str macspoof);

/**
	\ingroup arp
	\param[in] data structure associated ip deny attack
	\return TRUE If and 'associated with an error, FALSE otherwise
	\brief check whether the structure and initialized in init 'error associated with a
*/
l_bool l_arpdeny_err(void *data);

/** 
	\ingroup arp
	\param[in] data structure associated ip deny attack
	\return 0
	\brief prevent a victim host to communicate with another host
*/
void *l_arpdeny_exec(void *data);

/**
	\ingroup arp
	\param[in] pdata pointer to the structure associated ip deny attack
	\return output ip deny attack
	\brief closes the connection and free the memory associated with raw ip deny attack
*/
l_out l_arpdeny_close(void **pdata);

/**
	\ingroup arp
	\param[in] dev arp direct device associated
	\param[in] ipsrc ip source host associated
	\param[in] ipdst ip destination host associated
	\param[in] maclocal  mac address local,if 0 use the mac address associated with dev
	\param[in] filter optional filter(pcap mode) to receive packets from redirect,0 if not optional filter
	\return the structure to be passed as input to l_arpdirect_exec,l_arpdirect_err and l_arpdirect_close
	\brief creates the structure associated with the implementation of an IP direct
*/
void *l_arpdirect_init(const l_str dev,const l_str ipsrc,const l_str ipdst,const l_str maclocal,const l_str filter);

/**
	\ingroup arp
	\param[in] data structure associated ip direct
	\return TRUE If and 'associated with an error, FALSE otherwise
	\brief check whether the structure and initialized in init 'error associated with a
*/
l_bool l_arpdirect_err(void *data);

/** 
	\ingroup arp
	\param[in] data structure associated ip direct
	\return 0
	\brief reads the packets received by routing them to and from ipsrc ipdst.
		 To receive packages locally should combine this function with l_arpspoof by thread
*/
void *l_arpdirect_exec(void *data);

/**
	\ingroup arp
	\param[in] pdata pointer to the structure associated ip direct
	\return output ip direct
	\brief closes the connection and free the memory associated with raw ip direct
*/
l_out l_arpdirect_close(void **pdata);

#endif /*end lupin-proto.h*/ 
