/* -*- Mode:C; c-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella <vasta@ragnu.it>
 */
 
/**
	*\file lupin-net.h
	*\author Marco Guastella (contact:vasta@ragnu.it)
	*\brief Header file to define the net functions
*/

#ifndef	_LIBLUPINNET_H
#define	_LIBLUPINNET_H	1

/**
	\defgroup udp udp section
*/

/**
	*\ingroup udp
	*\brief socket udp type
*/


typedef struct sudp
{
	/**
		*\ingroup udp
		*\brief socket descriptor
	*/
	l_int32 sockfd;
	
	/**
		*\ingroup udp
		*\brief local machine network address format
	*/
	struct addrinfo *laddr;
	
	/**
		*\ingroup udp
		*\brief remote machine network address format
	*/
	struct addrinfo *raddr;
	
	/**
		*\ingroup udp
		*\brief send buffer
	*/
	l_buf bsend;
	
	/**
		*\ingroup udp
		*\brief receive buffer
	*/
	l_buf brecv;
}*l_sudp;

/**
	\ingroup udp
	\param[in] ip address the server udp
	\param[in] port port the server udp 
	\param[out] out output return value
	\return new struct socket connection listen udp,0 if error
	\brief listen a udp connection and returns the structure refers to the connection
*/
l_sudp l_udpsocklisten(const l_str ip,const l_str port,l_out out);

/**
	\ingroup udp
	\param[in] udp struct  socket connection udp
	\param[out] out output return value
	\return new struct socket connection listen udp,0 if error
	\brief  accepts a connection and returns a new struct udp socket
*/
l_sudp l_udpsockaccept(l_sudp udp,l_out out);

/**
	\ingroup udp
	\param[in] udp struct  socket connection udp
	\param[out] out output return value
	\return the number of bytes received
	\brief  receives a message from remote machine (message in udp->brecv)
*/
l_size l_udpsockrecv(l_sudp udp,l_out out);

/**
	\ingroup udp
	\param[in] udp  struct  socket connection udp
	\param[out] out output return value
	\brief  send a message to remote machine (message in udp -> bsend)
*/
void l_udpsocksend(l_sudp udp,l_out out);

/**
	\ingroup udp
	\param[in] pudp pointer to socket connection udp
	\brief  free struct connection udp
*/
void l_freesudp(l_sudp *pudp);

/**
	\ingroup udp
	\param[in] pudp pointer to socket connection udp 
	\brief  close udp socket and free struct connection udp
*/
void l_closesudp(l_sudp *pudp);

/**
	\defgroup tcp tcp section
*/

/**
	*\ingroup tcp
	*\brief socket tcp type
*/
typedef struct stcp
{
	/**
		*\ingroup tcp
		*\brief socket descriptor
	*/
	l_int32 sockfd;
	
	/**
		*\ingroup tcp
		*\brief local machine network address format
	*/
	struct addrinfo *laddr;
	
	/**
		*\ingroup tcp
		*\brief remote machine network address format
	*/
	struct addrinfo *raddr;
	
	/**
		*\ingroup tcp
		*\brief send buffer
	*/
	l_buf bsend;
	
	/**
		*\ingroup tcp
		*\brief receive buffer
	*/
	l_buf brecv;
}*l_stcp;

/**
	\ingroup tcp
	\param[in] rhost remote host
	\param[in] rport remote port
	\param[out] out output return value
	\return structure associated with the tcp connection, 0 if error
	\brief  opens a TCP client
*/

l_stcp l_tcpsockclient(const l_str rhost,const l_str rport,l_out out);

/**
	\ingroup tcp
	\param[in] tcp structure associated with the tcp connection
	\param[out] out output return value
	\brief  sends a TCP packet in remote

*/
void l_tcpsocksend(l_stcp tcp,l_out out);

/**
	\ingroup tcp
	\param[in] tcp structure associated with the tcp connection
	\param[out] out output return value
	\return the number of bytes received
	\brief recvs a TCP packet from remote

*/
l_size l_tcpsockrecv(l_stcp tcp,l_out out);

/**
	\ingroup tcp
	\param[in] ptcp pointer to socket connection tcp
	\brief  free struct connection tcp
*/
void l_freestcp(l_stcp *ptcp);

/**
	\ingroup tcp
	\param[in] ptcp pointer to socket connection tcp 
	\brief  close tcp socket and free struct connection tcp
*/
void l_closestcp(l_stcp *ptcp);

/**
	\ingroup raw
	\brief socket tcp type
*/

typedef struct sraw
{
	/**
		\ingroup raw
		\brief pcap handle
	*/
	void *handle;

	/**
		\ingroup raw
		\brief send buffer
	*/
	l_buf bsend;
	
	/**
		\ingroup raw
		\brief receive buffer
	*/
	l_buf brecv;
	
}*l_sraw;

/**
	\ingroup raw
	\param[in] dev device to open raw connection
	\param[in] filter pcap filter string
	\param[out] out output return value
	\return raw structure, if error 0
	\brief open a raw connection in dev applying the pcap filter 
*/
l_sraw l_opensraw(const l_str dev,const l_str filter,l_out out);

/**
	\ingroup raw
	\param[in] praw pointer to raw connection 
	\brief  free struct connection raw
*/
void l_freesraw(l_sraw *praw);

/**
	\ingroup raw
	\param[in] praw pointer to raw connection 
	\brief  close raw connection and free struct connection raw
*/
void l_closesraw(l_sraw *praw);

/**
	\ingroup raw
	\param[in] sraw raw connection
	\param[out] out output return value
	\brief  sends a packet via a raw
*/
void l_srawsend(l_sraw sraw,l_out out);

/**
	\ingroup raw
	\param[in] sraw raw connection
	\param[out] out output return value
	\brief recv a packet via raw 
*/	
l_size srawrecv(l_sraw sraw,l_out out);

/**
	\ingroup raw
	\param[in] sraw raw connection
	\param[in] callback function to process packets (void *callback)(l_uchar *args, const struct pcap_pkthdr *header,const l_uchar *packet)
			  args = args callback , header header libpcap type, packet the packet to process
	\param[in] user input args callback
	\param[in] out output return value
	\brief receives raw packets from one connection and processes them through the callback function
*/
void l_srawroutine(l_sraw sraw,void *callback,l_uchar *user ,l_out out);

/**
	\ingroup raw
	\param[in] praw pointer to raw connection
	\brief break routine l_srawroutine,close raw connection and free struct to by praw
*/
void l_closesrawroutine(l_sraw *praw);

#endif /*end lupin-net.h*/ 

