/* -*- Mode:C; c-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella <vasta@ragnu.it>
 */
 
/**
	*\file lupin-core.h
	*\author Marco Guastella (contact:vasta@ragnu.it)
	*\brief Header file to define the basic functions
*/

#ifndef	_LIBLUPINCORE_H
#define	_LIBLUPINCORE_H	1

#include "lupin-str.h"
#include "lupin-buf.h"

/*
	defgroup lupin lupin section
*/

/**
	\ingroup lupin
	\brief autotools function to call to verify the existence of the library
*/

void l_lupin();

/**
	\defgroup out out section
	
*/

/**
	*\ingroup out 
	*\brief  return value type 
*/
typedef enum ret{LOK,LERR,LUNDEF} l_ret;

/**
	*\ingroup out 
	*\brief  output type 
*/
typedef struct out
{
	/**
		*\ingroup out
		*\brief	standard error buffer
	*/
	l_buf stderr;
	
	/**
		*\ingroup out
		*\brief	standard output buffer
	*/
	l_buf stdout;
	
	/**
		*\ingroup out
		*\brief	return value of functions
	*/
	l_ret retvalue;
	
}*l_out;

/**
	\ingroup out
	\return structure out 
	\brief  creates a structure out
*/	
l_out l_newout();

/**
	\ingroup out
	\param[in] pout 
	\brief  pointer to the output structure to deallocate
*/
void l_freeout(l_out *pout);

/**
	\ingroup out
	\param[in] out the structure to be printed out
	\brief check out the condition of the structure as input. 
	*If LOK prints the value stdout of the standard output, 
	*if LERR prints the value stderr of the standard error  
	*else if LUNDEF not print anything. 
*/
void l_printout(l_out out);

/**
	\ingroup out
	\param[out] str where the string should be copied to the value
	\param[in] out the structure to be copied out
	\brief check out the condition of the structure as input. 
	*If LOK copies stdout value in str , 
	*if LERR copies stderr value in str  
	*else if LUNDEF not copies anything. 
*/
void l_sprintout(l_str *str,l_out out);

/**
	\ingroup out
	\param[in] out the structure to be verified
	\return out and if TRUE 'state LERR, FALSE otherwise
	\brief return out and if true 'state LERR, FALSE otherwise	
*/
l_bool l_isouterr(l_out out);

/**
	\defgroup system system section
*/

/**
	\ingroup system
	\param[in] fmt format string 
	\param[in] ...  	arguments
	\brief  produces an output based on the description of the 
	* format string and writes it to standard output
*/
void l_stdout(const l_str fmt,...);

/**
	\ingroup system
	\param[in] fmt format string 
	\param[in] ...  	arguments
	\brief  produces an output based on the description of the 
	* format string and writes it to standard error
*/
void l_stderr(const l_str fmt,...);

/**
	\defgroup struct struct section
	
*/

/**
	\ingroup struct
	\param[in] str string to convert in integer
	\param[out] ret pointer integer to write value
	\return TRUE if convert is valid FALSE otherwise
	\brief convert str value in format integer to integer
*/	
l_bool l_atoi(const l_str str,l_int32 *ret);

/**
	\ingroup struct
	\param[in] buf buffer to copy
	\param[in] point  point from which the conversion
	\param[out] pstr pointer to output string
	\param[in] len number of bytes to copy in the string
	\brief  copy len bytes from buffer buf point point in the string pointed to by pstr
*/
l_size l_buftostr(l_buf buf,l_num point,l_str *pstr,l_size len);

/**
	\ingroup struct
	\param[in] str string to convert
	\param[out] buf buffer to copy value hexadecimal
	\param[in] point point buf to copy value hexadecimal
	\return True if the string is the hexadecimal otherwise False
	\brief  copy the values of a hexadecimal string in a buffer
*/
l_bool l_strexatobuf(const l_str str,l_buf buf,l_num point);

/**
	\ingroup struct
	\param[in] str mac address(00:11:22:33:44:55)
	\param[out] buf buffer to write mac address in byte format
	\param[in] point point in buffer to write mac 
	\return TRUE if mac format is bad FALSE otherwise
	\brief copy str in mac format to buf	
	
*/
l_bool l_strmactobuf(const l_str str,l_buf buf,l_num point);

/**
	\ingroup struct
	\param[in] host hosts to know the ip
	\param[out] out output return value
	\return the ip address in string format
	\brief resolves the address of a host via the local resolver
*/
l_str l_resolver(const l_str host,l_out out);

#include "lupin-net.h"

/**
	\ingroup crypt
	\param[in] input buffer to apply the function md5
	\return encrypted buffer
	\brief apply the function md5 to the buffer of input and returns the encrypted buffer
*/	
l_buf l_md5(l_buf input);

/**
	\ingroup crypt
	\param[in] input buffer to apply the function sha256
	\return encrypted buffer
	\brief apply the function sha256 to the buffer of input and returns the encrypted buffer
*/	
l_buf l_sha256(l_buf input);

#endif /*end lupin-core.h*/ 

